"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PublishAssetScriptGenerator = void 0;
const fs = __importStar(require("node:fs"));
const path = __importStar(require("node:path"));
// Constants
/**
 * The prefix used for asset hash names in the generated script.
 */
const ASSET_HASH_NAME = "asset-hash";
/**
 * Converts a given path to POSIX format (using forward slashes).
 *
 * @param p - The input path.
 * @returns The path in POSIX format.
 */
const posixPath = (p) => p.split(path.sep).join(path.posix.sep);
/**
 * A helper class that generates a single script file required to publish assets.
 */
class PublishAssetScriptGenerator {
    cdkoutDir;
    assets;
    addOnlyTemplatesToOutput;
    /**
     * A map that stores unique asset IDs and their corresponding output variable names.
     */
    assetIdMap;
    /**
     * The complete script content that will be written to a file.
     */
    content;
    /**
     * Constructs a new instance of the `PublishAssetScriptGenerator` class.
     *
     * @param cdkoutDir - The output directory for CDK assets.
     * @param assets - A list of `StackAsset` objects to be published.
     * @param addOnlyTemplatesToOutput - If `true`, only assets marked as templates will be added to the output map.
     */
    constructor(cdkoutDir, assets, addOnlyTemplatesToOutput = true) {
        this.cdkoutDir = cdkoutDir;
        this.assets = assets;
        this.addOnlyTemplatesToOutput = addOnlyTemplatesToOutput;
        this.assetIdMap = new Map();
        this.content = this.generatePublishScripts();
    }
    /**
     * Generates the publish commands for each asset and stores them as script content.
     *
     * @returns A string representing all commands to be included in the publish script.
     *
     * @remarks
     * The method iterates over each asset, creating a publish command for it, and appends it to
     * the list of script commands. If `addOnlyTemplatesToOutput` is set to `true`, only assets
     * marked as templates are added to the `assetIdMap` for outputting.
     */
    generatePublishScripts() {
        const commands = ["set -ex"];
        for (const asset of this.assets) {
            const { assetId, assetManifestPath, assetSelector } = asset;
            // Command to publish asset
            commands.push(`npx cdk-assets --path "${this.relativeToAssembly(assetManifestPath)}" --verbose publish "${assetSelector}"`);
            // Add asset ID to output map if criteria met
            if (!this.assetIdMap.has(assetId) && (!this.addOnlyTemplatesToOutput || asset.isTemplate)) {
                const assetOutputName = `${ASSET_HASH_NAME}-${this.assetIdMap.size + 1}`;
                this.assetIdMap.set(assetId, assetOutputName);
            }
        }
        // Append commands to output each asset's hash
        this.assetIdMap.forEach((assetOutputName, assetId) => {
            commands.push(`echo '${assetOutputName}=${assetId}' >> $GITHUB_OUTPUT`);
        });
        return commands.join("\n");
    }
    /**
     * Writes the generated publish script content to a specified file.
     *
     * @param filePath - The path where the script file should be saved.
     *
     * @remarks
     * This method creates the necessary directories if they do not exist and writes
     * the accumulated `content` to the specified file path.
     */
    writePublishScript(filePath) {
        fs.mkdirSync(path.dirname(filePath), { recursive: true });
        fs.writeFileSync(filePath, this.content, { encoding: "utf-8" });
    }
    /**
     * Returns the path to an asset relative to the assembly directory.
     *
     * @param p - The absolute path to an asset.
     * @returns The path relative to the assembly directory in POSIX format.
     */
    relativeToAssembly(p) {
        return posixPath(path.join(this.cdkoutDir, path.relative(path.resolve(this.cdkoutDir), p)));
    }
}
exports.PublishAssetScriptGenerator = PublishAssetScriptGenerator;
//# sourceMappingURL=data:application/json;base64,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