import { Stage } from "aws-cdk-lib";
import { type StageDeployment } from "aws-cdk-lib/pipelines";
import { Construct } from "constructs";
import type { AwsCredentialsProvider } from "./aws-credentials";
import type { IJobPhase } from "./jobs";
import type { Synth } from "./steps";
import { GitHubWave, type StageOptions, type WaveOptions } from "./wave";
/**
 * Properties for configuring the GitHub Actions pipeline.
 *
 * @remarks
 * Provides options for defining the workflow environment, AWS credentials, job phases, and version overrides,
 * along with paths and naming conventions for GitHub Actions workflows.
 */
export interface GitHubActionsPipelineProps {
    /**
     * Optional name for the GitHub Actions workflow.
     *
     * @default "Deploy"
     */
    readonly workflowName?: string;
    /**
     * Directory path for the workflow output files.
     *
     * @default ".github/workflows"
     */
    readonly workflowOutdir?: string;
    /**
     * Filename for the workflow file.
     *
     * @default "deploy"
     */
    readonly workflowFilename?: string;
    /**
     * Whether to enable a single publisher for each asset type.
     *
     * @remarks
     * When true, consolidates publishing jobs to reduce redundant asset publishing.
     *
     * @default false
     */
    readonly singlePublisherPerAssetType?: boolean;
    /**
     * Environment variables to set in the workflow.
     */
    readonly workflowEnv?: Record<string, string>;
    /**
     * Optional phase for jobs to execute before the main build steps.
     */
    readonly preBuild?: IJobPhase;
    /**
     * Optional phase for jobs to execute after the main build steps.
     */
    readonly postBuild?: IJobPhase;
    /**
     * AWS credentials provider for authenticating AWS actions.
     */
    readonly awsCredentials: AwsCredentialsProvider;
    /**
     * Version overrides for GitHub Actions used in the workflow.
     */
    readonly versionOverrides?: Record<string, string>;
    /**
     * Synthesizer for CDK applications.
     */
    readonly synth: Synth;
}
/**
 * Constructs a GitHub Actions pipeline for deploying AWS resources.
 *
 * @remarks
 * The `GitHubActionsPipeline` provides methods to define and manage deployment stages and job waves in
 * a GitHub Actions pipeline, utilizing AWS credentials and CDK output for cloud infrastructure automation.
 */
export declare class GitHubActionsPipeline extends Construct {
    private readonly innerPipeline;
    /**
     * Constructs a new instance of `GitHubActionsPipeline`.
     *
     * @param scope - The parent construct scope.
     * @param id - Unique identifier for this pipeline construct.
     * @param props - Configuration properties for the pipeline.
     */
    constructor(scope: Construct, id: string, props: GitHubActionsPipelineProps);
    /**
     * Returns the name of the workflow.
     */
    get workflowName(): string;
    /**
     * Returns the output directory path for the workflow files.
     */
    get workflowOutdir(): string;
    /**
     * Returns the filename for the workflow file.
     */
    get workflowFilename(): string;
    /**
     * Adds a stage to the pipeline with GitHub-specific configuration options.
     *
     * @param stage - The CDK Stage to add to the pipeline.
     * @param options - Optional configuration for the stage.
     * @returns Deployment details for the added stage.
     */
    addStage(stage: Stage, options?: StageOptions): StageDeployment;
    /**
     * Adds a wave of jobs to the pipeline.
     *
     * @param id - Unique identifier for the wave.
     * @param options - Options for configuring the wave.
     * @returns The created GitHub wave instance.
     */
    addWave(id: string, options?: WaveOptions): GitHubWave;
}
