"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubActionsRole = exports.GitHubActionsOpenIdConnectProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
/**
 * GitHub OIDC provider thumbprints, updated as of 2023-07-27.
 *
 * For details and future updates, see:
 * https://github.blog/changelog/2023-06-27-github-actions-update-on-oidc-integration-with-aws/
 */
const GITHUB_OIDC_THUMBPRINTS = [
    "6938fd4d98bab03faadb97b34396831e3780aea1",
    "1c58a3a8518e8759bf075b76b750d4f2df264fcd",
];
/** GitHub Actions OIDC token endpoint */
const RAW_ENDPOINT = "token.actions.githubusercontent.com";
const PROVIDER_URL = `https://${RAW_ENDPOINT}`;
/**
 * Represents an OpenID Connect (OIDC) provider for GitHub Actions.
 *
 * This provider allows GitHub Actions to assume roles in AWS by connecting through OpenID Connect.
 */
class GitHubActionsOpenIdConnectProvider extends constructs_1.Construct {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.GitHubActionsOpenIdConnectProvider", version: "0.0.21" };
    /**
     * Imports an existing GitHub Actions OpenID Connect provider by ARN.
     *
     * @param scope - The construct scope to define the provider within.
     * @returns The imported OIDC provider interface.
     */
    static fromGitHubActionsOpenIdConnectProvider(scope) {
        return aws_iam_1.OpenIdConnectProvider.fromOpenIdConnectProviderArn(scope, "GitHubActionProvider", `arn:${aws_cdk_lib_1.Aws.PARTITION}:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:oidc-provider/${RAW_ENDPOINT}`);
    }
    /**
     * Constructs a new instance of `GitHubActionsOpenIdConnectProvider`.
     *
     * @param scope - The construct scope to define the provider within.
     * @param id - The unique identifier for this provider.
     * @param props - Optional properties for the OpenID Connect provider.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        new aws_iam_1.OpenIdConnectProvider(this, "GitHubProvider", {
            url: PROVIDER_URL,
            clientIds: ["sts.amazonaws.com"],
            thumbprints: props.thumbprints ?? GITHUB_OIDC_THUMBPRINTS,
        });
    }
}
exports.GitHubActionsOpenIdConnectProvider = GitHubActionsOpenIdConnectProvider;
/**
 * Creates an IAM Role for GitHub Actions workflows using an OpenID Connect provider.
 *
 * The role includes policies allowing the assumption of bootstrap roles and access to ECR authorization.
 */
class GitHubActionsRole extends aws_iam_1.Role {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/aws-cdk.GitHubActionsRole", version: "0.0.21" };
    /**
     * Constructs a new instance of `GitHubActionsRole`.
     *
     * @param scope - The construct scope to define the role within.
     * @param id - The unique identifier for this role.
     * @param props - The properties for configuring the GitHub Actions role.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            roleName: props.roleName ?? "GitHubActionsRole",
            assumedBy: new aws_iam_1.FederatedPrincipal(props.provider.openIdConnectProviderArn, {
                StringLike: {
                    [`${RAW_ENDPOINT}:sub`]: formatRepos(props.repos ?? []).concat(props.subjectClaims ?? []),
                    [`${RAW_ENDPOINT}:aud`]: "sts.amazonaws.com",
                },
            }, "sts:AssumeRoleWithWebIdentity"),
            inlinePolicies: {
                AssumeBootstrapRoles: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["sts:AssumeRole"],
                            resources: ["*"],
                            conditions: {
                                "ForAnyValue:StringEquals": {
                                    "iam:ResourceTag/aws-cdk:bootstrap-role": [
                                        "deploy",
                                        "lookup",
                                        "file-publishing",
                                        "image-publishing",
                                    ],
                                },
                            },
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["ecr:GetAuthorizationToken"],
                            resources: ["*"],
                        }),
                    ],
                }),
            },
        });
    }
}
exports.GitHubActionsRole = GitHubActionsRole;
/**
 * Formats GitHub repository identifiers for the OIDC `sub` claim.
 *
 * Each entry is formatted as `repo:owner/repo:*`.
 *
 * @param repos - A list of GitHub repositories in the format `owner/repo`.
 * @returns A list of formatted repository subject claims.
 */
function formatRepos(repos) {
    return repos.map((repo) => `repo:${repo}:*`);
}
//# sourceMappingURL=data:application/json;base64,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