import os
import subprocess
from typing import Union

import click
from gable.options import TypescriptLibrary
from loguru import logger

BASE_NPX_CMD = [
    "npx",
    "-y",
    "-q",
    "@gable-eng/sca",
]


def check_node_installed():
    try:
        result = subprocess.run(
            ["node", "--version"], check=True, stdout=subprocess.PIPE, text=True
        )
        version = result.stdout.strip().replace("v", "")
        if int(version.split(".")[0]) < 18:
            raise click.ClickException(
                f"Node.js version {version} is not supported. Please install Node.js 18 or later."
            )
    except FileNotFoundError:
        raise click.ClickException(
            "Node.js is not installed. Please install Node.js 18 or later."
        )


def run_sca_python(
    project_root: str,
    emitter_file_path: str,
    emitter_function: str,
    emitter_payload_parameter: str,
    event_name_key: str,
    api_endpoint: Union[str, None] = None,
):
    try:
        cmd = get_cmd(
            api_endpoint,
            [
                "python",
                project_root,
                "--emitter-file-path",
                emitter_file_path,
                "--emitter-function",
                emitter_function,
                "--emitter-payload-parameter",
                emitter_payload_parameter,
                "--event-name-key",
                event_name_key,
            ],
        )

        result = subprocess.run(
            cmd,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
        )
        if result.returncode != 0:
            logger.debug(result.stderr)
            raise click.ClickException(f"Error running Gable SCA: {result.stderr}")
        logger.debug(result.stdout)
        logger.trace(result.stderr)
        # The sca CLI prints the results to stdout,and everything else to trace/warn/debug/error
        return result.stdout
    except Exception as e:
        logger.opt(exception=e).debug("Error running Gable SCA")
        raise click.ClickException(
            "Error running Gable SCA: Please ensure you have the @gable-eng/sca package installed."
        )


def run_sca_typescript(
    library: str,
    api_endpoint: Union[str, None] = None,
):
    try:
        cmd = get_cmd(api_endpoint, ["typescript", "--library", library])
        result = subprocess.run(
            cmd,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
        )
        if result.returncode != 0:
            logger.debug(result.stderr)
            raise click.ClickException(f"Error running Gable SCA: {result.stderr}")
        logger.debug(result.stdout)
        logger.debug(result.stderr)
        # The sca CLI prints the results to stdout,and everything else to trace/warn/debug/error
        return result.stdout
    except Exception as e:
        logger.opt(exception=e).debug("Error running Gable SCA")
        raise click.ClickException(
            "Error running Gable SCA: Please ensure you have the @gable-eng/sca package installed."
        )


def get_cmd(gable_api_endpoint: Union[str, None], args: list[str]) -> list[str]:
    """Constructs the full command to run sca"""
    cmd = get_base_npx_cmd(gable_api_endpoint) + args
    logger.trace("sca cmd: " + " ".join(cmd))
    return cmd


def get_base_npx_cmd(gable_api_endpoint: Union[str, None]) -> list[str]:
    """Based on the endpoint and GABLE_LOCAL environment variable, decide if we should use the local package
    Returns: list[str] - The base command to run sca, either using npx + @gable-eng/sca or node + local path
    """
    gable_local = os.environ.get("GABLE_LOCAL")
    is_endpoint_localhost = (
        gable_api_endpoint is not None
        and gable_api_endpoint.startswith("http://localhost")
    )
    if gable_local == "false":
        return BASE_NPX_CMD
    if gable_local == "true" or is_endpoint_localhost:
        logger.trace("Configuring local settings")
        try:
            # Needs to be a dynamic import because this file is excluded from the bundled package
            from gable.local import config_local

            local_sca_path = config_local()
            return [
                "node",
                local_sca_path,
            ]
        except ImportError as e:
            logger.trace(f"Error importing local config")
    return BASE_NPX_CMD
