
import types

from pybud.drawer import ColoredString as CStr
from pybud.drawer import ColoredStringList as CStrList
from pybud.drawer import ColorType, Drawer
from readchar import key as Key

from .utils import Point, Size, DEFAULT_BACKGROUND_COLOR


class WidgetBase():
    def __init__(self, ctype: ColorType = None, size: Size = None, pos: Point = Point(0, 0)):
        # set variabled
        self.ctype = ctype
        self.size = Size(10, 20) if size is None else size
        self.pos = Point(0, 0) if pos is None else pos
        # defaults
        self.background_color = DEFAULT_BACKGROUND_COLOR

    def on_interrupt(self):
        pass

    def get_drawer(self):
        return Drawer(size=(self.size.getWidth(), self.size.getHeight()), background_color=self.background_color)

    def render(self):
        pass

    def get_render(self):
        self.drawer = self.get_drawer()
        self.render()
        return self.drawer

    def update(self, key):
        return key


class Widget(WidgetBase):
    def __init__(self, ctype: ColorType = None, size: Size = None, pos: Point = Point(0, 0)):
        super().__init__(ctype, size, pos)
        # defaults
        self.is_disabled = False
        self.is_selected = False
        self.selectable = True
        self.parent = None

    def run_callbacks(self):
        pass

    def on_enter(self):
        return self.run_callbacks()

    def on_interrupt(self):
        super().on_interrupt()
        self.is_disabled = True

    def update(self, key):
        key = super().update(key)
        if key == Key.ENTER:
            self.on_enter()
            return
        return key


class UnselectableWidget(WidgetBase):
    def __init__(self, ctype: ColorType = None, size: Size = None, pos: Point = Point(0, 0)):
        super().__init__(ctype, size, pos)
        self.selectable = False
        self.is_selected = False


class WidgetLabel(UnselectableWidget):
    def __init__(self,
                 text,
                 centered: bool = True,
                 wordwrap: bool = True,
                 size: Size = None,
                 pos: Point = Point(0, 0),
                 ctype: ColorType = None
                 ):

        super().__init__(ctype, size, pos)

        if isinstance(text, str):
            self.text = CStr(text, forecolor=(220, 220, 220))
        elif isinstance(text, CStr) or isinstance(text, CStrList):
            self.text = text
        else:
            raise NotImplementedError()

        self.size.h = (len(text) // (self.size.w - 8)) + 1
        self.centered = centered
        self.wordwrap = wordwrap

    def run_callbacks(self):
        pass

    def _place_text(self, text, ln_idx):
        if self.centered:
            self.drawer.center_place(text, ln_idx=ln_idx)
        else:
            self.drawer.place(text, pos=(0, ln_idx))

    def place_text(self, t, ln_idx):
        #print(t)
        if len(t) > (self.size.getWidth() - 8):
            for i in range(len(t)):
                i_ = len(t) - i - 1
                if i_ > (self.size.getWidth() - 8):
                    continue
                c = t[i_]
                if isinstance(c, str) and c == " ":
                    self._place_text(t[:i_], ln_idx)
                    return self.place_text(t[i_+1:], ln_idx + 1)
                if isinstance(c, CStr) and c.str == " ":
                    self._place_text(t[:i_], ln_idx)
                    return self.place_text(t[i_+1:], ln_idx + 1)
        else:
            self._place_text(t, ln_idx)

    def render(self):
        self.place_text(self.text, ln_idx=0)


class WidgetOptions(Widget):
    def __init__(self,
                 options: list[tuple[str, types.FunctionType]],
                 text: str = "Options:",
                 default_option: int = 0,
                 size: Size = None,
                 pos: Point = Point(0, 0),
                 ctype: ColorType = None
                 ):

        super().__init__(ctype, size, pos)
        self.n_options = len(options)
        self.options = list(map(lambda x: x[0], options))
        self.text = text
        self.callbacks = list(map(lambda x: x[1], options))
        self.selected = default_option
        self.size.h = 1 + self.n_options

    def run_callbacks(self):
        return self.callbacks[self.selected](self.parent)

    def update(self, key):
        key = super().update(key)
        if key == Key.UP:
            self.selected = (self.selected - 1) % self.n_options
            return
        if key == Key.DOWN:
            self.selected = (self.selected + 1) % self.n_options
            return
        return key

    def render(self):
        caption_start = 2
        __options = self.text.ljust(self.size.getWidth()-caption_start)
        self.drawer.place(CStr(__options, forecolor=(
            220, 220, 220)), pos=(caption_start, 0))
        for i, option in enumerate(self.options):
            if i == self.selected:
                option_color = (50, 220, 80)
                option_indicator = CStr("> ", forecolor=(220, 220, 220))
            else:
                option_color = (220, 220, 220)
                option_indicator = CStr("  ", forecolor=(220, 220, 220))
            __option = CStr(option, forecolor=option_color)
            self.drawer.place(option_indicator + __option,
                              pos=(caption_start, 1 + i))


class WidgetInput(Widget):
    def __init__(self,
                 text: str,
                 size: Size = None,
                 pos: Point = Point(0, 0),
                 ctype: ColorType = None,
                 allowed_characters: str = " abcdefghijklmnopqrstuvwxyz0123456789_-+/\\()[]."
                 ):
        super().__init__(ctype, size, pos)
        # p = 1
        self.size.h = 1
        self.text = text
        self.height = 1
        self.input = ""
        self.pointer = 0
        self.view = 0
        self.show_pointer = False
        # characters that this dialouge will listen to
        self.allowed_characters = allowed_characters
        self.tick = 0

    def run_callbacks(self):
        super().run_callbacks()
        self.parent.close()
        return self.input

    def reset(self):
        self.input = ""
        self.pointer = 0
        self.view = 0

    def getmaxlen(self):
        p = 2
        max_input_len = self.size.getWidth()-(p * 2)-len(self.text)
        return max_input_len

    def update(self, key):
        key = super().update(key)
        if key is None:
            return
        max_input_len = self.getmaxlen()
        # capture and place characters
        if key.lower() in self.allowed_characters:
            self.input = self.input[:self.pointer] + \
                key + self.input[self.pointer:]
            self.pointer += 1
            if (len(self.input)-self.view) > (max_input_len-1):
                self.view += 1
            return

        # backspace
        if key == Key.BACKSPACE:
            if self.pointer != 0:
                self.input = self.input[:self.pointer -
                                        1] + self.input[self.pointer:]
            else:
                self.input = self.input[self.pointer:]
            self.pointer = max(self.pointer - 1, 0)
            if self.pointer-self.view < 0:
                self.view = max(self.view - max_input_len, 0)
            return

        # delete
        if key == Key.DELETE:
            self.input = self.input[:self.pointer] + \
                self.input[self.pointer+1:]
            # self.pointer = max(self.pointer - 1, 0)
            return

        # left arrow
        if key == Key.LEFT:
            self.pointer = max(self.pointer - 1, 0)
            if self.pointer-self.view < 0:
                self.view = max(self.view - 1, 0)
            return

        # right arrow
        if key == Key.RIGHT:
            self.pointer = min(self.pointer + 1, len(self.input))
            if self.view == (self.pointer-max_input_len):
                self.view = min(self.view + 1, len(self.input))
            return

        if key == "UPDATE":
            self.tick += 1
            self.show_pointer = (self.tick % 12) >= 6
        return key

    def format_textbox(self):
        text, inp, pointer, view = self.text, self.input, self.pointer, self.view
        max_input_len = self.getmaxlen()
        # get the input text with fixed length plus one extra space for the pointer
        if view+max_input_len-1 < len(inp):
            inp_ = inp[view:view+max_input_len]
        else:
            inp_ = inp[view:view+max_input_len-1] + " "

        if not self.is_disabled and self.show_pointer or inp_[pointer-view] != " ":
            pointer_str = CStr(
                inp_[pointer-view], backcolor=(220, 220, 220), forecolor=(20, 20, 20))
        else:
            pointer_str = CStr(inp_[pointer-view])
        # insert the pointer into input text at the correct position
        input_plus_pointer = CStr(
            inp_[:pointer-view]) + pointer_str + CStr(inp_[pointer-view+1:])

        text_c = (50, 200, 50)

        fstr = CStr(text, forecolor=text_c)

        back_exists = CStr("<", forecolor=text_c)
        forward_exists = CStr(">", forecolor=text_c)

        fstr += (back_exists if view != 0 else " ")
        fstr += input_plus_pointer + " " * \
            (max_input_len - len(input_plus_pointer))
        fstr += (forward_exists if view+max_input_len-1 < len(inp) else " ")

        return fstr

    def render(self):
        p = 1
        # self.drawer.place(CStr(" " * (self.size.getWidth() - (p * 2) - len(self.text)),
        #                  backcolor=(20, 20, 20)), pos=(p + len(self.text), 0))
        self.drawer.place(self.format_textbox(), pos=(p, 0))
