import logging

logger = logging.getLogger(__name__)


def resolve_field_path(instance, field_path):
    """
    Recursively resolve a field path using Django's __ notation, e.g., "author__profile__name".

    CRITICAL: For foreign key fields, uses attname to access the ID directly
    to avoid hooking Django's descriptor protocol which causes N+1 queries.
    """
    # For simple field access (no __), use optimized field access
    if "__" not in field_path:
        try:
            # Get the field from the model's meta to check if it's a foreign key
            field = instance._meta.get_field(field_path)
            if field.is_relation and not field.many_to_many:
                # For foreign key fields, use attname to get the ID directly
                # This avoids hooking Django's descriptor protocol
                return getattr(instance, field.attname, None)
            # For regular fields, use normal getattr
            return getattr(instance, field_path, None)
        except Exception:
            # If field lookup fails, fall back to normal getattr
            return getattr(instance, field_path, None)

    # For paths with __, traverse the relationship chain with FK optimization
    current_instance = instance
    for i, attr in enumerate(field_path.split("__")):
        if current_instance is None:
            return None

        try:
            # Check if this is the last attribute and if it's a FK field
            is_last_attr = i == len(field_path.split("__")) - 1
            if is_last_attr and hasattr(current_instance, "_meta"):
                try:
                    field = current_instance._meta.get_field(attr)
                    if field.is_relation and not field.many_to_many:
                        # Use attname for the final FK field access
                        current_instance = getattr(
                            current_instance,
                            field.attname,
                            None,
                        )
                        continue
                except:
                    pass  # Fall through to normal getattr

            # Normal getattr for non-FK fields or when FK optimization fails
            current_instance = getattr(current_instance, attr, None)
        except Exception:
            current_instance = None

    return current_instance


class HookCondition:
    def check(self, instance, original_instance=None):
        raise NotImplementedError

    def __call__(self, instance, original_instance=None):
        return self.check(instance, original_instance)

    def __and__(self, other):
        return AndCondition(self, other)

    def __or__(self, other):
        return OrCondition(self, other)

    def __invert__(self):
        return NotCondition(self)


class IsNotEqual(HookCondition):
    def __init__(self, field, value, only_on_change=False):
        self.field = field
        self.value = value
        self.only_on_change = only_on_change

    def check(self, instance, original_instance=None):
        current = resolve_field_path(instance, self.field)
        if self.only_on_change:
            if original_instance is None:
                return False
            previous = resolve_field_path(original_instance, self.field)
            return previous == self.value and current != self.value
        return current != self.value


class IsEqual(HookCondition):
    def __init__(self, field, value, only_on_change=False):
        self.field = field
        self.value = value
        self.only_on_change = only_on_change

    def check(self, instance, original_instance=None):
        current = resolve_field_path(instance, self.field)

        if self.only_on_change:
            if original_instance is None:
                return False
            previous = resolve_field_path(original_instance, self.field)
            return previous != self.value and current == self.value
        return current == self.value


class HasChanged(HookCondition):
    def __init__(self, field, has_changed=True):
        self.field = field
        self.has_changed = has_changed

    def check(self, instance, original_instance=None):
        if not original_instance:
            return False

        current = resolve_field_path(instance, self.field)
        previous = resolve_field_path(original_instance, self.field)

        return (current != previous) == self.has_changed


class WasEqual(HookCondition):
    def __init__(self, field, value, only_on_change=False):
        """
        Check if a field's original value was `value`.
        If only_on_change is True, only return True when the field has changed away from that value.
        """
        self.field = field
        self.value = value
        self.only_on_change = only_on_change

    def check(self, instance, original_instance=None):
        if original_instance is None:
            return False
        previous = resolve_field_path(original_instance, self.field)
        if self.only_on_change:
            current = resolve_field_path(instance, self.field)
            return previous == self.value and current != self.value
        return previous == self.value


class ChangesTo(HookCondition):
    def __init__(self, field, value):
        """
        Check if a field's value has changed to `value`.
        Only returns True when original value != value and current value == value.
        """
        self.field = field
        self.value = value

    def check(self, instance, original_instance=None):
        if original_instance is None:
            return False
        previous = resolve_field_path(original_instance, self.field)
        current = resolve_field_path(instance, self.field)
        return previous != self.value and current == self.value


class IsGreaterThan(HookCondition):
    def __init__(self, field, value):
        self.field = field
        self.value = value

    def check(self, instance, original_instance=None):
        current = resolve_field_path(instance, self.field)
        return current is not None and current > self.value


class IsGreaterThanOrEqual(HookCondition):
    def __init__(self, field, value):
        self.field = field
        self.value = value

    def check(self, instance, original_instance=None):
        current = resolve_field_path(instance, self.field)
        return current is not None and current >= self.value


class IsLessThan(HookCondition):
    def __init__(self, field, value):
        self.field = field
        self.value = value

    def check(self, instance, original_instance=None):
        current = resolve_field_path(instance, self.field)
        return current is not None and current < self.value


class IsLessThanOrEqual(HookCondition):
    def __init__(self, field, value):
        self.field = field
        self.value = value

    def check(self, instance, original_instance=None):
        current = resolve_field_path(instance, self.field)
        return current is not None and current <= self.value


class AndCondition(HookCondition):
    def __init__(self, cond1, cond2):
        self.cond1 = cond1
        self.cond2 = cond2

    def check(self, instance, original_instance=None):
        return self.cond1.check(instance, original_instance) and self.cond2.check(
            instance,
            original_instance,
        )


class OrCondition(HookCondition):
    def __init__(self, cond1, cond2):
        self.cond1 = cond1
        self.cond2 = cond2

    def check(self, instance, original_instance=None):
        return self.cond1.check(instance, original_instance) or self.cond2.check(
            instance,
            original_instance,
        )


class NotCondition(HookCondition):
    def __init__(self, cond):
        self.cond = cond

    def check(self, instance, original_instance=None):
        return not self.cond.check(instance, original_instance)
