"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticWebsiteOrigin = exports.StaticWebsite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const aws_cloudfront_origins_1 = require("aws-cdk-lib/aws-cloudfront-origins");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const cloudfront_web_acl_1 = require("./cloudfront-web-acl");
const DEFAULT_RUNTIME_CONFIG_FILENAME = "runtime-config.json";
/**
 * Deploys a Static Website using by default a private S3 bucket as an origin and Cloudfront as the entrypoint.
 *
 * This construct configures a webAcl containing rules that are generally applicable to web applications. This
 * provides protection against exploitation of a wide range of vulnerabilities, including some of the high risk
 * and commonly occurring vulnerabilities described in OWASP publications such as OWASP Top 10.
 *
 */
class StaticWebsite extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.validateProps = (props) => {
            this.validateEncryptionSettings(props);
            props.runtimeOptions && this.validateRuntimeConfig(props.runtimeOptions);
            props.websiteBucket && this.validateBucketConfig(props.websiteBucket);
        };
        this.validateRuntimeConfig = (config) => {
            if (!config) {
                throw new Error("validateRuntimeConfig only accepts non-null RuntimeOptions.");
            }
            if (config.jsonFileName && !config.jsonFileName.endsWith(".json")) {
                throw new Error("RuntimeOptions.jsonFileName must be a json file.");
            }
        };
        this.validateBucketConfig = (bucket) => {
            if (bucket.isWebsite) {
                throw new Error("Website buckets cannot be configured as websites as this will break Cloudfront hosting!");
            }
        };
        this.validateEncryptionSettings = ({ defaultWebsiteBucketEncryption, defaultWebsiteBucketEncryptionKey, }) => {
            if (defaultWebsiteBucketEncryptionKey &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS) {
                throw new Error("Bucket encryption should be set to KMS if providing a defaultWebsiteBucketEncryptionKey.");
            }
            if (defaultWebsiteBucketEncryption &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.S3_MANAGED) {
                throw new Error("Only KMS and S3_MANAGED encryption are supported on the default bucket.");
            }
        };
        this.suppressCDKNagViolations = (props) => {
            const stack = aws_cdk_lib_1.Stack.of(this);
            !props.distributionProps?.certificate &&
                [
                    "AwsSolutions-CFR4",
                    "AwsPrototyping-CloudFrontDistributionHttpsViewerNoOutdatedSSL",
                ].forEach((RuleId) => {
                    cdk_nag_1.NagSuppressions.addResourceSuppressions(this.cloudFrontDistribution, [
                        {
                            id: RuleId,
                            reason: "Certificate is not mandatory therefore the Cloudfront certificate will be used.",
                        },
                    ]);
                });
            ["AwsSolutions-L1", "AwsPrototyping-LambdaLatestVersion"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                    {
                        id: RuleId,
                        reason: "Latest runtime cannot be configured. CDK will need to upgrade the BucketDeployment construct accordingly.",
                    },
                ], true);
            });
            ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                    {
                        id: RuleId,
                        reason: "All Policies have been scoped to a Bucket. Given Buckets can contain arbitrary content, wildcard resources with bucket scope are required.",
                        appliesTo: [
                            {
                                regex: "/^Action::s3:.*$/g",
                            },
                            {
                                regex: `/^Resource::.*$/g`,
                            },
                        ],
                    },
                ], true);
            });
            ["AwsSolutions-IAM4", "AwsPrototyping-IAMNoManagedPolicies"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                    {
                        id: RuleId,
                        reason: "Buckets can contain arbitrary content, therefore wildcard resources under a bucket are required.",
                        appliesTo: [
                            {
                                regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole$/g`,
                            },
                        ],
                    },
                ], true);
            });
            ["AwsSolutions-S1", "AwsPrototyping-S3BucketLoggingEnabled"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                    {
                        id: RuleId,
                        reason: "Access Log buckets should not have s3 bucket logging",
                    },
                ], true);
            });
        };
        this.node.setContext("@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy", true);
        this.validateProps(props);
        const accessLogsBucket = new aws_s3_1.Bucket(this, "AccessLogsBucket", {
            versioned: false,
            enforceSSL: true,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            objectOwnership: aws_s3_1.ObjectOwnership.OBJECT_WRITER,
            publicReadAccess: false,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
        });
        // S3 Bucket to hold website files
        this.websiteBucket =
            props.websiteBucket ??
                new aws_s3_1.Bucket(this, "WebsiteBucket", {
                    versioned: true,
                    enforceSSL: true,
                    autoDeleteObjects: true,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                    encryption: props.defaultWebsiteBucketEncryption ?? aws_s3_1.BucketEncryption.S3_MANAGED,
                    objectOwnership: aws_s3_1.ObjectOwnership.BUCKET_OWNER_ENFORCED,
                    encryptionKey: props.defaultWebsiteBucketEncryptionKey,
                    publicReadAccess: false,
                    blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                    serverAccessLogsPrefix: "website-access-logs",
                    serverAccessLogsBucket: accessLogsBucket,
                });
        // Web ACL
        const { distributionProps } = props;
        const webAclArn = distributionProps?.webAclId ?? props.webAclProps?.disable
            ? undefined
            : new cloudfront_web_acl_1.CloudfrontWebAcl(this, "WebsiteAcl", props.webAclProps).webAclArn;
        // Cloudfront Distribution
        const logBucket = props.distributionProps?.logBucket ||
            new aws_s3_1.Bucket(this, "DistributionLogBucket", {
                enforceSSL: true,
                autoDeleteObjects: true,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                encryption: props.defaultWebsiteBucketEncryption ?? aws_s3_1.BucketEncryption.S3_MANAGED,
                objectOwnership: aws_s3_1.ObjectOwnership.BUCKET_OWNER_PREFERRED,
                encryptionKey: props.defaultWebsiteBucketEncryptionKey,
                publicReadAccess: false,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                serverAccessLogsPrefix: "distribution-access-logs",
                serverAccessLogsBucket: accessLogsBucket,
            });
        const originAccessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, "OriginAccessIdentity");
        this.websiteBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            resources: [this.websiteBucket.bucketArn],
            actions: ["s3:ListBucket"],
            principals: [originAccessIdentity.grantPrincipal],
        }));
        const defaultRootObject = distributionProps?.defaultRootObject ?? "index.html";
        this.cloudFrontDistribution = new aws_cloudfront_1.Distribution(this, "CloudfrontDistribution", {
            ...distributionProps,
            webAclId: webAclArn,
            enableLogging: true,
            logBucket: logBucket,
            defaultBehavior: {
                ...distributionProps?.defaultBehavior,
                origin: new aws_cloudfront_origins_1.S3Origin(this.websiteBucket, {
                    originAccessIdentity,
                }),
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            defaultRootObject,
            errorResponses: distributionProps?.errorResponses ?? [
                {
                    httpStatus: 404,
                    responseHttpStatus: 200,
                    responsePagePath: `/${defaultRootObject}`,
                },
            ],
        });
        // Deploy Website
        this.bucketDeployment = new aws_s3_deployment_1.BucketDeployment(this, "WebsiteDeployment", {
            ...props.bucketDeploymentProps,
            sources: [
                aws_s3_deployment_1.Source.asset(props.websiteContentPath),
                ...(props.runtimeOptions
                    ? [
                        aws_s3_deployment_1.Source.jsonData(props.runtimeOptions?.jsonFileName ||
                            DEFAULT_RUNTIME_CONFIG_FILENAME, props.runtimeOptions?.jsonPayload),
                    ]
                    : []),
            ],
            destinationBucket: this.websiteBucket,
            // Files in the distribution's edge caches will be invalidated after files are uploaded to the destination bucket.
            distribution: this.cloudFrontDistribution,
        });
        new aws_cdk_lib_1.CfnOutput(this, "DistributionDomainName", {
            value: this.cloudFrontDistribution.domainName,
        });
        this.suppressCDKNagViolations(props);
    }
}
_a = JSII_RTTI_SYMBOL_1;
StaticWebsite[_a] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsite", version: "0.19.64" };
exports.StaticWebsite = StaticWebsite;
/**
 * If passing in distributionProps, the default behaviour.origin is a required parameter. An instance of this class can be passed in
 * to make the compiler happy.
 */
class StaticWebsiteOrigin {
    bind(_scope, _options) {
        throw new Error("This should never be called");
    }
}
_b = JSII_RTTI_SYMBOL_1;
StaticWebsiteOrigin[_b] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsiteOrigin", version: "0.19.64" };
exports.StaticWebsiteOrigin = StaticWebsiteOrigin;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhdGljLXdlYnNpdGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvc3RhdGljLXdlYnNpdGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTtzQ0FDc0M7QUFDdEMsMERBQXNEO0FBQ3RELDZDQUE4RDtBQUM5RCwrREFRb0M7QUFFcEMsK0VBQThEO0FBRTlELGlEQUFzRDtBQUl0RCwrQ0FNNEI7QUFFNUIscUVBTXVDO0FBRXZDLHFDQUEwQztBQUMxQywyQ0FBdUM7QUFDdkMsNkRBQStFO0FBRS9FLE1BQU0sK0JBQStCLEdBQUcscUJBQXFCLENBQUM7QUFpVDlEOzs7Ozs7O0dBT0c7QUFDSCxNQUFhLGFBQWMsU0FBUSxzQkFBUztJQUsxQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXlCO1FBQ2pFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFpSVgsa0JBQWEsR0FBRyxDQUFDLEtBQXlCLEVBQUUsRUFBRTtZQUNwRCxJQUFJLENBQUMsMEJBQTBCLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDdkMsS0FBSyxDQUFDLGNBQWMsSUFBSSxJQUFJLENBQUMscUJBQXFCLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1lBQ3pFLEtBQUssQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDLG9CQUFvQixDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUN4RSxDQUFDLENBQUM7UUFFTSwwQkFBcUIsR0FBRyxDQUFDLE1BQXNCLEVBQUUsRUFBRTtZQUN6RCxJQUFJLENBQUMsTUFBTSxFQUFFO2dCQUNYLE1BQU0sSUFBSSxLQUFLLENBQ2IsNkRBQTZELENBQzlELENBQUM7YUFDSDtZQUVELElBQUksTUFBTSxDQUFDLFlBQVksSUFBSSxDQUFDLE1BQU0sQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFO2dCQUNqRSxNQUFNLElBQUksS0FBSyxDQUFDLGtEQUFrRCxDQUFDLENBQUM7YUFDckU7UUFDSCxDQUFDLENBQUM7UUFFTSx5QkFBb0IsR0FBRyxDQUFDLE1BQWUsRUFBRSxFQUFFO1lBQ2pELElBQUksTUFBTSxDQUFDLFNBQVMsRUFBRTtnQkFDcEIsTUFBTSxJQUFJLEtBQUssQ0FDYix5RkFBeUYsQ0FDMUYsQ0FBQzthQUNIO1FBQ0gsQ0FBQyxDQUFDO1FBRU0sK0JBQTBCLEdBQUcsQ0FBQyxFQUNwQyw4QkFBOEIsRUFDOUIsaUNBQWlDLEdBQ2QsRUFBRSxFQUFFO1lBQ3ZCLElBQ0UsaUNBQWlDO2dCQUNqQyw4QkFBOEIsS0FBSyx5QkFBZ0IsQ0FBQyxHQUFHLEVBQ3ZEO2dCQUNBLE1BQU0sSUFBSSxLQUFLLENBQ2IsMEZBQTBGLENBQzNGLENBQUM7YUFDSDtZQUVELElBQ0UsOEJBQThCO2dCQUM5Qiw4QkFBOEIsS0FBSyx5QkFBZ0IsQ0FBQyxHQUFHO2dCQUN2RCw4QkFBOEIsS0FBSyx5QkFBZ0IsQ0FBQyxVQUFVLEVBQzlEO2dCQUNBLE1BQU0sSUFBSSxLQUFLLENBQ2IseUVBQXlFLENBQzFFLENBQUM7YUFDSDtRQUNILENBQUMsQ0FBQztRQUVNLDZCQUF3QixHQUFHLENBQUMsS0FBeUIsRUFBRSxFQUFFO1lBQy9ELE1BQU0sS0FBSyxHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzdCLENBQUMsS0FBSyxDQUFDLGlCQUFpQixFQUFFLFdBQVc7Z0JBQ25DO29CQUNFLG1CQUFtQjtvQkFDbkIsK0RBQStEO2lCQUNoRSxDQUFDLE9BQU8sQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFO29CQUNuQix5QkFBZSxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyxzQkFBc0IsRUFBRTt3QkFDbkU7NEJBQ0UsRUFBRSxFQUFFLE1BQU07NEJBQ1YsTUFBTSxFQUNKLGlGQUFpRjt5QkFDcEY7cUJBQ0YsQ0FBQyxDQUFDO2dCQUNMLENBQUMsQ0FBQyxDQUFDO1lBRUwsQ0FBQyxpQkFBaUIsRUFBRSxvQ0FBb0MsQ0FBQyxDQUFDLE9BQU8sQ0FDL0QsQ0FBQyxNQUFNLEVBQUUsRUFBRTtnQkFDVCx5QkFBZSxDQUFDLHVCQUF1QixDQUNyQyxJQUFJLEVBQ0o7b0JBQ0U7d0JBQ0UsRUFBRSxFQUFFLE1BQU07d0JBQ1YsTUFBTSxFQUNKLDJHQUEyRztxQkFDOUc7aUJBQ0YsRUFDRCxJQUFJLENBQ0wsQ0FBQztZQUNKLENBQUMsQ0FDRixDQUFDO1lBRUYsQ0FBQyxtQkFBbUIsRUFBRSx5Q0FBeUMsQ0FBQyxDQUFDLE9BQU8sQ0FDdEUsQ0FBQyxNQUFNLEVBQUUsRUFBRTtnQkFDVCx5QkFBZSxDQUFDLHVCQUF1QixDQUNyQyxJQUFJLEVBQ0o7b0JBQ0U7d0JBQ0UsRUFBRSxFQUFFLE1BQU07d0JBQ1YsTUFBTSxFQUNKLDRJQUE0STt3QkFDOUksU0FBUyxFQUFFOzRCQUNUO2dDQUNFLEtBQUssRUFBRSxvQkFBb0I7NkJBQzVCOzRCQUNEO2dDQUNFLEtBQUssRUFBRSxtQkFBbUI7NkJBQzNCO3lCQUNGO3FCQUNGO2lCQUNGLEVBQ0QsSUFBSSxDQUNMLENBQUM7WUFDSixDQUFDLENBQ0YsQ0FBQztZQUVGLENBQUMsbUJBQW1CLEVBQUUscUNBQXFDLENBQUMsQ0FBQyxPQUFPLENBQ2xFLENBQUMsTUFBTSxFQUFFLEVBQUU7Z0JBQ1QseUJBQWUsQ0FBQyx1QkFBdUIsQ0FDckMsSUFBSSxFQUNKO29CQUNFO3dCQUNFLEVBQUUsRUFBRSxNQUFNO3dCQUNWLE1BQU0sRUFDSixrR0FBa0c7d0JBQ3BHLFNBQVMsRUFBRTs0QkFDVDtnQ0FDRSxLQUFLLEVBQUUsaUJBQWlCLGdCQUFNLENBQUMsc0JBQXNCLENBQ25ELEtBQUssQ0FDTiw4REFBOEQ7NkJBQ2hFO3lCQUNGO3FCQUNGO2lCQUNGLEVBQ0QsSUFBSSxDQUNMLENBQUM7WUFDSixDQUFDLENBQ0YsQ0FBQztZQUVGLENBQUMsaUJBQWlCLEVBQUUsdUNBQXVDLENBQUMsQ0FBQyxPQUFPLENBQ2xFLENBQUMsTUFBTSxFQUFFLEVBQUU7Z0JBQ1QseUJBQWUsQ0FBQyx1QkFBdUIsQ0FDckMsSUFBSSxFQUNKO29CQUNFO3dCQUNFLEVBQUUsRUFBRSxNQUFNO3dCQUNWLE1BQU0sRUFBRSxzREFBc0Q7cUJBQy9EO2lCQUNGLEVBQ0QsSUFBSSxDQUNMLENBQUM7WUFDSixDQUFDLENBQ0YsQ0FBQztRQUNKLENBQUMsQ0FBQztRQTlRQSxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FDbEIsaURBQWlELEVBQ2pELElBQUksQ0FDTCxDQUFDO1FBRUYsSUFBSSxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUUxQixNQUFNLGdCQUFnQixHQUFHLElBQUksZUFBTSxDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtZQUM1RCxTQUFTLEVBQUUsS0FBSztZQUNoQixVQUFVLEVBQUUsSUFBSTtZQUNoQixpQkFBaUIsRUFBRSxJQUFJO1lBQ3ZCLGFBQWEsRUFBRSwyQkFBYSxDQUFDLE9BQU87WUFDcEMsVUFBVSxFQUFFLHlCQUFnQixDQUFDLFVBQVU7WUFDdkMsZUFBZSxFQUFFLHdCQUFlLENBQUMsYUFBYTtZQUM5QyxnQkFBZ0IsRUFBRSxLQUFLO1lBQ3ZCLGlCQUFpQixFQUFFLDBCQUFpQixDQUFDLFNBQVM7U0FDL0MsQ0FBQyxDQUFDO1FBRUgsa0NBQWtDO1FBQ2xDLElBQUksQ0FBQyxhQUFhO1lBQ2hCLEtBQUssQ0FBQyxhQUFhO2dCQUNuQixJQUFJLGVBQU0sQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO29CQUNoQyxTQUFTLEVBQUUsSUFBSTtvQkFDZixVQUFVLEVBQUUsSUFBSTtvQkFDaEIsaUJBQWlCLEVBQUUsSUFBSTtvQkFDdkIsYUFBYSxFQUFFLDJCQUFhLENBQUMsT0FBTztvQkFDcEMsVUFBVSxFQUNSLEtBQUssQ0FBQyw4QkFBOEIsSUFBSSx5QkFBZ0IsQ0FBQyxVQUFVO29CQUNyRSxlQUFlLEVBQUUsd0JBQWUsQ0FBQyxxQkFBcUI7b0JBQ3RELGFBQWEsRUFBRSxLQUFLLENBQUMsaUNBQWlDO29CQUN0RCxnQkFBZ0IsRUFBRSxLQUFLO29CQUN2QixpQkFBaUIsRUFBRSwwQkFBaUIsQ0FBQyxTQUFTO29CQUM5QyxzQkFBc0IsRUFBRSxxQkFBcUI7b0JBQzdDLHNCQUFzQixFQUFFLGdCQUFnQjtpQkFDekMsQ0FBQyxDQUFDO1FBRUwsVUFBVTtRQUNWLE1BQU0sRUFBRSxpQkFBaUIsRUFBRSxHQUFHLEtBQUssQ0FBQztRQUNwQyxNQUFNLFNBQVMsR0FDYixpQkFBaUIsRUFBRSxRQUFRLElBQUksS0FBSyxDQUFDLFdBQVcsRUFBRSxPQUFPO1lBQ3ZELENBQUMsQ0FBQyxTQUFTO1lBQ1gsQ0FBQyxDQUFDLElBQUkscUNBQWdCLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRSxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBRTVFLDBCQUEwQjtRQUMxQixNQUFNLFNBQVMsR0FDYixLQUFLLENBQUMsaUJBQWlCLEVBQUUsU0FBUztZQUNsQyxJQUFJLGVBQU0sQ0FBQyxJQUFJLEVBQUUsdUJBQXVCLEVBQUU7Z0JBQ3hDLFVBQVUsRUFBRSxJQUFJO2dCQUNoQixpQkFBaUIsRUFBRSxJQUFJO2dCQUN2QixhQUFhLEVBQUUsMkJBQWEsQ0FBQyxPQUFPO2dCQUNwQyxVQUFVLEVBQ1IsS0FBSyxDQUFDLDhCQUE4QixJQUFJLHlCQUFnQixDQUFDLFVBQVU7Z0JBQ3JFLGVBQWUsRUFBRSx3QkFBZSxDQUFDLHNCQUFzQjtnQkFDdkQsYUFBYSxFQUFFLEtBQUssQ0FBQyxpQ0FBaUM7Z0JBQ3RELGdCQUFnQixFQUFFLEtBQUs7Z0JBQ3ZCLGlCQUFpQixFQUFFLDBCQUFpQixDQUFDLFNBQVM7Z0JBQzlDLHNCQUFzQixFQUFFLDBCQUEwQjtnQkFDbEQsc0JBQXNCLEVBQUUsZ0JBQWdCO2FBQ3pDLENBQUMsQ0FBQztRQUVMLE1BQU0sb0JBQW9CLEdBQUcsSUFBSSxxQ0FBb0IsQ0FDbkQsSUFBSSxFQUNKLHNCQUFzQixDQUN2QixDQUFDO1FBQ0YsSUFBSSxDQUFDLGFBQWEsQ0FBQyxtQkFBbUIsQ0FDcEMsSUFBSSx5QkFBZSxDQUFDO1lBQ2xCLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDO1lBQ3pDLE9BQU8sRUFBRSxDQUFDLGVBQWUsQ0FBQztZQUMxQixVQUFVLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxjQUFjLENBQUM7U0FDbEQsQ0FBQyxDQUNILENBQUM7UUFFRixNQUFNLGlCQUFpQixHQUNyQixpQkFBaUIsRUFBRSxpQkFBaUIsSUFBSSxZQUFZLENBQUM7UUFDdkQsSUFBSSxDQUFDLHNCQUFzQixHQUFHLElBQUksNkJBQVksQ0FDNUMsSUFBSSxFQUNKLHdCQUF3QixFQUN4QjtZQUNFLEdBQUcsaUJBQWlCO1lBQ3BCLFFBQVEsRUFBRSxTQUFTO1lBQ25CLGFBQWEsRUFBRSxJQUFJO1lBQ25CLFNBQVMsRUFBRSxTQUFTO1lBQ3BCLGVBQWUsRUFBRTtnQkFDZixHQUFHLGlCQUFpQixFQUFFLGVBQWU7Z0JBQ3JDLE1BQU0sRUFBRSxJQUFJLGlDQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsRUFBRTtvQkFDdkMsb0JBQW9CO2lCQUNyQixDQUFDO2dCQUNGLG9CQUFvQixFQUFFLHFDQUFvQixDQUFDLGlCQUFpQjthQUM3RDtZQUNELGlCQUFpQjtZQUNqQixjQUFjLEVBQUUsaUJBQWlCLEVBQUUsY0FBYyxJQUFJO2dCQUNuRDtvQkFDRSxVQUFVLEVBQUUsR0FBRztvQkFDZixrQkFBa0IsRUFBRSxHQUFHO29CQUN2QixnQkFBZ0IsRUFBRSxJQUFJLGlCQUFpQixFQUFFO2lCQUMxQzthQUNGO1NBQ0YsQ0FDRixDQUFDO1FBRUYsaUJBQWlCO1FBQ2pCLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxJQUFJLG9DQUFnQixDQUFDLElBQUksRUFBRSxtQkFBbUIsRUFBRTtZQUN0RSxHQUFHLEtBQUssQ0FBQyxxQkFBcUI7WUFDOUIsT0FBTyxFQUFFO2dCQUNQLDBCQUFNLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQztnQkFDdEMsR0FBRyxDQUFDLEtBQUssQ0FBQyxjQUFjO29CQUN0QixDQUFDLENBQUM7d0JBQ0UsMEJBQU0sQ0FBQyxRQUFRLENBQ2IsS0FBSyxDQUFDLGNBQWMsRUFBRSxZQUFZOzRCQUNoQywrQkFBK0IsRUFDakMsS0FBSyxDQUFDLGNBQWMsRUFBRSxXQUFXLENBQ2xDO3FCQUNGO29CQUNILENBQUMsQ0FBQyxFQUFFLENBQUM7YUFDUjtZQUNELGlCQUFpQixFQUFFLElBQUksQ0FBQyxhQUFhO1lBQ3JDLGtIQUFrSDtZQUNsSCxZQUFZLEVBQUUsSUFBSSxDQUFDLHNCQUFzQjtTQUMxQyxDQUFDLENBQUM7UUFFSCxJQUFJLHVCQUFTLENBQUMsSUFBSSxFQUFFLHdCQUF3QixFQUFFO1lBQzVDLEtBQUssRUFBRSxJQUFJLENBQUMsc0JBQXNCLENBQUMsVUFBVTtTQUM5QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsd0JBQXdCLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDdkMsQ0FBQzs7OztBQXJJVSxzQ0FBYTtBQXlSMUI7OztHQUdHO0FBQ0gsTUFBYSxtQkFBbUI7SUFDOUIsSUFBSSxDQUFDLE1BQWlCLEVBQUUsUUFBMkI7UUFDakQsTUFBTSxJQUFJLEtBQUssQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7Ozs7QUFIVSxrREFBbUIiLCJzb3VyY2VzQ29udGVudCI6WyIvKiEgQ29weXJpZ2h0IFtBbWF6b24uY29tXShodHRwOi8vYW1hem9uLmNvbS8pLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjAgKi9cbmltcG9ydCB7IFBES05hZyB9IGZyb20gXCJAYXdzLXByb3RvdHlwaW5nLXNkay9wZGstbmFnXCI7XG5pbXBvcnQgeyBDZm5PdXRwdXQsIFJlbW92YWxQb2xpY3ksIFN0YWNrIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5pbXBvcnQge1xuICBEaXN0cmlidXRpb24sXG4gIERpc3RyaWJ1dGlvblByb3BzLFxuICBJT3JpZ2luLFxuICBPcmlnaW5BY2Nlc3NJZGVudGl0eSxcbiAgT3JpZ2luQmluZENvbmZpZyxcbiAgT3JpZ2luQmluZE9wdGlvbnMsXG4gIFZpZXdlclByb3RvY29sUG9saWN5LFxufSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWNsb3VkZnJvbnRcIjtcbmltcG9ydCAqIGFzIGNsb3VkZnJvbnQgZnJvbSBcImF3cy1jZGstbGliL2F3cy1jbG91ZGZyb250XCI7XG5pbXBvcnQgeyBTM09yaWdpbiB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtY2xvdWRmcm9udC1vcmlnaW5zXCI7XG5pbXBvcnQgKiBhcyBlYzIgZnJvbSBcImF3cy1jZGstbGliL2F3cy1lYzJcIjtcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtaWFtXCI7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1pYW1cIjtcbmltcG9ydCB7IEtleSB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3Mta21zXCI7XG5pbXBvcnQgKiBhcyBsb2dzIGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtbG9nc1wiO1xuaW1wb3J0IHtcbiAgQmxvY2tQdWJsaWNBY2Nlc3MsXG4gIEJ1Y2tldCxcbiAgQnVja2V0RW5jcnlwdGlvbixcbiAgSUJ1Y2tldCxcbiAgT2JqZWN0T3duZXJzaGlwLFxufSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLXMzXCI7XG5pbXBvcnQgKiBhcyBzMyBmcm9tIFwiYXdzLWNkay1saWIvYXdzLXMzXCI7XG5pbXBvcnQge1xuICBCdWNrZXREZXBsb3ltZW50LFxuICBDYWNoZUNvbnRyb2wsXG4gIFNlcnZlclNpZGVFbmNyeXB0aW9uLFxuICBTb3VyY2UsXG4gIFN0b3JhZ2VDbGFzcyxcbn0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1zMy1kZXBsb3ltZW50XCI7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSBcImF3cy1jZGstbGliL2NvcmVcIjtcbmltcG9ydCB7IE5hZ1N1cHByZXNzaW9ucyB9IGZyb20gXCJjZGstbmFnXCI7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tIFwiY29uc3RydWN0c1wiO1xuaW1wb3J0IHsgQ2xvdWRmcm9udFdlYkFjbCwgQ2xvdWRGcm9udFdlYkFjbFByb3BzIH0gZnJvbSBcIi4vY2xvdWRmcm9udC13ZWItYWNsXCI7XG5cbmNvbnN0IERFRkFVTFRfUlVOVElNRV9DT05GSUdfRklMRU5BTUUgPSBcInJ1bnRpbWUtY29uZmlnLmpzb25cIjtcblxuLyoqXG4gKiBCdWNrZXQgRGVwbG95bWVudCBwcm9wcy5cbiAqXG4gKiBOT1RFOiBmb3JrZWQgZnJvbSBhd3MtY2RrLWxpYi9zMy1kZXBsb3ltZW50IHdpdGhvdXQgYW55IHJlcXVpcmVkIHByb3BzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEJ1Y2tldERlcGxveW1lbnRQcm9wcyB7XG4gIC8qKlxuICAgKiBLZXkgcHJlZml4IGluIHRoZSBkZXN0aW5hdGlvbiBidWNrZXQuXG4gICAqXG4gICAqIE11c3QgYmUgPD0xMDQgY2hhcmFjdGVyc1xuICAgKlxuICAgKiBAZGVmYXVsdCBcIi9cIiAodW56aXAgdG8gcm9vdCBvZiB0aGUgZGVzdGluYXRpb24gYnVja2V0KVxuICAgKi9cbiAgcmVhZG9ubHkgZGVzdGluYXRpb25LZXlQcmVmaXg/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBJZiB0aGlzIGlzIHNldCwgdGhlIHppcCBmaWxlIHdpbGwgYmUgc3luY2VkIHRvIHRoZSBkZXN0aW5hdGlvbiBTMyBidWNrZXQgYW5kIGV4dHJhY3RlZC5cbiAgICogSWYgZmFsc2UsIHRoZSBmaWxlIHdpbGwgcmVtYWluIHppcHBlZCBpbiB0aGUgZGVzdGluYXRpb24gYnVja2V0LlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBleHRyYWN0PzogYm9vbGVhbjtcbiAgLyoqXG4gICAqIElmIHRoaXMgaXMgc2V0LCBtYXRjaGluZyBmaWxlcyBvciBvYmplY3RzIHdpbGwgYmUgZXhjbHVkZWQgZnJvbSB0aGUgZGVwbG95bWVudCdzIHN5bmNcbiAgICogY29tbWFuZC4gVGhpcyBjYW4gYmUgdXNlZCB0byBleGNsdWRlIGEgZmlsZSBmcm9tIGJlaW5nIHBydW5lZCBpbiB0aGUgZGVzdGluYXRpb24gYnVja2V0LlxuICAgKlxuICAgKiBJZiB5b3Ugd2FudCB0byBqdXN0IGV4Y2x1ZGUgZmlsZXMgZnJvbSB0aGUgZGVwbG95bWVudCBwYWNrYWdlICh3aGljaCBleGNsdWRlcyB0aGVzZSBmaWxlc1xuICAgKiBldmFsdWF0ZWQgd2hlbiBpbnZhbGlkYXRpbmcgdGhlIGFzc2V0KSwgeW91IHNob3VsZCBsZXZlcmFnZSB0aGUgYGV4Y2x1ZGVgIHByb3BlcnR5IG9mXG4gICAqIGBBc3NldE9wdGlvbnNgIHdoZW4gZGVmaW5pbmcgeW91ciBzb3VyY2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gZXhjbHVkZSBmaWx0ZXJzIGFyZSB1c2VkXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NsaS9sYXRlc3QvcmVmZXJlbmNlL3MzL2luZGV4Lmh0bWwjdXNlLW9mLWV4Y2x1ZGUtYW5kLWluY2x1ZGUtZmlsdGVyc1xuICAgKi9cbiAgcmVhZG9ubHkgZXhjbHVkZT86IHN0cmluZ1tdO1xuICAvKipcbiAgICogSWYgdGhpcyBpcyBzZXQsIG1hdGNoaW5nIGZpbGVzIG9yIG9iamVjdHMgd2lsbCBiZSBpbmNsdWRlZCB3aXRoIHRoZSBkZXBsb3ltZW50J3Mgc3luY1xuICAgKiBjb21tYW5kLiBTaW5jZSBhbGwgZmlsZXMgZnJvbSB0aGUgZGVwbG95bWVudCBwYWNrYWdlIGFyZSBpbmNsdWRlZCBieSBkZWZhdWx0LCB0aGlzIHByb3BlcnR5XG4gICAqIGlzIHVzdWFsbHkgbGV2ZXJhZ2VkIGFsb25nc2lkZSBhbiBgZXhjbHVkZWAgZmlsdGVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGluY2x1ZGUgZmlsdGVycyBhcmUgdXNlZCBhbmQgYWxsIGZpbGVzIGFyZSBpbmNsdWRlZCB3aXRoIHRoZSBzeW5jIGNvbW1hbmRcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2xpL2xhdGVzdC9yZWZlcmVuY2UvczMvaW5kZXguaHRtbCN1c2Utb2YtZXhjbHVkZS1hbmQtaW5jbHVkZS1maWx0ZXJzXG4gICAqL1xuICByZWFkb25seSBpbmNsdWRlPzogc3RyaW5nW107XG4gIC8qKlxuICAgKiBJZiB0aGlzIGlzIHNldCB0byBmYWxzZSwgZmlsZXMgaW4gdGhlIGRlc3RpbmF0aW9uIGJ1Y2tldCB0aGF0XG4gICAqIGRvIG5vdCBleGlzdCBpbiB0aGUgYXNzZXQsIHdpbGwgTk9UIGJlIGRlbGV0ZWQgZHVyaW5nIGRlcGxveW1lbnQgKGNyZWF0ZS91cGRhdGUpLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jbGkvbGF0ZXN0L3JlZmVyZW5jZS9zMy9zeW5jLmh0bWxcbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJ1bmU/OiBib29sZWFuO1xuICAvKipcbiAgICogSWYgdGhpcyBpcyBzZXQgdG8gXCJmYWxzZVwiLCB0aGUgZGVzdGluYXRpb24gZmlsZXMgd2lsbCBiZSBkZWxldGVkIHdoZW4gdGhlXG4gICAqIHJlc291cmNlIGlzIGRlbGV0ZWQgb3IgdGhlIGRlc3RpbmF0aW9uIGlzIHVwZGF0ZWQuXG4gICAqXG4gICAqIE5PVElDRTogQ29uZmlndXJpbmcgdGhpcyB0byBcImZhbHNlXCIgbWlnaHQgaGF2ZSBvcGVyYXRpb25hbCBpbXBsaWNhdGlvbnMuIFBsZWFzZVxuICAgKiB2aXNpdCB0byB0aGUgcGFja2FnZSBkb2N1bWVudGF0aW9uIHJlZmVycmVkIGJlbG93IHRvIG1ha2Ugc3VyZSB5b3UgZnVsbHkgdW5kZXJzdGFuZCB0aG9zZSBpbXBsaWNhdGlvbnMuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL3RyZWUvbWFpbi9wYWNrYWdlcy8lNDBhd3MtY2RrL2F3cy1zMy1kZXBsb3ltZW50I3JldGFpbi1vbi1kZWxldGVcbiAgICogQGRlZmF1bHQgdHJ1ZSAtIHdoZW4gcmVzb3VyY2UgaXMgZGVsZXRlZC91cGRhdGVkLCBmaWxlcyBhcmUgcmV0YWluZWRcbiAgICovXG4gIHJlYWRvbmx5IHJldGFpbk9uRGVsZXRlPzogYm9vbGVhbjtcbiAgLyoqXG4gICAqIFRoZSBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbiB1c2luZyB0aGUgZGVzdGluYXRpb24gYnVja2V0IGFzIGFuIG9yaWdpbi5cbiAgICogRmlsZXMgaW4gdGhlIGRpc3RyaWJ1dGlvbidzIGVkZ2UgY2FjaGVzIHdpbGwgYmUgaW52YWxpZGF0ZWQgYWZ0ZXJcbiAgICogZmlsZXMgYXJlIHVwbG9hZGVkIHRvIHRoZSBkZXN0aW5hdGlvbiBidWNrZXQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gaW52YWxpZGF0aW9uIG9jY3Vyc1xuICAgKi9cbiAgcmVhZG9ubHkgZGlzdHJpYnV0aW9uPzogY2xvdWRmcm9udC5JRGlzdHJpYnV0aW9uO1xuICAvKipcbiAgICogVGhlIGZpbGUgcGF0aHMgdG8gaW52YWxpZGF0ZSBpbiB0aGUgQ2xvdWRGcm9udCBkaXN0cmlidXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQWxsIGZpbGVzIHVuZGVyIHRoZSBkZXN0aW5hdGlvbiBidWNrZXQga2V5IHByZWZpeCB3aWxsIGJlIGludmFsaWRhdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgZGlzdHJpYnV0aW9uUGF0aHM/OiBzdHJpbmdbXTtcbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgZGF5cyB0aGF0IHRoZSBsYW1iZGEgZnVuY3Rpb24ncyBsb2cgZXZlbnRzIGFyZSBrZXB0IGluIENsb3VkV2F0Y2ggTG9ncy5cbiAgICpcbiAgICogQGRlZmF1bHQgbG9ncy5SZXRlbnRpb25EYXlzLklORklOSVRFXG4gICAqL1xuICByZWFkb25seSBsb2dSZXRlbnRpb24/OiBsb2dzLlJldGVudGlvbkRheXM7XG4gIC8qKlxuICAgKiBUaGUgYW1vdW50IG9mIG1lbW9yeSAoaW4gTWlCKSB0byBhbGxvY2F0ZSB0byB0aGUgQVdTIExhbWJkYSBmdW5jdGlvbiB3aGljaFxuICAgKiByZXBsaWNhdGVzIHRoZSBmaWxlcyBmcm9tIHRoZSBDREsgYnVja2V0IHRvIHRoZSBkZXN0aW5hdGlvbiBidWNrZXQuXG4gICAqXG4gICAqIElmIHlvdSBhcmUgZGVwbG95aW5nIGxhcmdlIGZpbGVzLCB5b3Ugd2lsbCBuZWVkIHRvIGluY3JlYXNlIHRoaXMgbnVtYmVyXG4gICAqIGFjY29yZGluZ2x5LlxuICAgKlxuICAgKiBAZGVmYXVsdCAxMjhcbiAgICovXG4gIHJlYWRvbmx5IG1lbW9yeUxpbWl0PzogbnVtYmVyO1xuICAvKipcbiAgICogVGhlIHNpemUgb2YgdGhlIEFXUyBMYW1iZGEgZnVuY3Rpb27igJlzIC90bXAgZGlyZWN0b3J5IGluIE1pQi5cbiAgICpcbiAgICogQGRlZmF1bHQgNTEyIE1pQlxuICAgKi9cbiAgcmVhZG9ubHkgZXBoZW1lcmFsU3RvcmFnZVNpemU/OiBjZGsuU2l6ZTtcbiAgLyoqXG4gICAqICBNb3VudCBhbiBFRlMgZmlsZSBzeXN0ZW0uIEVuYWJsZSB0aGlzIGlmIHlvdXIgYXNzZXRzIGFyZSBsYXJnZSBhbmQgeW91IGVuY291bnRlciBkaXNrIHNwYWNlIGVycm9ycy5cbiAgICogIEVuYWJsaW5nIHRoaXMgb3B0aW9uIHdpbGwgcmVxdWlyZSBhIFZQQyB0byBiZSBzcGVjaWZpZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gRUZTLiBMYW1iZGEgaGFzIGFjY2VzcyBvbmx5IHRvIDUxMk1CIG9mIGRpc2sgc3BhY2UuXG4gICAqL1xuICByZWFkb25seSB1c2VFZnM/OiBib29sZWFuO1xuICAvKipcbiAgICogRXhlY3V0aW9uIHJvbGUgYXNzb2NpYXRlZCB3aXRoIHRoaXMgZnVuY3Rpb25cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBIHJvbGUgaXMgYXV0b21hdGljYWxseSBjcmVhdGVkXG4gICAqL1xuICByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuICAvKipcbiAgICogVXNlci1kZWZpbmVkIG9iamVjdCBtZXRhZGF0YSB0byBiZSBzZXQgb24gYWxsIG9iamVjdHMgaW4gdGhlIGRlcGxveW1lbnRcbiAgICogQGRlZmF1bHQgLSBObyB1c2VyIG1ldGFkYXRhIGlzIHNldFxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L1VzaW5nTWV0YWRhdGEuaHRtbCNVc2VyTWV0YWRhdGFcbiAgICovXG4gIHJlYWRvbmx5IG1ldGFkYXRhPzoge1xuICAgIFtrZXk6IHN0cmluZ106IHN0cmluZztcbiAgfTtcbiAgLyoqXG4gICAqIFN5c3RlbS1kZWZpbmVkIGNhY2hlLWNvbnRyb2wgbWV0YWRhdGEgdG8gYmUgc2V0IG9uIGFsbCBvYmplY3RzIGluIHRoZSBkZXBsb3ltZW50LlxuICAgKiBAZGVmYXVsdCAtIE5vdCBzZXQuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvVXNpbmdNZXRhZGF0YS5odG1sI1N5c01ldGFkYXRhXG4gICAqL1xuICByZWFkb25seSBjYWNoZUNvbnRyb2w/OiBDYWNoZUNvbnRyb2xbXTtcbiAgLyoqXG4gICAqIFN5c3RlbS1kZWZpbmVkIGNhY2hlLWRpc3Bvc2l0aW9uIG1ldGFkYXRhIHRvIGJlIHNldCBvbiBhbGwgb2JqZWN0cyBpbiB0aGUgZGVwbG95bWVudC5cbiAgICogQGRlZmF1bHQgLSBOb3Qgc2V0LlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L1VzaW5nTWV0YWRhdGEuaHRtbCNTeXNNZXRhZGF0YVxuICAgKi9cbiAgcmVhZG9ubHkgY29udGVudERpc3Bvc2l0aW9uPzogc3RyaW5nO1xuICAvKipcbiAgICogU3lzdGVtLWRlZmluZWQgY29udGVudC1lbmNvZGluZyBtZXRhZGF0YSB0byBiZSBzZXQgb24gYWxsIG9iamVjdHMgaW4gdGhlIGRlcGxveW1lbnQuXG4gICAqIEBkZWZhdWx0IC0gTm90IHNldC5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ01ldGFkYXRhLmh0bWwjU3lzTWV0YWRhdGFcbiAgICovXG4gIHJlYWRvbmx5IGNvbnRlbnRFbmNvZGluZz86IHN0cmluZztcbiAgLyoqXG4gICAqIFN5c3RlbS1kZWZpbmVkIGNvbnRlbnQtbGFuZ3VhZ2UgbWV0YWRhdGEgdG8gYmUgc2V0IG9uIGFsbCBvYmplY3RzIGluIHRoZSBkZXBsb3ltZW50LlxuICAgKiBAZGVmYXVsdCAtIE5vdCBzZXQuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvVXNpbmdNZXRhZGF0YS5odG1sI1N5c01ldGFkYXRhXG4gICAqL1xuICByZWFkb25seSBjb250ZW50TGFuZ3VhZ2U/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBTeXN0ZW0tZGVmaW5lZCBjb250ZW50LXR5cGUgbWV0YWRhdGEgdG8gYmUgc2V0IG9uIGFsbCBvYmplY3RzIGluIHRoZSBkZXBsb3ltZW50LlxuICAgKiBAZGVmYXVsdCAtIE5vdCBzZXQuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvVXNpbmdNZXRhZGF0YS5odG1sI1N5c01ldGFkYXRhXG4gICAqL1xuICByZWFkb25seSBjb250ZW50VHlwZT86IHN0cmluZztcbiAgLyoqXG4gICAqIFN5c3RlbS1kZWZpbmVkIGV4cGlyZXMgbWV0YWRhdGEgdG8gYmUgc2V0IG9uIGFsbCBvYmplY3RzIGluIHRoZSBkZXBsb3ltZW50LlxuICAgKiBAZGVmYXVsdCAtIFRoZSBvYmplY3RzIGluIHRoZSBkaXN0cmlidXRpb24gd2lsbCBub3QgZXhwaXJlLlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L1VzaW5nTWV0YWRhdGEuaHRtbCNTeXNNZXRhZGF0YVxuICAgKi9cbiAgcmVhZG9ubHkgZXhwaXJlcz86IGNkay5FeHBpcmF0aW9uO1xuICAvKipcbiAgICogU3lzdGVtLWRlZmluZWQgeC1hbXotc2VydmVyLXNpZGUtZW5jcnlwdGlvbiBtZXRhZGF0YSB0byBiZSBzZXQgb24gYWxsIG9iamVjdHMgaW4gdGhlIGRlcGxveW1lbnQuXG4gICAqIEBkZWZhdWx0IC0gU2VydmVyIHNpZGUgZW5jcnlwdGlvbiBpcyBub3QgdXNlZC5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ01ldGFkYXRhLmh0bWwjU3lzTWV0YWRhdGFcbiAgICovXG4gIHJlYWRvbmx5IHNlcnZlclNpZGVFbmNyeXB0aW9uPzogU2VydmVyU2lkZUVuY3J5cHRpb247XG4gIC8qKlxuICAgKiBTeXN0ZW0tZGVmaW5lZCB4LWFtei1zdG9yYWdlLWNsYXNzIG1ldGFkYXRhIHRvIGJlIHNldCBvbiBhbGwgb2JqZWN0cyBpbiB0aGUgZGVwbG95bWVudC5cbiAgICogQGRlZmF1bHQgLSBEZWZhdWx0IHN0b3JhZ2UtY2xhc3MgZm9yIHRoZSBidWNrZXQgaXMgdXNlZC5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ01ldGFkYXRhLmh0bWwjU3lzTWV0YWRhdGFcbiAgICovXG4gIHJlYWRvbmx5IHN0b3JhZ2VDbGFzcz86IFN0b3JhZ2VDbGFzcztcbiAgLyoqXG4gICAqIFN5c3RlbS1kZWZpbmVkIHgtYW16LXdlYnNpdGUtcmVkaXJlY3QtbG9jYXRpb24gbWV0YWRhdGEgdG8gYmUgc2V0IG9uIGFsbCBvYmplY3RzIGluIHRoZSBkZXBsb3ltZW50LlxuICAgKiBAZGVmYXVsdCAtIE5vIHdlYnNpdGUgcmVkaXJlY3Rpb24uXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvVXNpbmdNZXRhZGF0YS5odG1sI1N5c01ldGFkYXRhXG4gICAqL1xuICByZWFkb25seSB3ZWJzaXRlUmVkaXJlY3RMb2NhdGlvbj86IHN0cmluZztcbiAgLyoqXG4gICAqIFN5c3RlbS1kZWZpbmVkIHgtYW16LXNlcnZlci1zaWRlLWVuY3J5cHRpb24tYXdzLWttcy1rZXktaWQgbWV0YWRhdGEgdG8gYmUgc2V0IG9uIGFsbCBvYmplY3RzIGluIHRoZSBkZXBsb3ltZW50LlxuICAgKiBAZGVmYXVsdCAtIE5vdCBzZXQuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvVXNpbmdNZXRhZGF0YS5odG1sI1N5c01ldGFkYXRhXG4gICAqL1xuICByZWFkb25seSBzZXJ2ZXJTaWRlRW5jcnlwdGlvbkF3c0ttc0tleUlkPzogc3RyaW5nO1xuICAvKipcbiAgICogU3lzdGVtLWRlZmluZWQgeC1hbXotc2VydmVyLXNpZGUtZW5jcnlwdGlvbi1jdXN0b21lci1hbGdvcml0aG0gbWV0YWRhdGEgdG8gYmUgc2V0IG9uIGFsbCBvYmplY3RzIGluIHRoZSBkZXBsb3ltZW50LlxuICAgKiBXYXJuaW5nOiBUaGlzIGlzIG5vdCBhIHVzZWZ1bCBwYXJhbWV0ZXIgdW50aWwgdGhpcyBidWcgaXMgZml4ZWQ6IGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9pc3N1ZXMvNjA4MFxuICAgKiBAZGVmYXVsdCAtIE5vdCBzZXQuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvU2VydmVyU2lkZUVuY3J5cHRpb25DdXN0b21lcktleXMuaHRtbCNzc2UtYy1ob3ctdG8tcHJvZ3JhbW1hdGljYWxseS1pbnRyb1xuICAgKi9cbiAgcmVhZG9ubHkgc2VydmVyU2lkZUVuY3J5cHRpb25DdXN0b21lckFsZ29yaXRobT86IHN0cmluZztcbiAgLyoqXG4gICAqIFN5c3RlbS1kZWZpbmVkIHgtYW16LWFjbCBtZXRhZGF0YSB0byBiZSBzZXQgb24gYWxsIG9iamVjdHMgaW4gdGhlIGRlcGxveW1lbnQuXG4gICAqIEBkZWZhdWx0IC0gTm90IHNldC5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L3VzZXJndWlkZS9hY2wtb3ZlcnZpZXcuaHRtbCNjYW5uZWQtYWNsXG4gICAqL1xuICByZWFkb25seSBhY2Nlc3NDb250cm9sPzogczMuQnVja2V0QWNjZXNzQ29udHJvbDtcbiAgLyoqXG4gICAqIFRoZSBWUEMgbmV0d29yayB0byBwbGFjZSB0aGUgZGVwbG95bWVudCBsYW1iZGEgaGFuZGxlciBpbi5cbiAgICogVGhpcyBpcyByZXF1aXJlZCBpZiBgdXNlRWZzYCBpcyBzZXQuXG4gICAqXG4gICAqIEBkZWZhdWx0IE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuICAvKipcbiAgICogV2hlcmUgaW4gdGhlIFZQQyB0byBwbGFjZSB0aGUgZGVwbG95bWVudCBsYW1iZGEgaGFuZGxlci5cbiAgICogT25seSB1c2VkIGlmICd2cGMnIGlzIHN1cHBsaWVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRoZSBWcGMgZGVmYXVsdCBzdHJhdGVneSBpZiBub3Qgc3BlY2lmaWVkXG4gICAqL1xuICByZWFkb25seSB2cGNTdWJuZXRzPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcbiAgLyoqXG4gICAqIElmIHNldCB0byB0cnVlLCB1cGxvYWRzIHdpbGwgcHJlY29tcHV0ZSB0aGUgdmFsdWUgb2YgYHgtYW16LWNvbnRlbnQtc2hhMjU2YFxuICAgKiBhbmQgaW5jbHVkZSBpdCBpbiB0aGUgc2lnbmVkIFMzIHJlcXVlc3QgaGVhZGVycy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBgeC1hbXotY29udGVudC1zaGEyNTZgIHdpbGwgbm90IGJlIGNvbXB1dGVkXG4gICAqL1xuICByZWFkb25seSBzaWduQ29udGVudD86IGJvb2xlYW47XG59XG5cbi8qKlxuICogRHluYW1pYyBjb25maWd1cmF0aW9uIHdoaWNoIGdldHMgcmVzb2x2ZWQgb25seSBkdXJpbmcgZGVwbG95bWVudC5cbiAqXG4gKiBAZXhhbXBsZVxuICpcbiAqIC8vIFdpbGwgc3RvcmUgYSBKU09OIGZpbGUgY2FsbGVkIHJ1bnRpbWUtY29uZmlnLmpzb24gaW4gdGhlIHJvb3Qgb2YgdGhlIFN0YXRpY1dlYnNpdGUgUzMgYnVja2V0IGNvbnRhaW5pbmcgYW55XG4gKiAvLyBhbmQgYWxsIHJlc29sdmVkIHZhbHVlcy5cbiAqIGNvbnN0IHJ1bnRpbWVDb25maWcgPSB7anNvblBheWxvYWQ6IHtidWNrZXRBcm46IHMzQnVja2V0LmJ1Y2tldEFybn19O1xuICogbmV3IFN0YXRpY1dlYnNpdGUoc2NvcGUsICdTdGF0aWNXZWJzaXRlJywge3dlYnNpdGVDb250ZW50UGF0aDogJ3BhdGgvdG8vd2Vic2l0ZScsIHJ1bnRpbWVDb25maWd9KTtcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSdW50aW1lT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBGaWxlIG5hbWUgdG8gc3RvcmUgcnVudGltZSBjb25maWd1cmF0aW9uIChqc29uUGF5bG9hZCkuXG4gICAqXG4gICAqIE11c3QgZm9sbG93IHBhdHRlcm46ICcqLmpzb24nXG4gICAqXG4gICAqIEBkZWZhdWx0IFwicnVudGltZS1jb25maWcuanNvblwiXG4gICAqL1xuICByZWFkb25seSBqc29uRmlsZU5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFyYml0cmFyeSBKU09OIHBheWxvYWQgY29udGFpbmluZyBydW50aW1lIHZhbHVlcyB0byBkZXBsb3kuIFR5cGljYWxseSB0aGlzIGNvbnRhaW5zIHJlc291cmNlQXJucywgZXRjIHdoaWNoXG4gICAqIGFyZSBvbmx5IGtub3duIGF0IGRlcGxveSB0aW1lLlxuICAgKlxuICAgKiBAZXhhbXBsZSB7IHVzZXJQb29sSWQ6IHNvbWUudXNlclBvb2wudXNlclBvb2xJZCwgc29tZVJlc291cmNlQXJuOiBzb21lLnJlc291cmNlLkFybiB9XG4gICAqL1xuICByZWFkb25seSBqc29uUGF5bG9hZDogYW55O1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGNvbmZpZ3VyaW5nIHRoZSBTdGF0aWNXZWJzaXRlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFN0YXRpY1dlYnNpdGVQcm9wcyB7XG4gIC8qKlxuICAgKiBQYXRoIHRvIHRoZSBkaXJlY3RvcnkgY29udGFpbmluZyB0aGUgc3RhdGljIHdlYnNpdGUgZmlsZXMgYW5kIGFzc2V0cy4gVGhpcyBkaXJlY3RvcnkgbXVzdCBjb250YWluIGFuIGluZGV4Lmh0bWwgZmlsZS5cbiAgICovXG4gIHJlYWRvbmx5IHdlYnNpdGVDb250ZW50UGF0aDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBEeW5hbWljIGNvbmZpZ3VyYXRpb24gd2hpY2ggZ2V0cyByZXNvbHZlZCBvbmx5IGR1cmluZyBkZXBsb3ltZW50LlxuICAgKi9cbiAgcmVhZG9ubHkgcnVudGltZU9wdGlvbnM/OiBSdW50aW1lT3B0aW9ucztcblxuICAvKipcbiAgICogQnVja2V0IGVuY3J5cHRpb24gdG8gdXNlIGZvciB0aGUgZGVmYXVsdCBidWNrZXQuXG4gICAqXG4gICAqIFN1cHBvcnRlZCBvcHRpb25zIGFyZSBLTVMgb3IgUzNNQU5BR0VELlxuICAgKlxuICAgKiBOb3RlOiBJZiBwbGFubmluZyB0byB1c2UgS01TLCBlbnN1cmUgeW91IGFzc29jaWF0ZSBhIExhbWJkYSBFZGdlIGZ1bmN0aW9uIHRvIHNpZ24gcmVxdWVzdHMgdG8gUzMgYXMgT0FJIGRvZXMgbm90IGN1cnJlbnRseSBzdXBwb3J0IEtNUyBlbmNyeXB0aW9uLiBSZWZlciB0byB7QGxpbmsgaHR0cHM6Ly9hd3MuYW1hem9uLmNvbS9ibG9ncy9uZXR3b3JraW5nLWFuZC1jb250ZW50LWRlbGl2ZXJ5L3NlcnZpbmctc3NlLWttcy1lbmNyeXB0ZWQtY29udGVudC1mcm9tLXMzLXVzaW5nLWNsb3VkZnJvbnQvfVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFwiUzNNQU5BR0VEXCJcbiAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHRXZWJzaXRlQnVja2V0RW5jcnlwdGlvbj86IEJ1Y2tldEVuY3J5cHRpb247XG5cbiAgLyoqXG4gICAqIEEgcHJlZGVmaW5lZCBLTVMgY3VzdG9tZXIgZW5jcnlwdGlvbiBrZXkgdG8gdXNlIGZvciB0aGUgZGVmYXVsdCBidWNrZXQgdGhhdCBnZXRzIGNyZWF0ZWQuXG4gICAqXG4gICAqIE5vdGU6IFRoaXMgaXMgb25seSB1c2VkIGlmIHRoZSB3ZWJzaXRlQnVja2V0IGlzIGxlZnQgdW5kZWZpbmVkLCBvdGhlcndpc2UgYWxsIHNldHRpbmdzIGZyb20gdGhlIHByb3ZpZGVkIHdlYnNpdGVCdWNrZXQgd2lsbCBiZSB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgZGVmYXVsdFdlYnNpdGVCdWNrZXRFbmNyeXB0aW9uS2V5PzogS2V5O1xuXG4gIC8qKlxuICAgKiBQcmVkZWZpbmVkIGJ1Y2tldCB0byBkZXBsb3kgdGhlIHdlYnNpdGUgaW50by5cbiAgICovXG4gIHJlYWRvbmx5IHdlYnNpdGVCdWNrZXQ/OiBJQnVja2V0O1xuXG4gIC8qKlxuICAgKiBDdXN0b20gZGlzdHJpYnV0aW9uIHByb3BlcnRpZXMuXG4gICAqXG4gICAqIE5vdGU6IGRlZmF1bHRCZWhhdmlvdXIub3JpZ2luIGlzIGEgcmVxdWlyZWQgcGFyYW1ldGVyLCBob3dldmVyIGl0IHdpbGwgbm90IGJlIHVzZWQgYXMgdGhpcyBjb25zdHJ1Y3Qgd2lsbCB3aXJlIGl0IG9uIHlvdXIgYmVoYWxmLlxuICAgKiBZb3Ugd2lsbCBuZWVkIHRvIHBhc3MgaW4gYW4gaW5zdGFuY2Ugb2YgU3RhdGljV2Vic2l0ZU9yaWdpbiAoTm9PcCkgdG8ga2VlcCB0aGUgY29tcGlsZXIgaGFwcHkuXG4gICAqL1xuICByZWFkb25seSBkaXN0cmlidXRpb25Qcm9wcz86IERpc3RyaWJ1dGlvblByb3BzO1xuXG4gIC8qKlxuICAgKiBDdXN0b20gYnVja2V0IGRlcGxveW1lbnQgcHJvcGVydGllcy5cbiAgICpcbiAgICogYGBgXG4gICAqL1xuICByZWFkb25seSBidWNrZXREZXBsb3ltZW50UHJvcHM/OiBCdWNrZXREZXBsb3ltZW50UHJvcHM7XG5cbiAgLyoqXG4gICAqIExpbWl0ZWQgY29uZmlndXJhdGlvbiBzZXR0aW5ncyBmb3IgdGhlIGdlbmVyYXRlZCB3ZWJBY2wuIEZvciBtb3JlIGFkdmFuY2VkIHNldHRpbmdzLCBjcmVhdGUgeW91ciBvd24gQUNMIGFuZCBwYXNzIGluIHRoZSB3ZWJBY2xJZCBhcyBhIHBhcmFtIHRvIGRpc3RyaWJ1dGlvblByb3BzLlxuICAgKlxuICAgKiBOb3RlOiBJZiBwYXNzIGluIHlvdXIgb3duIEFDTCwgbWFrZSBzdXJlIHRoZSBTQ09QRSBpcyBDTE9VREZST05UIGFuZCBpdCBpcyBjcmVhdGVkIGluIHVzLWVhc3QtMS5cbiAgICovXG4gIHJlYWRvbmx5IHdlYkFjbFByb3BzPzogQ2xvdWRGcm9udFdlYkFjbFByb3BzO1xufVxuXG4vKipcbiAqIERlcGxveXMgYSBTdGF0aWMgV2Vic2l0ZSB1c2luZyBieSBkZWZhdWx0IGEgcHJpdmF0ZSBTMyBidWNrZXQgYXMgYW4gb3JpZ2luIGFuZCBDbG91ZGZyb250IGFzIHRoZSBlbnRyeXBvaW50LlxuICpcbiAqIFRoaXMgY29uc3RydWN0IGNvbmZpZ3VyZXMgYSB3ZWJBY2wgY29udGFpbmluZyBydWxlcyB0aGF0IGFyZSBnZW5lcmFsbHkgYXBwbGljYWJsZSB0byB3ZWIgYXBwbGljYXRpb25zLiBUaGlzXG4gKiBwcm92aWRlcyBwcm90ZWN0aW9uIGFnYWluc3QgZXhwbG9pdGF0aW9uIG9mIGEgd2lkZSByYW5nZSBvZiB2dWxuZXJhYmlsaXRpZXMsIGluY2x1ZGluZyBzb21lIG9mIHRoZSBoaWdoIHJpc2tcbiAqIGFuZCBjb21tb25seSBvY2N1cnJpbmcgdnVsbmVyYWJpbGl0aWVzIGRlc2NyaWJlZCBpbiBPV0FTUCBwdWJsaWNhdGlvbnMgc3VjaCBhcyBPV0FTUCBUb3AgMTAuXG4gKlxuICovXG5leHBvcnQgY2xhc3MgU3RhdGljV2Vic2l0ZSBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHB1YmxpYyByZWFkb25seSB3ZWJzaXRlQnVja2V0OiBJQnVja2V0O1xuICBwdWJsaWMgcmVhZG9ubHkgY2xvdWRGcm9udERpc3RyaWJ1dGlvbjogRGlzdHJpYnV0aW9uO1xuICBwdWJsaWMgcmVhZG9ubHkgYnVja2V0RGVwbG95bWVudDogQnVja2V0RGVwbG95bWVudDtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogU3RhdGljV2Vic2l0ZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMubm9kZS5zZXRDb250ZXh0KFxuICAgICAgXCJAYXdzLWNkay9hd3MtczM6c2VydmVyQWNjZXNzTG9nc1VzZUJ1Y2tldFBvbGljeVwiLFxuICAgICAgdHJ1ZVxuICAgICk7XG5cbiAgICB0aGlzLnZhbGlkYXRlUHJvcHMocHJvcHMpO1xuXG4gICAgY29uc3QgYWNjZXNzTG9nc0J1Y2tldCA9IG5ldyBCdWNrZXQodGhpcywgXCJBY2Nlc3NMb2dzQnVja2V0XCIsIHtcbiAgICAgIHZlcnNpb25lZDogZmFsc2UsXG4gICAgICBlbmZvcmNlU1NMOiB0cnVlLFxuICAgICAgYXV0b0RlbGV0ZU9iamVjdHM6IHRydWUsXG4gICAgICByZW1vdmFsUG9saWN5OiBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgICBlbmNyeXB0aW9uOiBCdWNrZXRFbmNyeXB0aW9uLlMzX01BTkFHRUQsXG4gICAgICBvYmplY3RPd25lcnNoaXA6IE9iamVjdE93bmVyc2hpcC5PQkpFQ1RfV1JJVEVSLFxuICAgICAgcHVibGljUmVhZEFjY2VzczogZmFsc2UsXG4gICAgICBibG9ja1B1YmxpY0FjY2VzczogQmxvY2tQdWJsaWNBY2Nlc3MuQkxPQ0tfQUxMLFxuICAgIH0pO1xuXG4gICAgLy8gUzMgQnVja2V0IHRvIGhvbGQgd2Vic2l0ZSBmaWxlc1xuICAgIHRoaXMud2Vic2l0ZUJ1Y2tldCA9XG4gICAgICBwcm9wcy53ZWJzaXRlQnVja2V0ID8/XG4gICAgICBuZXcgQnVja2V0KHRoaXMsIFwiV2Vic2l0ZUJ1Y2tldFwiLCB7XG4gICAgICAgIHZlcnNpb25lZDogdHJ1ZSxcbiAgICAgICAgZW5mb3JjZVNTTDogdHJ1ZSxcbiAgICAgICAgYXV0b0RlbGV0ZU9iamVjdHM6IHRydWUsXG4gICAgICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICAgICAgZW5jcnlwdGlvbjpcbiAgICAgICAgICBwcm9wcy5kZWZhdWx0V2Vic2l0ZUJ1Y2tldEVuY3J5cHRpb24gPz8gQnVja2V0RW5jcnlwdGlvbi5TM19NQU5BR0VELFxuICAgICAgICBvYmplY3RPd25lcnNoaXA6IE9iamVjdE93bmVyc2hpcC5CVUNLRVRfT1dORVJfRU5GT1JDRUQsXG4gICAgICAgIGVuY3J5cHRpb25LZXk6IHByb3BzLmRlZmF1bHRXZWJzaXRlQnVja2V0RW5jcnlwdGlvbktleSxcbiAgICAgICAgcHVibGljUmVhZEFjY2VzczogZmFsc2UsXG4gICAgICAgIGJsb2NrUHVibGljQWNjZXNzOiBCbG9ja1B1YmxpY0FjY2Vzcy5CTE9DS19BTEwsXG4gICAgICAgIHNlcnZlckFjY2Vzc0xvZ3NQcmVmaXg6IFwid2Vic2l0ZS1hY2Nlc3MtbG9nc1wiLFxuICAgICAgICBzZXJ2ZXJBY2Nlc3NMb2dzQnVja2V0OiBhY2Nlc3NMb2dzQnVja2V0LFxuICAgICAgfSk7XG5cbiAgICAvLyBXZWIgQUNMXG4gICAgY29uc3QgeyBkaXN0cmlidXRpb25Qcm9wcyB9ID0gcHJvcHM7XG4gICAgY29uc3Qgd2ViQWNsQXJuID1cbiAgICAgIGRpc3RyaWJ1dGlvblByb3BzPy53ZWJBY2xJZCA/PyBwcm9wcy53ZWJBY2xQcm9wcz8uZGlzYWJsZVxuICAgICAgICA/IHVuZGVmaW5lZFxuICAgICAgICA6IG5ldyBDbG91ZGZyb250V2ViQWNsKHRoaXMsIFwiV2Vic2l0ZUFjbFwiLCBwcm9wcy53ZWJBY2xQcm9wcykud2ViQWNsQXJuO1xuXG4gICAgLy8gQ2xvdWRmcm9udCBEaXN0cmlidXRpb25cbiAgICBjb25zdCBsb2dCdWNrZXQgPVxuICAgICAgcHJvcHMuZGlzdHJpYnV0aW9uUHJvcHM/LmxvZ0J1Y2tldCB8fFxuICAgICAgbmV3IEJ1Y2tldCh0aGlzLCBcIkRpc3RyaWJ1dGlvbkxvZ0J1Y2tldFwiLCB7XG4gICAgICAgIGVuZm9yY2VTU0w6IHRydWUsXG4gICAgICAgIGF1dG9EZWxldGVPYmplY3RzOiB0cnVlLFxuICAgICAgICByZW1vdmFsUG9saWN5OiBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgICAgIGVuY3J5cHRpb246XG4gICAgICAgICAgcHJvcHMuZGVmYXVsdFdlYnNpdGVCdWNrZXRFbmNyeXB0aW9uID8/IEJ1Y2tldEVuY3J5cHRpb24uUzNfTUFOQUdFRCxcbiAgICAgICAgb2JqZWN0T3duZXJzaGlwOiBPYmplY3RPd25lcnNoaXAuQlVDS0VUX09XTkVSX1BSRUZFUlJFRCxcbiAgICAgICAgZW5jcnlwdGlvbktleTogcHJvcHMuZGVmYXVsdFdlYnNpdGVCdWNrZXRFbmNyeXB0aW9uS2V5LFxuICAgICAgICBwdWJsaWNSZWFkQWNjZXNzOiBmYWxzZSxcbiAgICAgICAgYmxvY2tQdWJsaWNBY2Nlc3M6IEJsb2NrUHVibGljQWNjZXNzLkJMT0NLX0FMTCxcbiAgICAgICAgc2VydmVyQWNjZXNzTG9nc1ByZWZpeDogXCJkaXN0cmlidXRpb24tYWNjZXNzLWxvZ3NcIixcbiAgICAgICAgc2VydmVyQWNjZXNzTG9nc0J1Y2tldDogYWNjZXNzTG9nc0J1Y2tldCxcbiAgICAgIH0pO1xuXG4gICAgY29uc3Qgb3JpZ2luQWNjZXNzSWRlbnRpdHkgPSBuZXcgT3JpZ2luQWNjZXNzSWRlbnRpdHkoXG4gICAgICB0aGlzLFxuICAgICAgXCJPcmlnaW5BY2Nlc3NJZGVudGl0eVwiXG4gICAgKTtcbiAgICB0aGlzLndlYnNpdGVCdWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShcbiAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICByZXNvdXJjZXM6IFt0aGlzLndlYnNpdGVCdWNrZXQuYnVja2V0QXJuXSxcbiAgICAgICAgYWN0aW9uczogW1wiczM6TGlzdEJ1Y2tldFwiXSxcbiAgICAgICAgcHJpbmNpcGFsczogW29yaWdpbkFjY2Vzc0lkZW50aXR5LmdyYW50UHJpbmNpcGFsXSxcbiAgICAgIH0pXG4gICAgKTtcblxuICAgIGNvbnN0IGRlZmF1bHRSb290T2JqZWN0ID1cbiAgICAgIGRpc3RyaWJ1dGlvblByb3BzPy5kZWZhdWx0Um9vdE9iamVjdCA/PyBcImluZGV4Lmh0bWxcIjtcbiAgICB0aGlzLmNsb3VkRnJvbnREaXN0cmlidXRpb24gPSBuZXcgRGlzdHJpYnV0aW9uKFxuICAgICAgdGhpcyxcbiAgICAgIFwiQ2xvdWRmcm9udERpc3RyaWJ1dGlvblwiLFxuICAgICAge1xuICAgICAgICAuLi5kaXN0cmlidXRpb25Qcm9wcyxcbiAgICAgICAgd2ViQWNsSWQ6IHdlYkFjbEFybixcbiAgICAgICAgZW5hYmxlTG9nZ2luZzogdHJ1ZSxcbiAgICAgICAgbG9nQnVja2V0OiBsb2dCdWNrZXQsXG4gICAgICAgIGRlZmF1bHRCZWhhdmlvcjoge1xuICAgICAgICAgIC4uLmRpc3RyaWJ1dGlvblByb3BzPy5kZWZhdWx0QmVoYXZpb3IsXG4gICAgICAgICAgb3JpZ2luOiBuZXcgUzNPcmlnaW4odGhpcy53ZWJzaXRlQnVja2V0LCB7XG4gICAgICAgICAgICBvcmlnaW5BY2Nlc3NJZGVudGl0eSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgICB2aWV3ZXJQcm90b2NvbFBvbGljeTogVmlld2VyUHJvdG9jb2xQb2xpY3kuUkVESVJFQ1RfVE9fSFRUUFMsXG4gICAgICAgIH0sXG4gICAgICAgIGRlZmF1bHRSb290T2JqZWN0LFxuICAgICAgICBlcnJvclJlc3BvbnNlczogZGlzdHJpYnV0aW9uUHJvcHM/LmVycm9yUmVzcG9uc2VzID8/IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBodHRwU3RhdHVzOiA0MDQsIC8vIFdlIG5lZWQgdG8gcmVkaXJlY3QgXCJrZXkgbm90IGZvdW5kIGVycm9yc1wiIHRvIGluZGV4Lmh0bWwgZm9yIHNpbmdsZSBwYWdlIGFwcHNcbiAgICAgICAgICAgIHJlc3BvbnNlSHR0cFN0YXR1czogMjAwLFxuICAgICAgICAgICAgcmVzcG9uc2VQYWdlUGF0aDogYC8ke2RlZmF1bHRSb290T2JqZWN0fWAsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH1cbiAgICApO1xuXG4gICAgLy8gRGVwbG95IFdlYnNpdGVcbiAgICB0aGlzLmJ1Y2tldERlcGxveW1lbnQgPSBuZXcgQnVja2V0RGVwbG95bWVudCh0aGlzLCBcIldlYnNpdGVEZXBsb3ltZW50XCIsIHtcbiAgICAgIC4uLnByb3BzLmJ1Y2tldERlcGxveW1lbnRQcm9wcyxcbiAgICAgIHNvdXJjZXM6IFtcbiAgICAgICAgU291cmNlLmFzc2V0KHByb3BzLndlYnNpdGVDb250ZW50UGF0aCksXG4gICAgICAgIC4uLihwcm9wcy5ydW50aW1lT3B0aW9uc1xuICAgICAgICAgID8gW1xuICAgICAgICAgICAgICBTb3VyY2UuanNvbkRhdGEoXG4gICAgICAgICAgICAgICAgcHJvcHMucnVudGltZU9wdGlvbnM/Lmpzb25GaWxlTmFtZSB8fFxuICAgICAgICAgICAgICAgICAgREVGQVVMVF9SVU5USU1FX0NPTkZJR19GSUxFTkFNRSxcbiAgICAgICAgICAgICAgICBwcm9wcy5ydW50aW1lT3B0aW9ucz8uanNvblBheWxvYWRcbiAgICAgICAgICAgICAgKSxcbiAgICAgICAgICAgIF1cbiAgICAgICAgICA6IFtdKSxcbiAgICAgIF0sXG4gICAgICBkZXN0aW5hdGlvbkJ1Y2tldDogdGhpcy53ZWJzaXRlQnVja2V0LFxuICAgICAgLy8gRmlsZXMgaW4gdGhlIGRpc3RyaWJ1dGlvbidzIGVkZ2UgY2FjaGVzIHdpbGwgYmUgaW52YWxpZGF0ZWQgYWZ0ZXIgZmlsZXMgYXJlIHVwbG9hZGVkIHRvIHRoZSBkZXN0aW5hdGlvbiBidWNrZXQuXG4gICAgICBkaXN0cmlidXRpb246IHRoaXMuY2xvdWRGcm9udERpc3RyaWJ1dGlvbixcbiAgICB9KTtcblxuICAgIG5ldyBDZm5PdXRwdXQodGhpcywgXCJEaXN0cmlidXRpb25Eb21haW5OYW1lXCIsIHtcbiAgICAgIHZhbHVlOiB0aGlzLmNsb3VkRnJvbnREaXN0cmlidXRpb24uZG9tYWluTmFtZSxcbiAgICB9KTtcblxuICAgIHRoaXMuc3VwcHJlc3NDREtOYWdWaW9sYXRpb25zKHByb3BzKTtcbiAgfVxuXG4gIHByaXZhdGUgdmFsaWRhdGVQcm9wcyA9IChwcm9wczogU3RhdGljV2Vic2l0ZVByb3BzKSA9PiB7XG4gICAgdGhpcy52YWxpZGF0ZUVuY3J5cHRpb25TZXR0aW5ncyhwcm9wcyk7XG4gICAgcHJvcHMucnVudGltZU9wdGlvbnMgJiYgdGhpcy52YWxpZGF0ZVJ1bnRpbWVDb25maWcocHJvcHMucnVudGltZU9wdGlvbnMpO1xuICAgIHByb3BzLndlYnNpdGVCdWNrZXQgJiYgdGhpcy52YWxpZGF0ZUJ1Y2tldENvbmZpZyhwcm9wcy53ZWJzaXRlQnVja2V0KTtcbiAgfTtcblxuICBwcml2YXRlIHZhbGlkYXRlUnVudGltZUNvbmZpZyA9IChjb25maWc6IFJ1bnRpbWVPcHRpb25zKSA9PiB7XG4gICAgaWYgKCFjb25maWcpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgXCJ2YWxpZGF0ZVJ1bnRpbWVDb25maWcgb25seSBhY2NlcHRzIG5vbi1udWxsIFJ1bnRpbWVPcHRpb25zLlwiXG4gICAgICApO1xuICAgIH1cblxuICAgIGlmIChjb25maWcuanNvbkZpbGVOYW1lICYmICFjb25maWcuanNvbkZpbGVOYW1lLmVuZHNXaXRoKFwiLmpzb25cIikpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcIlJ1bnRpbWVPcHRpb25zLmpzb25GaWxlTmFtZSBtdXN0IGJlIGEganNvbiBmaWxlLlwiKTtcbiAgICB9XG4gIH07XG5cbiAgcHJpdmF0ZSB2YWxpZGF0ZUJ1Y2tldENvbmZpZyA9IChidWNrZXQ6IElCdWNrZXQpID0+IHtcbiAgICBpZiAoYnVja2V0LmlzV2Vic2l0ZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBcIldlYnNpdGUgYnVja2V0cyBjYW5ub3QgYmUgY29uZmlndXJlZCBhcyB3ZWJzaXRlcyBhcyB0aGlzIHdpbGwgYnJlYWsgQ2xvdWRmcm9udCBob3N0aW5nIVwiXG4gICAgICApO1xuICAgIH1cbiAgfTtcblxuICBwcml2YXRlIHZhbGlkYXRlRW5jcnlwdGlvblNldHRpbmdzID0gKHtcbiAgICBkZWZhdWx0V2Vic2l0ZUJ1Y2tldEVuY3J5cHRpb24sXG4gICAgZGVmYXVsdFdlYnNpdGVCdWNrZXRFbmNyeXB0aW9uS2V5LFxuICB9OiBTdGF0aWNXZWJzaXRlUHJvcHMpID0+IHtcbiAgICBpZiAoXG4gICAgICBkZWZhdWx0V2Vic2l0ZUJ1Y2tldEVuY3J5cHRpb25LZXkgJiZcbiAgICAgIGRlZmF1bHRXZWJzaXRlQnVja2V0RW5jcnlwdGlvbiAhPT0gQnVja2V0RW5jcnlwdGlvbi5LTVNcbiAgICApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgXCJCdWNrZXQgZW5jcnlwdGlvbiBzaG91bGQgYmUgc2V0IHRvIEtNUyBpZiBwcm92aWRpbmcgYSBkZWZhdWx0V2Vic2l0ZUJ1Y2tldEVuY3J5cHRpb25LZXkuXCJcbiAgICAgICk7XG4gICAgfVxuXG4gICAgaWYgKFxuICAgICAgZGVmYXVsdFdlYnNpdGVCdWNrZXRFbmNyeXB0aW9uICYmXG4gICAgICBkZWZhdWx0V2Vic2l0ZUJ1Y2tldEVuY3J5cHRpb24gIT09IEJ1Y2tldEVuY3J5cHRpb24uS01TICYmXG4gICAgICBkZWZhdWx0V2Vic2l0ZUJ1Y2tldEVuY3J5cHRpb24gIT09IEJ1Y2tldEVuY3J5cHRpb24uUzNfTUFOQUdFRFxuICAgICkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBcIk9ubHkgS01TIGFuZCBTM19NQU5BR0VEIGVuY3J5cHRpb24gYXJlIHN1cHBvcnRlZCBvbiB0aGUgZGVmYXVsdCBidWNrZXQuXCJcbiAgICAgICk7XG4gICAgfVxuICB9O1xuXG4gIHByaXZhdGUgc3VwcHJlc3NDREtOYWdWaW9sYXRpb25zID0gKHByb3BzOiBTdGF0aWNXZWJzaXRlUHJvcHMpID0+IHtcbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHRoaXMpO1xuICAgICFwcm9wcy5kaXN0cmlidXRpb25Qcm9wcz8uY2VydGlmaWNhdGUgJiZcbiAgICAgIFtcbiAgICAgICAgXCJBd3NTb2x1dGlvbnMtQ0ZSNFwiLFxuICAgICAgICBcIkF3c1Byb3RvdHlwaW5nLUNsb3VkRnJvbnREaXN0cmlidXRpb25IdHRwc1ZpZXdlck5vT3V0ZGF0ZWRTU0xcIixcbiAgICAgIF0uZm9yRWFjaCgoUnVsZUlkKSA9PiB7XG4gICAgICAgIE5hZ1N1cHByZXNzaW9ucy5hZGRSZXNvdXJjZVN1cHByZXNzaW9ucyh0aGlzLmNsb3VkRnJvbnREaXN0cmlidXRpb24sIFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBpZDogUnVsZUlkLFxuICAgICAgICAgICAgcmVhc29uOlxuICAgICAgICAgICAgICBcIkNlcnRpZmljYXRlIGlzIG5vdCBtYW5kYXRvcnkgdGhlcmVmb3JlIHRoZSBDbG91ZGZyb250IGNlcnRpZmljYXRlIHdpbGwgYmUgdXNlZC5cIixcbiAgICAgICAgICB9LFxuICAgICAgICBdKTtcbiAgICAgIH0pO1xuXG4gICAgW1wiQXdzU29sdXRpb25zLUwxXCIsIFwiQXdzUHJvdG90eXBpbmctTGFtYmRhTGF0ZXN0VmVyc2lvblwiXS5mb3JFYWNoKFxuICAgICAgKFJ1bGVJZCkgPT4ge1xuICAgICAgICBOYWdTdXBwcmVzc2lvbnMuYWRkUmVzb3VyY2VTdXBwcmVzc2lvbnMoXG4gICAgICAgICAgdGhpcyxcbiAgICAgICAgICBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIGlkOiBSdWxlSWQsXG4gICAgICAgICAgICAgIHJlYXNvbjpcbiAgICAgICAgICAgICAgICBcIkxhdGVzdCBydW50aW1lIGNhbm5vdCBiZSBjb25maWd1cmVkLiBDREsgd2lsbCBuZWVkIHRvIHVwZ3JhZGUgdGhlIEJ1Y2tldERlcGxveW1lbnQgY29uc3RydWN0IGFjY29yZGluZ2x5LlwiLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIHRydWVcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICApO1xuXG4gICAgW1wiQXdzU29sdXRpb25zLUlBTTVcIiwgXCJBd3NQcm90b3R5cGluZy1JQU1Ob1dpbGRjYXJkUGVybWlzc2lvbnNcIl0uZm9yRWFjaChcbiAgICAgIChSdWxlSWQpID0+IHtcbiAgICAgICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKFxuICAgICAgICAgIHRoaXMsXG4gICAgICAgICAgW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBpZDogUnVsZUlkLFxuICAgICAgICAgICAgICByZWFzb246XG4gICAgICAgICAgICAgICAgXCJBbGwgUG9saWNpZXMgaGF2ZSBiZWVuIHNjb3BlZCB0byBhIEJ1Y2tldC4gR2l2ZW4gQnVja2V0cyBjYW4gY29udGFpbiBhcmJpdHJhcnkgY29udGVudCwgd2lsZGNhcmQgcmVzb3VyY2VzIHdpdGggYnVja2V0IHNjb3BlIGFyZSByZXF1aXJlZC5cIixcbiAgICAgICAgICAgICAgYXBwbGllc1RvOiBbXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgcmVnZXg6IFwiL15BY3Rpb246OnMzOi4qJC9nXCIsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICByZWdleDogYC9eUmVzb3VyY2U6Oi4qJC9nYCxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIHRydWVcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICApO1xuXG4gICAgW1wiQXdzU29sdXRpb25zLUlBTTRcIiwgXCJBd3NQcm90b3R5cGluZy1JQU1Ob01hbmFnZWRQb2xpY2llc1wiXS5mb3JFYWNoKFxuICAgICAgKFJ1bGVJZCkgPT4ge1xuICAgICAgICBOYWdTdXBwcmVzc2lvbnMuYWRkUmVzb3VyY2VTdXBwcmVzc2lvbnMoXG4gICAgICAgICAgdGhpcyxcbiAgICAgICAgICBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIGlkOiBSdWxlSWQsXG4gICAgICAgICAgICAgIHJlYXNvbjpcbiAgICAgICAgICAgICAgICBcIkJ1Y2tldHMgY2FuIGNvbnRhaW4gYXJiaXRyYXJ5IGNvbnRlbnQsIHRoZXJlZm9yZSB3aWxkY2FyZCByZXNvdXJjZXMgdW5kZXIgYSBidWNrZXQgYXJlIHJlcXVpcmVkLlwiLFxuICAgICAgICAgICAgICBhcHBsaWVzVG86IFtcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICByZWdleDogYC9eUG9saWN5Ojphcm46JHtQREtOYWcuZ2V0U3RhY2tQYXJ0aXRpb25SZWdleChcbiAgICAgICAgICAgICAgICAgICAgc3RhY2tcbiAgICAgICAgICAgICAgICAgICl9OmlhbTo6YXdzOnBvbGljeS9zZXJ2aWNlLXJvbGUvQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlJC9nYCxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIHRydWVcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICApO1xuXG4gICAgW1wiQXdzU29sdXRpb25zLVMxXCIsIFwiQXdzUHJvdG90eXBpbmctUzNCdWNrZXRMb2dnaW5nRW5hYmxlZFwiXS5mb3JFYWNoKFxuICAgICAgKFJ1bGVJZCkgPT4ge1xuICAgICAgICBOYWdTdXBwcmVzc2lvbnMuYWRkUmVzb3VyY2VTdXBwcmVzc2lvbnMoXG4gICAgICAgICAgdGhpcyxcbiAgICAgICAgICBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIGlkOiBSdWxlSWQsXG4gICAgICAgICAgICAgIHJlYXNvbjogXCJBY2Nlc3MgTG9nIGJ1Y2tldHMgc2hvdWxkIG5vdCBoYXZlIHMzIGJ1Y2tldCBsb2dnaW5nXCIsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgICAgdHJ1ZVxuICAgICAgICApO1xuICAgICAgfVxuICAgICk7XG4gIH07XG59XG5cbi8qKlxuICogSWYgcGFzc2luZyBpbiBkaXN0cmlidXRpb25Qcm9wcywgdGhlIGRlZmF1bHQgYmVoYXZpb3VyLm9yaWdpbiBpcyBhIHJlcXVpcmVkIHBhcmFtZXRlci4gQW4gaW5zdGFuY2Ugb2YgdGhpcyBjbGFzcyBjYW4gYmUgcGFzc2VkIGluXG4gKiB0byBtYWtlIHRoZSBjb21waWxlciBoYXBweS5cbiAqL1xuZXhwb3J0IGNsYXNzIFN0YXRpY1dlYnNpdGVPcmlnaW4gaW1wbGVtZW50cyBJT3JpZ2luIHtcbiAgYmluZChfc2NvcGU6IENvbnN0cnVjdCwgX29wdGlvbnM6IE9yaWdpbkJpbmRPcHRpb25zKTogT3JpZ2luQmluZENvbmZpZyB7XG4gICAgdGhyb3cgbmV3IEVycm9yKFwiVGhpcyBzaG91bGQgbmV2ZXIgYmUgY2FsbGVkXCIpO1xuICB9XG59XG4iXX0=