"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudfrontWebAcl = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
/**
 * This construct creates a WAFv2 Web ACL for cloudfront in the us-east-1 region (required for cloudfront) no matter the
 * region of the parent cdk stack.
 */
class CloudfrontWebAcl extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const stack = aws_cdk_lib_1.Stack.of(this);
        const aclName = `${stack.stackName}-${id}`; // Unique per stack
        const onEventHandler = this.createOnEventHandler(stack, aclName);
        const customResource = this.createAclCustomResource(stack, aclName, onEventHandler, props);
        this.webAclId = customResource.getAttString("WebAclId");
        this.webAclArn = customResource.getAttString("WebAclArn");
    }
    /**
     * Creates an event handler for managing an ACL in us-east-1.
     *
     * @param stack containing Stack instance.
     * @param aclName name of the ACL to manage.
     * @private
     */
    createOnEventHandler(stack, aclName) {
        // NB without manually defining a name, the cdk generated name for the Provider function can become too long and
        // deployments fail. This is because the Provider's name references the onEvent handler name and appends "-Provider"
        // rather than being generated by cdk and truncated appropriately
        const onEventHandlerName = `${pdk_nag_1.PDKNag.getStackPrefix(stack)
            .split("/")
            .join("-")}AclEvent-${this.node.addr.slice(-6)}`;
        const onEventHandlerRole = new aws_iam_1.Role(this, "OnEventHandlerRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${onEventHandlerName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${onEventHandlerName}:*`,
                            ],
                        }),
                    ],
                }),
                wafv2: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "wafv2:CreateWebACL",
                                "wafv2:DeleteWebACL",
                                "wafv2:UpdateWebACL",
                                "wafv2:GetWebACL",
                            ],
                            resources: [
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/ipset/${aclName}-IPSet/*`,
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/webacl/${aclName}/*`,
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/managedruleset/*/*`,
                            ],
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "wafv2:CreateIPSet",
                                "wafv2:DeleteIPSet",
                                "wafv2:UpdateIPSet",
                                "wafv2:GetIPSet",
                            ],
                            resources: [
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/ipset/${aclName}-IPSet/*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const onEventHandler = new aws_lambda_1.Function(this, "CloudfrontWebAclOnEventHandler", {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "../lib/webacl_event_handler")),
            role: onEventHandlerRole,
            functionName: onEventHandlerName,
            handler: "index.onEvent",
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            timeout: aws_cdk_lib_1.Duration.seconds(300),
        });
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(onEventHandlerRole, [
                {
                    id: RuleId,
                    reason: "WafV2 resources have been scoped down to the ACL/IPSet level, however * is still needed as resource id's are created just in time.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:aws:wafv2:us-east-1:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:global/(.*)$/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:aws:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/lambda/${onEventHandlerName}:\*/g`,
                        },
                    ],
                },
            ], true);
        });
        return onEventHandler;
    }
    /**
     * Creates a Custom resource to manage the deployment of the ACL.
     *
     * @param stack containing Stack instance.
     * @param aclName name of the ACL to manage.
     * @param onEventHandler event handler to use for deployment.
     * @param props user provided properties for configuring the ACL.
     * @private
     */
    createAclCustomResource(stack, aclName, onEventHandler, props) {
        const providerFunctionName = `${onEventHandler.functionName}-Provider`;
        const providerRole = new aws_iam_1.Role(this, "CloudfrontWebAclProviderRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}:*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const provider = new custom_resources_1.Provider(this, "CloudfrontAclProvider", {
            onEventHandler,
            role: providerRole,
            providerFunctionName,
        });
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
                {
                    id: RuleId,
                    reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                },
            ], true);
        });
        ["AwsSolutions-L1", "AwsPrototyping-LambdaLatestVersion"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(provider, [
                {
                    id: RuleId,
                    reason: "Latest runtime cannot be configured. CDK will need to upgrade the Provider construct accordingly.",
                },
            ], true);
        });
        return new aws_cdk_lib_1.CustomResource(this, "CFAclCustomResource", {
            serviceToken: provider.serviceToken,
            properties: {
                ID: aclName,
                MANAGED_RULES: props?.managedRules ?? [
                    { vendor: "AWS", name: "AWSManagedRulesCommonRuleSet" },
                ],
                CIDR_ALLOW_LIST: props?.cidrAllowList,
            },
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
CloudfrontWebAcl[_a] = { fqn: "@aws-prototyping-sdk/static-website.CloudfrontWebAcl", version: "0.19.64" };
exports.CloudfrontWebAcl = CloudfrontWebAcl;
//# sourceMappingURL=data:application/json;base64,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