# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['datastruct', 'datastruct.fields', 'datastruct.utils']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'py-datastruct',
    'version': '0.1.0',
    'description': 'Combination of struct and dataclasses for easy parsing of binary formats',
    'long_description': '# py-datastruct\n\nThis is a (relatively) simple, **pure-Python, no dependency** library, aiming to simplify parsing and building binary data structures. It uses **[`dataclasses`](https://docs.python.org/3/library/dataclasses.html)** as its main container type, and **[`struct`](https://docs.python.org/3/library/struct.html)-compatible format specifiers** for writing field definitions.\n\nThe way of composing structures is somewhat similar to (and inspired by) [Construct](https://github.com/construct/construct). While probably not as powerful, it should give more flexibility and control over the data, as well as **full IDE type hinting**.\n\n## Installation\n\n```shell\npip install py-datastruct\n```\n\n## Usage\n\nThis simple example illustrates creating a 24-byte long structure, consisting of a 32-bit integer, an 8-byte 0xFF-filled padding, and a 12-byte `bytes` string.\n\n```python\nfrom hexdump import hexdump\nfrom dataclasses import dataclass\nfrom datastruct import DataStruct\nfrom datastruct.fields import field, padding\n\n@dataclass\nclass MyStruct(DataStruct):\n    my_number: int = field("I", default=123)\n    _1: ... = padding(8)\n    my_binary: bytes = field("12s")\n\nmy_object = MyStruct(my_binary=b"Hello Python")\nprint(my_object)\n# MyStruct(my_number=123, my_binary=b\'Hello World!\')\n\nmy_object = MyStruct(my_number=5, my_binary=b"Hello World!")\nprint(my_object)\n# MyStruct(my_number=5, my_binary=b\'Hello World!\')\n\npacked = my_object.pack()\nhexdump(packed)\n# 00000000: 05 00 00 00 FF FF FF FF  FF FF FF FF 48 65 6C 6C  ............Hell\n# 00000010: 6F 20 57 6F 72 6C 64 21                           o World!\n\nunpacked = MyStruct.unpack(packed)\nprint(unpacked)\n# MyStruct(my_number=5, my_binary=b\'Hello World!\')\nprint(my_object == unpacked)\n# True\n```\n\nYou might also pass a stream (file/BytesIO/etc.) to `pack()` and `unpack()`. Otherwise, `pack()` will create a BytesIO stream and return its contents after packing; `unpack()` will accept a `bytes` object as its parameter.\n\n`pack()` and `unpack()` also accept custom, keyword-only arguments, that are available in the Context, throughout the entire operation.\n\n### Context\n\nUpon starting a pack/unpack operation, a `Context` object is created. The context is a container scoped to the currently processed structure. It\'s composed of four main elements:\n\n- all values of the current structure - when packing; during unpacking, it contains all values of fields that were already processes (the context "grows")\n- `_: Context` - reference to the parent object\'s context (only when nesting `DataStruct`s)\n- `G` - global context - general-purpose container that is not scoped to the current structure (it\'s identical for nested structs)\n  - `io: IO[bytes]` - the stream being read from/written to\n  - `packing: bool` - whether current operation is packing\n  - `unpacking: bool` - whether current operation is unpacking\n  - `env: Container` - keyword arguments passed to `pack()`/`unpack()`\n  - `root: Context` - context of the topmost structure\n  - `tell: () -> int` - function returning the current position in the stream\n  - `seek: (offset: int, whence: int) -> int` - function allowing to seek to an absolute offset\n- `P` - local context - general-purpose container that is different for each nested struct\n  - `tell: () -> int` - function returning the current position in the current structure (in bytes)\n  - `seek: (offset: int, whence: int) -> int` - function allowing to seek to an offset within the current structure\n  - `skip: (length: int) -> int` - function allowing to skip `length` bytes\n  - `i: int` - (for `repeat()` fields only) index of the current item of the list\n  - `item: Any` - (for `repeat()` fields, in `last=` lambda only) item processed right before evaluation\n\nThe context is "general-purpose", meaning that the user can write custom values to it. All fields presented above can be accessed by lambda functions - see "Parameter evaluation".\n\n### Parameter evaluation\n\nMost field parameters support pack/unpack-time evaluation (which means they can e.g. depend on previously read fields). Lambda expressions are then given the current context, and expected to return a simple value, that would be statically valid in this parameter.\n\n```python\nan_unpredictable_field: int = field(lambda ctx: "I" if randint(1, 10) % 2 == 0 else "H")\n```\n\n### Ellipsis - special value\n\nA special value of type `Ellipsis`/`...` is used in the library, to indicate something not having a type or a value. **It\'s not the same as `None`**. `built()` fields, for example, have `...` as value after creating the struct, but before packing it for the first time.\n\nSpecial fields (like `padding()`, which don\'t have any value) must have `...` as their type hint.\n\n### Variable-length fields\n\nThis is a simple example of using parameter evaluation to dynamically size a `bytes` string. Binary strings use the `<len>s` specifier, which can be omitted (simple `int` can be used instead). \n\n```python\n@dataclass\nclass MyStruct(DataStruct):\n    data_length: int = field("I")\n    data: bytes = field(lambda ctx: ctx.data_length)\n```\n\nThe user is still responsible for adjusting `data_length` after changing `data`. The `built()` field comes in handy here:\n\n```python\n@dataclass\nclass MyStruct(DataStruct):\n    data_length: int = built("I", lambda ctx: len(ctx.data))\n    data: bytes = field(lambda ctx: ctx.data_length)\n```\n\nWhen unpacking, the `data_length` field will be used to dynamically size the `data` field. When packing, `data_length` will always be recalculated based on what\'s in `data`.\n\n### Wrapper fields - storing a list\n\nLists are also iterables, like `bytes`, but they store a number of items of a specific type. Thus, the `repeat()` field **wrapper** has to be used. \n\n**Wrapper fields** simply require calling them first with any used parameters, then with the "base" field.\n\n```python\n@dataclass\nclass MyStruct(DataStruct):\n    item_count: int = built("H", lambda ctx: len(ctx.items))\n    # This creates a list of 16-bit integers.\n    # The list is empty by default.\n    items: List[int] = repeat(lambda ctx: ctx.item_count)(field("H"))\n\nmy_object = MyStruct()\nmy_object.items = [0x5555, 0x4444, 0x3333, 0x2222]\nmy_object.item_count = 1  # this doesn\'t matter, as the field is rebuilt\npacked = my_object.pack()\nhexdump(packed)\n# 00000000: 04 00 55 55 44 44 33 33  22 22\n```\n\n### Conditional fields\n\nThey\'re also wrapper fields - if the condition is not met, they act like as if the field didn\'t exist at all.\n\n```python\n@dataclass\nclass MyStruct(DataStruct):\n    has_text: bool = field("?")\n    text: str = cond(lambda ctx: ctx.has_text)(field("8s", default=""))\n\nmy_object = MyStruct.unpack(b"\\x01HELOWRLD")\nprint(my_object)\n# MyStruct(has_text=True, text=\'HELOWRLD\')\n\nmy_object = MyStruct.unpack(b"\\x00")\nprint(my_object)\n# MyStruct(has_text=False, text=\'\')\n```\n\n### Switch fields\n\nSwitch fields are like more powerful conditional fields. The following example reads an 8/16/32-bit number, depending on the prefixing length byte. If the length is not supported, it reads the value as `bytes` instead.\n\n```python\nnumber_length: int = field("B", default=1)\nnumber: Union[int, bytes] = switch(lambda ctx: ctx.number_length)(\n    _1=(int, field("B")),\n    _2=(int, field("H")),\n    _4=(int, field("I")),\n    default=(bytes, field(lambda ctx: ctx.number_length)),\n)\n```\n\nThe values on the left (`_1`, `_2`, `_4`) are the **keys**. The key is picked depending on the key-lambda result (`ctx.number_length`). The value on the right is a tuple of the expected field type, and a `field()` specifier.\n\nSince it\'s not possible to pass just `1` as a keyword argument, integers are looked up prefixed with an underscore as well. Enums are additionally looked up by their name and value, and booleans are looked up by **lowercase** `true`/`false`.\n\nNote that you can pass (probably) any kind of field to the switch list.\n\n## To be continued\n\n## License\n\n```\nMIT License\n\nCopyright (c) 2023 Kuba Szczodrzyński\n\nPermission is hereby granted, free of charge, to any person obtaining a copy\nof this software and associated documentation files (the "Software"), to deal\nin the Software without restriction, including without limitation the rights\nto use, copy, modify, merge, publish, distribute, sublicense, and/or sell\ncopies of the Software, and to permit persons to whom the Software is\nfurnished to do so, subject to the following conditions:\n\nThe above copyright notice and this permission notice shall be included in all\ncopies or substantial portions of the Software.\n\nTHE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR\nIMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,\nFITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE\nAUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER\nLIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,\nOUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE\nSOFTWARE.\n```\n',
    'author': 'Kuba Szczodrzyński',
    'author_email': 'kuba@szczodrzynski.pl',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
