"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PersistentVolumeMode = exports.PersistentVolumeAccessMode = exports.PersistentVolumeClaim = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
/**
 * A PersistentVolumeClaim (PVC) is a request for storage by a user.
 * It is similar to a Pod. Pods consume node resources and PVCs consume PV resources.
 * Pods can request specific levels of resources (CPU and Memory).
 * Claims can request specific size and access modes
 */
class PersistentVolumeClaim extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _b;
        super(scope, id);
        this.storage = props.storage;
        this.volumeMode = (_b = props.volumeMode) !== null && _b !== void 0 ? _b : PersistentVolumeMode.FILE_SYSTEM;
        this.storageClassName = props.storageClassName;
        this._accessModes = props.accessModes;
        if (props.volume) {
            this.bind(props.volume);
        }
        this.apiObject = new k8s.KubePersistentVolumeClaim(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s.Lazy.any({ produce: () => this._toKube() }),
        });
    }
    /**
     * Imports a pvc from the cluster as a reference.
     * @param claimName The name of the pvc to reference.
     */
    static fromClaimName(claimName) {
        return { name: claimName };
    }
    /**
     * Access modes requirement of this claim.
     */
    get accessModes() {
        return this._accessModes ? [...this._accessModes] : undefined;
    }
    /**
     * PV this claim is bound to. Undefined means the claim is not bound
     * to any specific volume.
     */
    get volume() {
        return this._volume;
    }
    /**
     * Bind a claim to a specific volume.
     * Note that you must also bind the volume to the claim.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#binding
     *
     * @param pv The PV to bind to.
     */
    bind(pv) {
        if (this._volume && this._volume.name !== pv.name) {
            throw new Error(`Cannot bind claim '${this.name}' to volume '${pv.name}' since it is already bound to volume '${this._volume.name}'`);
        }
        this._volume = pv;
    }
    /**
     * @internal
     */
    _toKube() {
        var _b;
        const storage = this.storage ? k8s.Quantity.fromString(this.storage.toGibibytes() + 'Gi') : undefined;
        return {
            volumeName: this.volume ? this.volume.name : undefined,
            accessModes: (_b = this.accessModes) === null || _b === void 0 ? void 0 : _b.map(a => a.toString()),
            resources: storage ? { requests: { storage } } : undefined,
            volumeMode: this.volumeMode,
            storageClassName: this.storageClassName,
        };
    }
}
exports.PersistentVolumeClaim = PersistentVolumeClaim;
_a = JSII_RTTI_SYMBOL_1;
PersistentVolumeClaim[_a] = { fqn: "cdk8s-plus-22.PersistentVolumeClaim", version: "1.0.0-beta.201" };
/**
 * Access Modes.
 */
var PersistentVolumeAccessMode;
(function (PersistentVolumeAccessMode) {
    /**
     * The volume can be mounted as read-write by a single node.
     * ReadWriteOnce access mode still can allow multiple pods to access
     * the volume when the pods are running on the same node.
     */
    PersistentVolumeAccessMode["READ_WRITE_ONCE"] = "ReadWriteOnce";
    /**
     * The volume can be mounted as read-only by many nodes.
     */
    PersistentVolumeAccessMode["READ_ONLY_MANY"] = "ReadOnlyMany";
    /**
     * The volume can be mounted as read-write by many nodes.
     */
    PersistentVolumeAccessMode["READ_WRITE_MANY"] = "ReadWriteMany";
    /**
     * The volume can be mounted as read-write by a single Pod.
     * Use ReadWriteOncePod access mode if you want to ensure that
     * only one pod across whole cluster can read that PVC or write to it.
     * This is only supported for CSI volumes and Kubernetes version 1.22+.
     */
    PersistentVolumeAccessMode["READ_WRITE_ONCE_POD"] = "ReadWriteOncePod";
})(PersistentVolumeAccessMode = exports.PersistentVolumeAccessMode || (exports.PersistentVolumeAccessMode = {}));
/**
 * Volume Modes.
 */
var PersistentVolumeMode;
(function (PersistentVolumeMode) {
    /**
     * Volume is ounted into Pods into a directory.
     * If the volume is backed by a block device and the device is empty,
     * Kubernetes creates a filesystem on the device before mounting it
     * for the first time.
     */
    PersistentVolumeMode["FILE_SYSTEM"] = "Filesystem";
    /**
     * Use a volume as a raw block device. Such volume is presented into a Pod as a block device,
     * without any filesystem on it. This mode is useful to provide a Pod the fastest possible way
     * to access a volume, without any filesystem layer between the Pod
     * and the volume. On the other hand, the application running in
     * the Pod must know how to handle a raw block device
     */
    PersistentVolumeMode["BLOCK"] = "Block";
})(PersistentVolumeMode = exports.PersistentVolumeMode || (exports.PersistentVolumeMode = {}));
//# sourceMappingURL=data:application/json;base64,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