"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GCEPersistentDiskPersistentVolume = exports.AzureDiskPersistentVolume = exports.AwsElasticBlockStorePersistentVolume = exports.PersistentVolumeReclaimPolicy = exports.PersistentVolume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const pvc_1 = require("./pvc");
const volume_1 = require("./volume");
/**
 * A PersistentVolume (PV) is a piece of storage in the cluster that has been
 * provisioned by an administrator or dynamically provisioned using Storage Classes.
 * It is a resource in the cluster just like a node is a cluster resource.
 * PVs are volume plugins like Volumes, but have a lifecycle independent of any
 * individual Pod that uses the PV. This API object captures the details of the
 * implementation of the storage, be that NFS, iSCSI, or a
 * cloud-provider-specific storage system.
 */
class PersistentVolume extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _e, _f;
        super(scope, id);
        this.storage = props.storage;
        this.mode = (_e = props.volumeMode) !== null && _e !== void 0 ? _e : pvc_1.PersistentVolumeMode.FILE_SYSTEM;
        this.storageClassName = props.storageClassName;
        this._accessModes = props.accessModes;
        this.mountOptions = props.mountOptions;
        this.reclaimPolicy = (_f = props.reclaimPolicy) !== null && _f !== void 0 ? _f : PersistentVolumeReclaimPolicy.RETAIN;
        if (props.claim) {
            this.bind(props.claim);
        }
        this.apiObject = new k8s.KubePersistentVolume(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s.Lazy.any({ produce: () => this._toKube() }),
        });
    }
    /**
     * Imports a pv from the cluster as a reference.
     * @param volumeName The name of the pv to reference.
     */
    static fromPersistentVolumeName(volumeName) {
        return { name: volumeName };
    }
    /**
     * Access modes requirement of this claim.
     */
    get accessModes() {
        return this._accessModes ? [...this._accessModes] : undefined;
    }
    /**
     * PVC this volume is bound to. Undefined means this volume is not yet
     * claimed by any PVC.
     */
    get claim() {
        return this._claim;
    }
    /**
     * Reserve a `PersistentVolume` by creating a `PersistentVolumeClaim`
     * that is wired to claim this volume.
     *
     * Note that this method will throw in case the volume is already claimed.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#reserving-a-persistentvolume
     */
    reserve() {
        const claim = new pvc_1.PersistentVolumeClaim(this, `${this.name}PVC`, {
            metadata: { name: `pvc-${this.name}`, namespace: this.metadata.namespace },
            // the storage classes must match, otherwise the claim
            // will use the default class (or no class at all), which may be different than the class
            // of this volume. note that other requirements are not needed since
            // when they are not defined, any volume satisifies them.
            storageClassName: this.storageClassName,
        });
        this.bind(claim);
        claim.bind(this);
        return claim;
    }
    /**
     * Bind a volume to a specific claim.
     * Note that you must also bind the claim to the volume.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#binding
     *
     * @param pvc The PVC to bind to.
     */
    bind(pvc) {
        if (this._claim && this._claim.name !== pvc.name) {
            throw new Error(`Cannot bind volume '${this.name}' to claim '${pvc.name}' since it is already bound to claim '${this._claim.name}'`);
        }
        this._claim = pvc;
    }
    asVolume() {
        const claim = this.reserve();
        return volume_1.Volume.fromPersistentVolumeClaim(claim);
    }
    /**
     * @internal
     */
    _toKube() {
        var _e, _f, _g;
        const storage = this.storage ? k8s.Quantity.fromString(this.storage.toGibibytes() + 'Gi') : undefined;
        return {
            claimRef: this._claim ? { name: (_e = this._claim) === null || _e === void 0 ? void 0 : _e.name } : undefined,
            accessModes: (_f = this.accessModes) === null || _f === void 0 ? void 0 : _f.map(a => a.toString()),
            capacity: storage ? { storage } : undefined,
            mountOptions: (_g = this.mountOptions) === null || _g === void 0 ? void 0 : _g.map(o => o),
            storageClassName: this.storageClassName,
            persistentVolumeReclaimPolicy: this.reclaimPolicy,
            volumeMode: this.mode,
        };
    }
}
exports.PersistentVolume = PersistentVolume;
_a = JSII_RTTI_SYMBOL_1;
PersistentVolume[_a] = { fqn: "cdk8s-plus-22.PersistentVolume", version: "1.0.0-beta.201" };
/**
 * Reclaim Policies.
 */
var PersistentVolumeReclaimPolicy;
(function (PersistentVolumeReclaimPolicy) {
    /**
     * The Retain reclaim policy allows for manual reclamation of the resource.
     * When the PersistentVolumeClaim is deleted, the PersistentVolume still exists and the
     * volume is considered "released". But it is not yet available for another claim
     * because the previous claimant's data remains on the volume.
     * An administrator can manually reclaim the volume with the following steps:
     *
     * 1. Delete the PersistentVolume. The associated storage asset in external
     *    infrastructure (such as an AWS EBS, GCE PD, Azure Disk, or Cinder volume) still exists after the PV is deleted.
     * 2. Manually clean up the data on the associated storage asset accordingly.
     * 3. Manually delete the associated storage asset.
     *
     * If you want to reuse the same storage asset, create a new PersistentVolume
     * with the same storage asset definition.
     */
    PersistentVolumeReclaimPolicy["RETAIN"] = "Retain";
    /**
     * For volume plugins that support the Delete reclaim policy, deletion removes both the
     * PersistentVolume object from Kubernetes, as well as the associated storage asset in
     * the external infrastructure, such as an AWS EBS, GCE PD, Azure Disk, or Cinder volume.
     * Volumes that were dynamically provisioned inherit the reclaim policy of their StorageClass, which defaults to Delete.
     * The administrator should configure the StorageClass according to users' expectations; otherwise,
     * the PV must be edited or patched after it is created
     */
    PersistentVolumeReclaimPolicy["DELETE"] = "Delete";
})(PersistentVolumeReclaimPolicy = exports.PersistentVolumeReclaimPolicy || (exports.PersistentVolumeReclaimPolicy = {}));
/**
 * Represents an AWS Disk resource that is attached to a kubelet's host machine and
 * then exposed to the pod.
 *
 * @see https://kubernetes.io/docs/concepts/storage/volumes#awselasticblockstore
 */
class AwsElasticBlockStorePersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id, props);
        this.volumeId = props.volumeId;
        this.readOnly = (_e = props.readOnly) !== null && _e !== void 0 ? _e : false;
        this.fsType = (_f = props.fsType) !== null && _f !== void 0 ? _f : 'ext4';
        this.partition = props.partition;
    }
    /**
     * @internal
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            awsElasticBlockStore: {
                volumeId: this.volumeId,
                fsType: this.fsType,
                partition: this.partition,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.AwsElasticBlockStorePersistentVolume = AwsElasticBlockStorePersistentVolume;
_b = JSII_RTTI_SYMBOL_1;
AwsElasticBlockStorePersistentVolume[_b] = { fqn: "cdk8s-plus-22.AwsElasticBlockStorePersistentVolume", version: "1.0.0-beta.201" };
/**
 * AzureDisk represents an Azure Data Disk mount on the host and bind mount to the pod.
 */
class AzureDiskPersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        var _e, _f, _g, _h;
        super(scope, id, props);
        this.diskName = props.diskName;
        this.diskUri = props.diskUri;
        this.cachingMode = (_e = props.cachingMode) !== null && _e !== void 0 ? _e : volume_1.AzureDiskPersistentVolumeCachingMode.NONE;
        this.fsType = (_f = props.fsType) !== null && _f !== void 0 ? _f : 'ext4';
        this.kind = (_g = props.kind) !== null && _g !== void 0 ? _g : volume_1.AzureDiskPersistentVolumeKind.SHARED;
        this.readOnly = (_h = props.readOnly) !== null && _h !== void 0 ? _h : false;
    }
    /**
     * @internal
     *
     * @see https://github.com/kubernetes/examples/blob/master/staging/volumes/azure_disk/README.md
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            azureDisk: {
                diskName: this.diskName,
                diskUri: this.diskUri,
                cachingMode: this.cachingMode,
                fsType: this.fsType,
                kind: this.kind,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.AzureDiskPersistentVolume = AzureDiskPersistentVolume;
_c = JSII_RTTI_SYMBOL_1;
AzureDiskPersistentVolume[_c] = { fqn: "cdk8s-plus-22.AzureDiskPersistentVolume", version: "1.0.0-beta.201" };
/**
 * GCEPersistentDisk represents a GCE Disk resource that is attached to a kubelet's host machine
 * and then exposed to the pod. Provisioned by an admin.
 *
 * @see https://kubernetes.io/docs/concepts/storage/volumes#gcepersistentdisk
 */
class GCEPersistentDiskPersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id, props);
        this.pdName = props.pdName;
        this.readOnly = (_e = props.readOnly) !== null && _e !== void 0 ? _e : false;
        this.fsType = (_f = props.fsType) !== null && _f !== void 0 ? _f : 'ext4';
        this.partition = props.partition;
    }
    /**
     * @internal
     *
     * @see https://github.com/kubernetes/examples/blob/master/staging/volumes/azure_disk/README.md
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            gcePersistentDisk: {
                pdName: this.pdName,
                fsType: this.fsType,
                partition: this.partition,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.GCEPersistentDiskPersistentVolume = GCEPersistentDiskPersistentVolume;
_d = JSII_RTTI_SYMBOL_1;
GCEPersistentDiskPersistentVolume[_d] = { fqn: "cdk8s-plus-22.GCEPersistentDiskPersistentVolume", version: "1.0.0-beta.201" };
//# sourceMappingURL=data:application/json;base64,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