"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigMap = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cdk8s = require("cdk8s");
const minimatch_1 = require("minimatch");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const utils_1 = require("./utils");
/**
 * ConfigMap holds configuration data for pods to consume.
 */
class ConfigMap extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _b, _c, _d;
        super(scope, id);
        this._binaryData = {};
        this._data = {};
        this.immutable = (_b = props.immutable) !== null && _b !== void 0 ? _b : false;
        this.apiObject = new k8s.KubeConfigMap(this, 'Resource', {
            metadata: props.metadata,
            // we need lazy here because we filter empty
            data: cdk8s.Lazy.any({ produce: () => this.synthesizeData() }),
            binaryData: cdk8s.Lazy.any({ produce: () => this.synthesizeBinaryData() }),
            immutable: this.immutable,
        });
        for (const [k, v] of Object.entries((_c = props.data) !== null && _c !== void 0 ? _c : {})) {
            this.addData(k, v);
        }
        for (const [k, v] of Object.entries((_d = props.binaryData) !== null && _d !== void 0 ? _d : {})) {
            this.addBinaryData(k, v);
        }
    }
    /**
     * Represents a ConfigMap created elsewhere.
     * @param name The name of the config map to import
     */
    static fromConfigMapName(name) {
        return { name };
    }
    /**
     * Adds a data entry to the config map.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addData(key, value) {
        this.verifyKeyAvailable(key);
        this._data[key] = value;
    }
    /**
     * The data associated with this config map.
     *
     * Returns an copy. To add data records, use `addData()` or `addBinaryData()`.
     */
    get data() {
        return { ...this._data };
    }
    /**
     * Adds a binary data entry to the config map. BinaryData can contain byte
     * sequences that are not in the UTF-8 range.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addBinaryData(key, value) {
        this.verifyKeyAvailable(key);
        this._binaryData[key] = value;
    }
    /**
     * The binary data associated with this config map.
     *
     * Returns a copy. To add data records, use `addBinaryData()` or `addData()`.
     */
    get binaryData() {
        return { ...this._binaryData };
    }
    /**
     * Adds a file to the ConfigMap.
     * @param localFile The path to the local file
     * @param key The ConfigMap key (default to the file name).
     */
    addFile(localFile, key) {
        key = key !== null && key !== void 0 ? key : path.basename(localFile);
        const value = fs.readFileSync(localFile, 'utf-8');
        this.addData(key, value);
    }
    /**
     * Adds a directory to the ConfigMap.
     * @param localDir A path to a local directory
     * @param options Options
     */
    addDirectory(localDir, options = {}) {
        var _b, _c;
        const exclude = (_b = options.exclude) !== null && _b !== void 0 ? _b : [];
        const shouldInclude = (file) => {
            for (const pattern of exclude) {
                const mm = new minimatch_1.Minimatch(pattern);
                if (mm.match(file)) {
                    return false;
                }
            }
            return true;
        };
        const keyPrefix = (_c = options.keyPrefix) !== null && _c !== void 0 ? _c : '';
        for (const file of fs.readdirSync(localDir)) {
            const filePath = path.join(localDir, file);
            if (fs.statSync(filePath).isDirectory()) {
                continue;
            }
            if (!shouldInclude(file)) {
                continue;
            }
            const relativeFilePath = keyPrefix + file;
            this.addFile(filePath, relativeFilePath);
        }
    }
    verifyKeyAvailable(key) {
        if (key in this._data || key in this._binaryData) {
            throw new Error(`unable to add a ConfigMap entry with key "${key}". It is already used`);
        }
    }
    synthesizeData() {
        return utils_1.undefinedIfEmpty(this._data);
    }
    synthesizeBinaryData() {
        return utils_1.undefinedIfEmpty(this._binaryData);
    }
}
exports.ConfigMap = ConfigMap;
_a = JSII_RTTI_SYMBOL_1;
ConfigMap[_a] = { fqn: "cdk8s-plus-22.ConfigMap", version: "1.0.0-beta.201" };
//# sourceMappingURL=data:application/json;base64,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