"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToDynamoDB = void 0;
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const core_2 = require("@aws-solutions-constructs/core");
/**
 * @summary The ApiGatewayToDynamoDB class.
 */
class ApiGatewayToDynamoDB extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToDynamoDB class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {CloudFrontToApiGatewayToLambdaProps} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        let partitionKeyName;
        // Set the default props for DynamoDB table
        if (props.dynamoTableProps) {
            const dynamoTableProps = core_2.overrideProps(defaults.DefaultTableProps, props.dynamoTableProps);
            partitionKeyName = dynamoTableProps.partitionKey.name;
            this.dynamoTable = new dynamodb.Table(this, 'DynamoTable', dynamoTableProps);
        }
        else {
            partitionKeyName = defaults.DefaultTableProps.partitionKey.name;
            this.dynamoTable = new dynamodb.Table(this, 'DynamoTable', defaults.DefaultTableProps);
        }
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Setup the API Gateway Resource
        const apiGatewayResource = this.apiGateway.root.addResource("{" + partitionKeyName + "}");
        // Setup API Gateway Method
        // Create
        if (props.allowCreateOperation && props.allowCreateOperation === true && props.createRequestTemplate) {
            const createRequestTemplate = props.createRequestTemplate.replace("${Table}", this.dynamoTable.tableName);
            this.addActionToPolicy("dynamodb:PutItem");
            this.addMethod(this.apiGateway.root, createRequestTemplate, "PutItem", "POST");
        }
        // Read
        if (!props.allowReadOperation || props.allowReadOperation === true) {
            const getRequestTemplate = "{\r\n\"TableName\": \"" + this.dynamoTable.tableName + "\",\r\n \"KeyConditionExpression\": \"" + partitionKeyName + " = :v1\",\r\n    \"ExpressionAttributeValues\": {\r\n        \":v1\": {\r\n            \"S\": \"$input.params('" + partitionKeyName + "')\"\r\n        }\r\n    }\r\n}";
            this.addActionToPolicy("dynamodb:Query");
            this.addMethod(apiGatewayResource, getRequestTemplate, "Query", "GET");
        }
        // Update
        if (props.allowUpdateOperation && props.allowUpdateOperation === true && props.updateRequestTemplate) {
            const updateRequestTemplate = props.updateRequestTemplate.replace("${Table}", this.dynamoTable.tableName);
            this.addActionToPolicy("dynamodb:UpdateItem");
            this.addMethod(apiGatewayResource, updateRequestTemplate, "UpdateItem", "PUT");
        }
        // Delete
        if (props.allowDeleteOperation && props.allowDeleteOperation === true) {
            const deleteRequestTemplate = "{\r\n  \"TableName\": \"" + this.dynamoTable.tableName + "\",\r\n  \"Key\": {\r\n    \"" + partitionKeyName + "\": {\r\n      \"S\": \"$input.params('" + partitionKeyName + "')\"\r\n    }\r\n  },\r\n  \"ConditionExpression\": \"attribute_not_exists(Replies)\",\r\n  \"ReturnValues\": \"ALL_OLD\"\r\n}";
            this.addActionToPolicy("dynamodb:DeleteItem");
            this.addMethod(apiGatewayResource, deleteRequestTemplate, "DeleteItem", "DELETE");
        }
    }
    addActionToPolicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.dynamoTable.tableArn
            ],
            actions: [`${action}`]
        }));
    }
    addMethod(apiResource, requestTemplate, dynamodbAction, apiMethod) {
        // Setup the API Gateway Integration
        const apiGatewayIntegration = new api.AwsIntegration({
            service: "dynamodb",
            action: dynamodbAction,
            integrationHttpMethod: "POST",
            options: {
                passthroughBehavior: api.PassthroughBehavior.NEVER,
                credentialsRole: this.apiGatewayRole,
                requestParameters: {
                    "integration.request.header.Content-Type": "'application/json'"
                },
                requestTemplates: {
                    "application/json": requestTemplate
                },
                integrationResponses: [
                    {
                        statusCode: "200"
                    },
                    {
                        statusCode: "500",
                        responseTemplates: {
                            "text/html": "Error"
                        },
                        selectionPattern: "500"
                    }
                ]
            }
        });
        // Setup the API Gateway method(s)
        apiResource.addMethod(apiMethod, apiGatewayIntegration, {
            methodResponses: [
                {
                    statusCode: "200",
                    responseParameters: {
                        "method.response.header.Content-Type": true
                    }
                },
                {
                    statusCode: "500",
                    responseParameters: {
                        "method.response.header.Content-Type": true
                    },
                }
            ]
        });
    }
}
exports.ApiGatewayToDynamoDB = ApiGatewayToDynamoDB;
//# sourceMappingURL=data:application/json;base64,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