"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const dynamodb = require("@aws-cdk/aws-dynamodb");
const cdk = require("@aws-cdk/core");
const global_table_coordinator_1 = require("./global-table-coordinator");
/**
 * This class works by deploying an AWS DynamoDB table into each region specified in  GlobalTableProps.regions[],
 * then triggering a CloudFormation Custom Resource Lambda to link them all together to create linked AWS Global DynamoDB tables.
 */
class GlobalTable extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Creates dynamoDB tables across regions that will be able to be globbed together into a global table
         */
        this._regionalTables = new Array();
        this._regionalTables = [];
        if (props.stream != null && props.stream !== dynamodb.StreamViewType.NEW_AND_OLD_IMAGES) {
            throw new Error("dynamoProps.stream MUST be set to dynamodb.StreamViewType.NEW_AND_OLD_IMAGES");
        }
        // need to set this stream specification, otherwise global tables don't work
        // And no way to set a default value in an interface
        const regionalTableProps = Object.assign(Object.assign({}, props), { removalPolicy: props.removalPolicy, stream: dynamodb.StreamViewType.NEW_AND_OLD_IMAGES });
        // here we loop through the configured regions.
        // in each region we'll deploy a separate stack with a DynamoDB Table with identical properties in the individual stacks
        for (const reg of props.regions) {
            const regionalStack = new cdk.Stack(this, id + "-" + reg, { env: { region: reg } });
            const regionalTable = new dynamodb.Table(regionalStack, `${id}-GlobalTable-${reg}`, regionalTableProps);
            this._regionalTables.push(regionalTable);
        }
        this.lambdaGlobalTableCoordinator = new global_table_coordinator_1.GlobalTableCoordinator(scope, id + "-CustomResource", props);
        for (const table of this._regionalTables) {
            this.lambdaGlobalTableCoordinator.node.addDependency(table);
        }
    }
    /**
     * Obtain tables deployed in other each region
     */
    get regionalTables() {
        return this._regionalTables.map(x => x);
    }
}
exports.GlobalTable = GlobalTable;
//# sourceMappingURL=data:application/json;base64,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