# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['samplespace']

package_data = \
{'': ['*']}

install_requires = \
['xxhash>=1.4,<2.0']

setup_kwargs = {
    'name': 'pysamplespace',
    'version': '1.0.0',
    'description': 'SampleSpace is a cross-platform library for describing and sampling from random distributions.',
    'long_description': '![SampleSpace banner](docs/source/img/header.png "SampleSpace")\n\n# SampleSpace: Cross-Platform Tools for Generating Random Numbers\n\n[![pipeline status](https://gitlab.com/cvpines/pysamplespace/badges/master/pipeline.svg)](https://gitlab.com/cvpines/pysamplespace/-/commits/master)\n[![coverage report](https://gitlab.com/cvpines/pysamplespace/badges/master/coverage.svg)](https://gitlab.com/cvpines/pysamplespace/-/commits/master)\n[![PyPI](https://img.shields.io/pypi/v/pysamplespace)](https://pypi.org/project/pysamplespace/)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/pysamplespace)]((https://pypi.org/project/pysamplespace/))\n[![PyPI - License](https://img.shields.io/pypi/l/pysamplespace)](https://gitlab.com/cvpines/pysamplespace/-/blob/master/LICENSE)\n\nSampleSpace is a cross-platform library for describing and sampling from\nrandom distributions.\n\nWhile SampleSpace is primarily intended for creating\nprocedurally-generated content, it is also useful for Monte-Carlo\nsimulations, unit testing, and anywhere that flexible, repeatable random\nnumbers are required.\n\n## Platforms\n\nSampleSpace supports the following platforms:\n\n* Python &mdash; [pysamplespace](https://pypi.org/project/pysamplespace/)\n* C++ &mdash; In development\n* C# &mdash; In development\n\nSampleSpace guarantees that the value of random sequences is consistent\nand serialized states are compatible across each platform implementation.\n\n## Installation\n\nInstallation is simple using `pip`:\n\n> `$ pip install samplespace`\n\nSampleSpace\'s only dependency is\n[xxHash](https://pypi.org/project/xxhash/), though it optionally\noffers additional functionality if\n[PyYAML](https://pypi.org/project/PyYAML/) is installed.\n\n## [Usage](https://pysamplespace.readthedocs.io/en/latest/)\n\nAll documentation is available at on [Read the Docs](https://pysamplespace.readthedocs.io/en/latest/). \n\nSampleSpace provides the following submodules:\n\n* [samplespace.repeatablerandom](https://pysamplespace.readthedocs.io/en/latest/repeatablerandom.html) &mdash;  Repeatable Random Sequences\n* [samplespace.distributions](https://pysamplespace.readthedocs.io/en/latest/distributions.html) &mdash;  Serializable Probability Distributions\n* [samplespace.algorithms](https://pysamplespace.readthedocs.io/en/latest/algorithms.html) &mdash;  General Sampling Algorithms\n* [samplespace.pyyaml_support](https://pysamplespace.readthedocs.io/en/latest/pyyaml_support.html) &mdash;  YAML serialization support\n \n### Repeatable Random Sequences\n\n`samplespace.repeatablerandom` allows for generating repeatable, \ndeterministic random sequences. It is compatible random as a drop-in replacement.\n\nA key feature of `RepeatableRandomSequence` is its ability to get, serialize, and restore internal\nstate. This is especially useful when generating procedural content from a fixed seed.\n\nA `RepeatableRandomSequence` can also be used for unit testing by replacing the built-in random module.\nBecause each random sequence is deterministic and repeatable for a given seed, expected values can be \nrecorded and compared against within unit tests.\n\n`RepeatableRandomSequence` produces high-quality pseudo-random values. See \n[Randomness Test Results](https://pysamplespace.readthedocs.io/en/latest/rrs_quality.html) for \nresults from randomness tests.\n\n```python\nimport samplespace\n\nrrs = samplespace.RepeatableRandomSequence(seed=1234)\n\nsamples = [rrs.randrange(30) for _ in range(10)]\nprint(samples)\n# Will always print:\n# [21, 13, 28, 19, 16, 29, 28, 24, 29, 25]\n\n# Generate some random values to advance the state\n[rrs.random() for _ in range(90)]\n\n# Save the state for later recall\n# State can be serialzied to a dict and serialized as JSON\nstate = rrs.getstate()\nstate_as_dict = state.as_dict()\nstate_as_json = json.dumps(state_as_dict)\nprint(state_as_json)\n# Prints {"seed": 12345, "hash_input": "gxzNfDj4Ypc=", "index": 100}\n\nprint(rrs.random())\n# Will print 0.5940559149714152\n\n# Generate some more values\n[rrs.random() for _ in range(100)]\n\n# Return the sequence to the saved state. The next value will match\n# the value following when the state was saved.\nnew_state_as_dict = json.loads(state_as_json)\nnew_state = samplespace.repeatablerandom.RepeatableRandomSequenceState.from_dict(new_state_as_dict)\nrrs.setstate(new_state)\nprint(rrs.random())\n# Will also print 0.5940559149714152\n```\n\n### Distributions\n\n`samplespace.distributions` implements a number of useful probability distributions.\n\nEach distribution can be sampled using any random number generator providing at least the same\nfunctionality as the random module; this includes `samplespace.repeatablerandom`.\n\nThe classes in this module are primarily intended for storing information on random distributions\nin configuration files using `Distribution.as_dict()`/`distribution_from_dict()`\nor `Distribution.as_list()`/`distribution_from_list()`.\n\nDistributions can be serialized to strings:\n```python\nfrom samplespace.distributions import Pareto, DiscreteUniform, UniformCategorical\n\npareto = Pareto(2.5)\nprint(\'Pareto as dict:\', pareto.as_dict())  # {\'distribution\': \'pareto\', \'alpha\': 2.5}\nprint(\'Pareto as list:\', pareto.as_list())  # [\'pareto\', 2.5]\n\ndiscrete = DiscreteUniform(3, 8)\nprint(\'Discrete uniform as dict:\', discrete.as_dict())  # {\'distribution\': \'discreteuniform\', \'min_val\': 3, \'max_val\': 8}\nprint(\'Discrete uniform as list:\', discrete.as_list())  # [\'discreteuniform\', 3, 8]\n\ncat = UniformCategorical([\'string\', 4, {\'a\':\'dict\'}])\nprint(\'Uniform categorical as dict:\', cat.as_dict())  # {\'distribution\': \'uniformcategorical\', \'population\': [\'string\', 4, {\'a\': \'dict\'}]}\nprint(\'Uniform categorical as list:\', cat.as_list())  # [\'uniformcategorical\', [\'string\', 4, {\'a\': \'dict\'}]]\n``` \n\nDistributions can be specified in configuration files:\n```python\nfrom samplespace import distributions, RepeatableRandomSequence\n\ncity_config = {\n    "building_distribution": {\n        "distribution": "weightedcategorical",\n        "items": [\n            ["house", 0.2],\n            ["store", 0.4],\n            ["tree", 0.8],\n            ["ground", 5.0]\n        ]\n    }\n}\n\nrrs = RepeatableRandomSequence()\nbuilding_dist = distributions.distribution_from_dict(city_config[\'building_distribution\'])\n\nbuildings = [[building_dist.sample(rrs) for col in range(20)] for row in range(5)]\n\nfor row in buildings:\n    for building_type in row:\n        if building_type == \'house\':\n            print(\'H\', end=\'\')\n        elif building_type == \'store\':\n            print(\'S\', end=\'\')\n        elif building_type == \'tree\':\n            print(\'T\', end=\'\')\n        else:\n            print(\'.\', end=\'\')\n    print()\n```\n\n### Algorithms\n\n`samplespace.algorithms` implements several general-purpose sampling algorithms\nsuch as binary roulette wheel sampling and alias tables.\n\n\n### PyYAML Support\n\nSampleSpace provides optional support for [PyYAML](https://pypi.org/project/PyYAML/),\nwhich can be enabled via the `samplespace.pyyaml_support` submodule.\n\nRepeatable Random Sequences:\n\n```python\nimport yaml\nfrom samplespace import RepeatableRandomSequence\nimport samplespace.pyyaml_support\nsamplespace.pyyaml_support.enable_yaml_support()\n\nrrs = RepeatableRandomSequence(seed=678)\n[rrs.randrange(10) for _ in range(5)]\n\n# Serialize the sequence as YAML\nas_yaml = yaml.dump(rrs) # \'!samplespace.rrs\\nhash_input: s1enBV+SSXk=\\nindex: 5\\nseed: 678\\n\'\n\nrrs_from_yaml = yaml.load(as_yaml, Loader=yaml.FullLoader)\n``` \n\nDistributions:\n\n```python\nimport yaml\nfrom samplespace import distributions\nimport samplespace.pyyaml_support\nsamplespace.pyyaml_support.enable_yaml_support()\n\ngamma = distributions.Gamma(5.0, 3.0)\ngamma_as_yaml = yaml.dump(gamma) # \'!samplespace.distribution\\nalpha: 5.0\\nbeta: 3.0\\ndistribution: gamma\\n\'\n\ndist_from_yaml = yaml.load(gamma_as_yaml, Loader=yaml.FullLoader)\n```\n\n\n## Copyright and License \n\nSampleSpace was created by [Coriander V. Pines](http://corianderpines.org)\nand is available under the\n[BSD 3-Clause License](https://gitlab.com/cvpines/pysamplespace/-/blob/master/LICENSE).\n\nThe source is available on [GitLab](https://gitlab.com/cvpines/pysamplespace/).\n',
    'author': 'Coriander V. Pines',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
