"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.TargetDependencyProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * Supported enums for a TargetDependency.
 */
var TargetDependencyProject;
(function (TargetDependencyProject) {
    /**
     * Only rely on the package where the target is called.
     *
     * This is usually done for test like targets where you only want to run unit
     * tests on the target packages without testing all dependent packages.
     */
    TargetDependencyProject["SELF"] = "self";
    /**
     * Target relies on executing the target against all dependencies first.
     *
     * This is usually done for build like targets where you want to build all
     * dependant projects first.
     */
    TargetDependencyProject["DEPENDENCIES"] = "dependencies";
})(TargetDependencyProject = exports.TargetDependencyProject || (exports.TargetDependencyProject = {}));
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: false,
            jest: false,
            package: false,
            prettier: true,
            projenrcTs: true,
            release: false,
            sampleCode: false,
            defaultReleaseBranch: "mainline",
        });
        this.targetOverrides = {};
        this.nxConfig = options.nxConfig;
        this.workspaceConfig = options.workspaceConfig;
        this.implicitDependencies = this.nxConfig?.implicitDependencies || {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("@nrwl/cli", "@nrwl/workspace");
        options.nxConfig?.nxCloudReadOnlyAccessToken &&
            this.addDevDeps("@nrwl/nx-cloud");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(this.nxConfig?.nxIgnore || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs.readFileSync(getPluginPath()).toString("utf-8").split("\n"),
        });
        new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: options.nxConfig?.nxCloudReadOnlyAccessToken
                            ? "@nrwl/nx-cloud"
                            : "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: ["build", "test"],
                            accessToken: options.nxConfig?.nxCloudReadOnlyAccessToken,
                        },
                    },
                },
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(this.nxConfig?.targetDependencies || {}),
                },
                affected: {
                    defaultBase: "mainline",
                },
            },
        });
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    /**
     * Allow project specific target overrides.
     */
    overrideProjectTargets(project, targets) {
        const _package = project.tryFindObjectFile("package.json");
        _package?.addOverride("nx", {
            targets: targets,
        });
        !_package && (this.targetOverrides[project.outdir] = targets);
    }
    // Remove this hack once subProjects is made public in Projen
    get subProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.synthesizeNonNodePackageJson();
        super.synth();
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subProjects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * For non-node projects, a package.json is required in order to be discovered by NX.
     */
    synthesizeNonNodePackageJson() {
        this.subProjects
            .filter((subProject) => !isNodeProject(subProject))
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {};
            (manifest.nx = this.targetOverrides[subProject.outdir]
                ? { targets: this.targetOverrides[subProject.outdir] }
                : undefined),
                (manifest.name = subProject.name);
            manifest.private = true;
            manifest.__pdk__ = true;
            manifest.scripts = subProject.tasks.all.reduce((p, c) => ({
                [c.name]: `npx projen ${c.name}`,
                ...p,
            }), {});
            manifest.version = "0.0.0";
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.subProjects.map((subProject) => path.relative(this.outdir, subProject.outdir)),
                },
            });
        }
        else {
            this.package.addField("workspaces", {
                packages: this.subProjects.map((subProject) => path.relative(this.outdir, subProject.outdir)),
                nohoist: this.workspaceConfig?.noHoist,
            });
        }
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "aws-prototyping-sdk.nx_monorepo.NxMonorepoProject", version: "0.2.14" };
/**
 * Determines if the passed in project is of type NodeProject.
 *
 * @param project Project instance.
 * @returns true if the project instance is of type NodeProject.
 */
function isNodeProject(project) {
    return project instanceof javascript_1.NodeProject || project.package;
}
function getPluginPath() {
    return path.join(__dirname, "plugin/nx-monorepo-plugin.js");
}
//# sourceMappingURL=data:application/json;base64,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