"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const pod_1 = require("./pod");
const service_1 = require("./service");
/**
 * A Deployment provides declarative updates for Pods and ReplicaSets.
 *
 * You describe a desired state in a Deployment, and the Deployment Controller changes the actual
 * state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
 * existing Deployments and adopt all their resources with new Deployments.
 *
 * > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
 *
 * Use Case
 * ---------
 *
 * The following are typical use cases for Deployments:
 *
 * - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
 *    Check the status of the rollout to see if it succeeds or not.
 * - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
 *    A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
 *    Each new ReplicaSet updates the revision of the Deployment.
 * - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
 *    Each rollback updates the revision of the Deployment.
 * - Scale up the Deployment to facilitate more load.
 * - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
 * - Use the status of the Deployment as an indicator that a rollout has stuck.
 * - Clean up older ReplicaSets that you don't need anymore.
 *
 * @stability stable
 */
class Deployment extends base_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c;
        super(scope, id);
        this.apiObject = new k8s.KubeDeployment(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.replicas = (_b = props.replicas) !== null && _b !== void 0 ? _b : 1;
        this._podTemplate = new pod_1.PodTemplate(props);
        this._labelSelector = {};
        if ((_c = props.defaultSelector) !== null && _c !== void 0 ? _c : true) {
            const selector = 'cdk8s.deployment';
            const matcher = cdk8s_1.Names.toLabelValue(this);
            this.podMetadata.addLabel(selector, matcher);
            this.selectByLabel(selector, matcher);
        }
    }
    /**
     * Provides read/write access to the underlying pod metadata of the resource.
     *
     * @stability stable
     */
    get podMetadata() {
        return this._podTemplate.podMetadata;
    }
    /**
     * The labels this deployment will match against in order to select pods.
     *
     * Returns a a copy. Use `selectByLabel()` to add labels.
     *
     * @stability stable
     */
    get labelSelector() {
        return { ...this._labelSelector };
    }
    /**
     * The containers belonging to the pod.
     *
     * Use `addContainer` to add containers.
     *
     * @stability stable
     */
    get containers() {
        return this._podTemplate.containers;
    }
    /**
     * The volumes associated with this pod.
     *
     * Use `addVolume` to add volumes.
     *
     * @stability stable
     */
    get volumes() {
        return this._podTemplate.volumes;
    }
    /**
     * Restart policy for all containers within the pod.
     *
     * @stability stable
     */
    get restartPolicy() {
        return this._podTemplate.restartPolicy;
    }
    /**
     * The service account used to run this pod.
     *
     * @stability stable
     */
    get serviceAccount() {
        return this._podTemplate.serviceAccount;
    }
    /**
     * Configure a label selector to this deployment.
     *
     * Pods that have the label will be selected by deployments configured with this spec.
     *
     * @param key - The label key.
     * @param value - The label value.
     * @stability stable
     */
    selectByLabel(key, value) {
        this._labelSelector[key] = value;
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options Options to determine details of the service and port exposed.
     * @stability stable
     */
    exposeViaService(options = {}) {
        var _b;
        const service = new service_1.Service(this, 'Service', {
            metadata: options.name ? { name: options.name } : undefined,
            type: (_b = options.serviceType) !== null && _b !== void 0 ? _b : service_1.ServiceType.CLUSTER_IP,
        });
        service.addDeployment(this, { protocol: options.protocol, targetPort: options.targetPort, port: options.port });
        return service;
    }
    /**
     * Expose a deployment via an ingress.
     *
     * This will first expose the deployment with a service, and then expose the service via an ingress.
     *
     * @param path The ingress path to register under.
     * @param options Additional options.
     * @stability stable
     */
    exposeViaIngress(path, options = {}) {
        const service = this.exposeViaService(options);
        return service.exposeViaIngress(path, options);
    }
    /**
     * Add a container to the pod.
     *
     * @stability stable
     */
    addContainer(container) {
        return this._podTemplate.addContainer(container);
    }
    /**
     * Add a volume to the pod.
     *
     * @stability stable
     */
    addVolume(volume) {
        return this._podTemplate.addVolume(volume);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            replicas: this.replicas,
            template: this._podTemplate._toPodTemplateSpec(),
            selector: {
                matchLabels: this._labelSelector,
            },
        };
    }
}
exports.Deployment = Deployment;
_a = JSII_RTTI_SYMBOL_1;
Deployment[_a] = { fqn: "cdk8s-plus-22.Deployment", version: "1.0.0-beta.28" };
//# sourceMappingURL=data:application/json;base64,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