import { IConfigMap } from './config-map';
import * as k8s from './imports/k8s';
import { Probe } from './probe';
import { SecretValue } from './secret';
import { Volume } from './volume';
/**
 * Options to specify an envionment variable value from a ConfigMap key.
 *
 * @stability stable
 */
export interface EnvValueFromConfigMapOptions {
    /**
     * Specify whether the ConfigMap or its key must be defined.
     *
     * @default false
     * @stability stable
     */
    readonly optional?: boolean;
}
/**
 * Options to specify an environment variable value from a Secret.
 *
 * @stability stable
 */
export interface EnvValueFromSecretOptions {
    /**
     * Specify whether the Secret or its key must be defined.
     *
     * @default false
     * @stability stable
     */
    readonly optional?: boolean;
}
/**
 * Options to specify an environment variable value from the process environment.
 *
 * @stability stable
 */
export interface EnvValueFromProcessOptions {
    /**
     * Specify whether the key must exist in the environment.
     *
     * If this is set to true, and the key does not exist, an error will thrown.
     *
     * @default false
     * @stability stable
     */
    readonly required?: boolean;
}
/**
 * Utility class for creating reading env values from various sources.
 *
 * @stability stable
 */
export declare class EnvValue {
    readonly value?: any;
    readonly valueFrom?: any;
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     * @stability stable
     */
    static fromConfigMap(configMap: IConfigMap, key: string, options?: EnvValueFromConfigMapOptions): EnvValue;
    /**
     * Defines an environment value from a secret JSON value.
     *
     * @param secretValue The secret value (secrent + key).
     * @param options Additional options.
     * @stability stable
     */
    static fromSecretValue(secretValue: SecretValue, options?: EnvValueFromSecretOptions): EnvValue;
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     * @stability stable
     */
    static fromValue(value: string): EnvValue;
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     * @stability stable
     */
    static fromProcess(key: string, options?: EnvValueFromProcessOptions): EnvValue;
    private constructor();
}
/**
 * @stability stable
 */
export declare enum ImagePullPolicy {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry to resolve the name to an image digest.
     *
     * If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     *
     * @stability stable
     */
    ALWAYS = "Always",
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     *
     * @stability stable
     */
    IF_NOT_PRESENT = "IfNotPresent",
    /**
     * The image is assumed to exist locally.
     *
     * No attempt is made to pull the image.
     *
     * @stability stable
     */
    NEVER = "Never"
}
/**
 * Properties for creating a container.
 *
 * @stability stable
 */
export interface ContainerProps {
    /**
     * Docker image name.
     *
     * @stability stable
     */
    readonly image: string;
    /**
     * Name of the container specified as a DNS_LABEL.
     *
     * Each container in a pod must have a unique name (DNS_LABEL). Cannot be updated.
     *
     * @default 'main'
     * @stability stable
     */
    readonly name?: string;
    /**
     * Number of port to expose on the pod's IP address.
     *
     * This must be a valid port number, 0 < x < 65536.
     *
     * @default - No port is exposed.
     * @stability stable
     */
    readonly port?: number;
    /**
     * Entrypoint array.
     *
     * Not executed within a shell. The docker image's ENTRYPOINT is used if this is not provided. Variable references $(VAR_NAME) are expanded using the container's environment.
     * If a variable cannot be resolved, the reference in the input string will be unchanged. The $(VAR_NAME) syntax can be escaped with a double $$, ie: $$(VAR_NAME).
     * Escaped references will never be expanded, regardless of whether the variable exists or not. Cannot be updated.
     * More info: https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/#running-a-command-in-a-shell
     *
     * @default - The docker image's ENTRYPOINT.
     * @stability stable
     */
    readonly command?: string[];
    /**
     * Arguments to the entrypoint. The docker image's CMD is used if `command` is not provided.
     *
     * Variable references $(VAR_NAME) are expanded using the container's
     * environment. If a variable cannot be resolved, the reference in the input
     * string will be unchanged. The $(VAR_NAME) syntax can be escaped with a
     * double $$, ie: $$(VAR_NAME). Escaped references will never be expanded,
     * regardless of whether the variable exists or not.
     *
     * Cannot be updated.
     *
     * @default []
     * @see https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/#running-a-command-in-a-shell
     * @stability stable
     */
    readonly args?: string[];
    /**
     * Container's working directory.
     *
     * If not specified, the container runtime's default will be used, which might be configured in the container image. Cannot be updated.
     *
     * @default - The container runtime's default.
     * @stability stable
     */
    readonly workingDir?: string;
    /**
     * List of environment variables to set in the container.
     *
     * Cannot be updated.
     *
     * @default - No environment variables.
     * @stability stable
     */
    readonly env?: {
        [name: string]: EnvValue;
    };
    /**
     * Pod volumes to mount into the container's filesystem.
     *
     * Cannot be updated.
     *
     * @stability stable
     */
    readonly volumeMounts?: VolumeMount[];
    /**
     * Image pull policy for this container.
     *
     * @default ImagePullPolicy.ALWAYS
     * @stability stable
     */
    readonly imagePullPolicy?: ImagePullPolicy;
    /**
     * Determines when the container is ready to serve traffic.
     *
     * @default - no readiness probe is defined
     * @stability stable
     */
    readonly readiness?: Probe;
    /**
     * Periodic probe of container liveness.
     *
     * Container will be restarted if the probe fails.
     *
     * @default - no liveness probe is defined
     * @stability stable
     */
    readonly liveness?: Probe;
    /**
     * StartupProbe indicates that the Pod has successfully initialized.
     *
     * If specified, no other probes are executed until this completes successfully
     *
     * @default - no startup probe is defined.
     * @stability stable
     */
    readonly startup?: Probe;
}
/**
 * A single application container that you want to run within a pod.
 *
 * @stability stable
 */
export declare class Container {
    /**
     * The port this container exposes.
     *
     * @stability stable
     */
    readonly port?: number;
    /**
     * Volume mounts configured for this container.
     *
     * @stability stable
     */
    readonly mounts: VolumeMount[];
    /**
     * Image pull policy for this container.
     *
     * @stability stable
     */
    readonly imagePullPolicy: ImagePullPolicy;
    /**
     * The container image.
     *
     * @stability stable
     */
    readonly image: string;
    /**
     * The name of the container.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * The working directory inside the container.
     *
     * @stability stable
     */
    readonly workingDir?: string;
    private readonly _command?;
    private readonly _args?;
    private readonly _env;
    private readonly _readiness?;
    private readonly _liveness?;
    private readonly _startup?;
    /**
     * @stability stable
     */
    constructor(props: ContainerProps);
    /**
     * Entrypoint array (the command to execute when the container starts).
     *
     * @returns a copy of the entrypoint array, cannot be modified
     * @stability stable
     */
    get command(): string[] | undefined;
    /**
     * Arguments to the entrypoint.
     *
     * @returns a copy of the arguments array, cannot be modified.
     * @stability stable
     */
    get args(): string[] | undefined;
    /**
     * Add an environment value to the container.
     *
     * The variable value can come
     * from various dynamic sources such a secrets of config maps.
     *
     * @param name - The variable name.
     * @param value - The variable value.
     * @see EnvValue.fromXXX
     * @stability stable
     */
    addEnv(name: string, value: EnvValue): void;
    /**
     * The environment variables for this container.
     *
     * Returns a copy. To add environment variables use `addEnv()`.
     *
     * @stability stable
     */
    get env(): Record<string, EnvValue>;
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     *
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param volume - The volume to mount.
     * @stability stable
     */
    mount(path: string, volume: Volume, options?: MountOptions): void;
    /**
     * @internal
     */
    _toKube(): k8s.Container;
}
/**
 * Options for mounts.
 *
 * @stability stable
 */
export interface MountOptions {
    /**
     * Determines how mounts are propagated from the host to container and the other way around.
     *
     * When not set, MountPropagationNone is used.
     *
     * Mount propagation allows for sharing volumes mounted by a Container to
     * other Containers in the same Pod, or even to other Pods on the same node.
     *
     * This field is beta in 1.10.
     *
     * @default MountPropagation.NONE
     * @stability stable
     */
    readonly propagation?: MountPropagation;
    /**
     * Mounted read-only if true, read-write otherwise (false or unspecified).
     *
     * Defaults to false.
     *
     * @default false
     * @stability stable
     */
    readonly readOnly?: boolean;
    /**
     * Path within the volume from which the container's volume should be mounted.).
     *
     * @default "" the volume's root
     * @stability stable
     */
    readonly subPath?: string;
    /**
     * Expanded path within the volume from which the container's volume should be mounted.
     *
     * Behaves similarly to SubPath but environment variable references
     * $(VAR_NAME) are expanded using the container's environment. Defaults to ""
     * (volume's root). SubPathExpr and SubPath are mutually exclusive. This field
     * is beta in 1.15.
     *
     * `subPathExpr` and `subPath` are mutually exclusive. This field is beta in
     * 1.15.
     *
     * @default "" volume's root.
     * @stability stable
     */
    readonly subPathExpr?: string;
}
/**
 * Mount a volume from the pod to the container.
 *
 * @stability stable
 */
export interface VolumeMount extends MountOptions {
    /**
     * The volume to mount.
     *
     * @stability stable
     */
    readonly volume: Volume;
    /**
     * Path within the container at which the volume should be mounted.
     *
     * Must not
     * contain ':'.
     *
     * @stability stable
     */
    readonly path: string;
}
/**
 * @stability stable
 */
export declare enum MountPropagation {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted to this volume or any of its subdirectories by the host.
     *
     * In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     *
     * @stability stable
     */
    NONE = "None",
    /**
     * This volume mount will receive all subsequent mounts that are mounted to this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     *
     * @stability stable
     */
    HOST_TO_CONTAINER = "HostToContainer",
    /**
     * This volume mount behaves the same the HostToContainer mount.
     *
     * In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     * @stability stable
     */
    BIDIRECTIONAL = "Bidirectional"
}
