"""Test inclusive ('>=') vs exclusive ('>') comparison operator handling.

This module tests that the comparison operators are correctly threaded through
the optimization algorithms and that they produce the expected differences
in behavior when probabilities are tied at the optimal threshold.

The key principle: '>' excludes ties, '>=' includes ties, and this must be
reflected consistently in threshold selection and plateau sensitivity.
"""

import numpy as np
from hypothesis import given, settings

from optimal_cutoffs import get_optimal_threshold
from optimal_cutoffs.metrics import f1_score, get_confusion_matrix
from tests.strategies import tied_probabilities


class TestComparisonOperatorSemantics:
    """Test fundamental semantics of '>' vs '>=' comparison operators."""

    def test_basic_comparison_difference(self):
        """Basic test showing '>' excludes and '>=' includes tied values."""
        probs = np.array([0.3, 0.5, 0.5, 0.7])

        # With threshold = 0.5:
        # '>' means: 0.3 > 0.5 (False), 0.5 > 0.5 (False), 0.5 > 0.5 (False), 0.7 > 0.5 (True)
        # '>=' means: 0.3 >= 0.5 (False), 0.5 >= 0.5 (True), 0.5 >= 0.5 (True), 0.7 >= 0.5 (True)

        threshold = 0.5
        pred_exclusive = probs > threshold
        pred_inclusive = probs >= threshold

        expected_exclusive = np.array([False, False, False, True])
        expected_inclusive = np.array([False, True, True, True])

        assert np.array_equal(pred_exclusive, expected_exclusive), (
            f"Exclusive '>' failed: expected {expected_exclusive}, got {pred_exclusive}"
        )
        assert np.array_equal(pred_inclusive, expected_inclusive), (
            f"Inclusive '>=' failed: expected {expected_inclusive}, got {pred_inclusive}"
        )

        # Should be different when there are ties
        assert not np.array_equal(pred_exclusive, pred_inclusive), (
            "Exclusive and inclusive should differ when there are ties at threshold"
        )

    def test_no_ties_same_result(self):
        """When no probabilities equal threshold, both operators should give same result."""
        probs = np.array([0.2, 0.4, 0.6, 0.8])
        threshold = 0.5  # No probability equals 0.5

        pred_exclusive = probs > threshold
        pred_inclusive = probs >= threshold

        # Should be identical when no ties
        assert np.array_equal(pred_exclusive, pred_inclusive), (
            f"Should be identical with no ties: exclusive={pred_exclusive}, inclusive={pred_inclusive}"
        )

    def test_all_tied_at_threshold(self):
        """When all probabilities equal threshold, operators should give different results."""
        probs = np.array([0.5, 0.5, 0.5, 0.5])
        threshold = 0.5

        pred_exclusive = probs > threshold  # Should be all False
        pred_inclusive = probs >= threshold  # Should be all True

        assert not pred_exclusive.any(), "Exclusive '>' should exclude all tied values"
        assert pred_inclusive.all(), "Inclusive '>=' should include all tied values"


class TestOptimizationWithTies:
    """Test that optimization algorithms handle tied probabilities correctly."""

    def test_inclusive_changes_decision_on_ties(self):
        """Minimal example where inclusive vs exclusive changes optimal decision."""
        # Carefully constructed case: two items exactly at 0.5, one positive, one negative
        probs = np.array([0.5, 0.5, 0.2])
        labels = np.array([1, 0, 0])  # First tied item is positive, second is negative

        threshold_exclusive = get_optimal_threshold(
            labels, probs, metric="f1", method="sort_scan", comparison=">"
        )
        threshold_inclusive = get_optimal_threshold(
            labels, probs, metric="f1", method="sort_scan", comparison=">="
        )

        pred_exclusive = probs > threshold_exclusive
        pred_inclusive = probs >= threshold_inclusive

        # Should produce different predictions due to tie handling
        # (exact result depends on F1 optimization, but they should differ)
        if np.any(probs == threshold_exclusive) or np.any(probs == threshold_inclusive):
            # If threshold equals one of the probabilities, predictions should differ
            not np.array_equal(pred_exclusive, pred_inclusive)

            # This may not always be true depending on what threshold is selected,
            # so we just verify the algorithms don't crash and produce valid results
            assert len(pred_exclusive) == len(pred_inclusive) == len(probs)

        # Verify both produce valid F1 scores
        f1_exclusive = f1_score(
            *get_confusion_matrix(labels, probs, threshold_exclusive, comparison=">")
        )
        f1_inclusive = f1_score(
            *get_confusion_matrix(labels, probs, threshold_inclusive, comparison=">=")
        )

        assert 0 <= f1_exclusive <= 1, f"Exclusive F1 {f1_exclusive} out of range"
        assert 0 <= f1_inclusive <= 1, f"Inclusive F1 {f1_inclusive} out of range"

    @given(tied_probabilities(tie_fraction=0.4, min_size=8, max_size=30))
    @settings(deadline=None, max_examples=40)
    def test_tied_probabilities_consistency(self, probs):
        """Test consistency with tied probabilities using property-based testing."""
        rng = np.random.default_rng(42)
        labels = (rng.uniform(0, 1, size=len(probs)) < 0.5).astype(int)

        # Ensure both classes present
        if labels.sum() == 0:
            labels[0] = 1
        if labels.sum() == labels.size:
            labels[0] = 0

        for metric in ["f1", "accuracy"]:
            try:
                threshold_exclusive = get_optimal_threshold(
                    labels, probs, metric=metric, method="sort_scan", comparison=">"
                )
                threshold_inclusive = get_optimal_threshold(
                    labels, probs, metric=metric, method="sort_scan", comparison=">="
                )

                # Both should produce valid thresholds
                assert 0 <= threshold_exclusive <= 1
                assert 0 <= threshold_inclusive <= 1

                # Verify predictions are consistent with comparison semantics
                pred_exclusive = probs > threshold_exclusive
                pred_inclusive = probs >= threshold_inclusive

                # Check behavior when probabilities are exactly equal to threshold
                # Use exact equality rather than np.isclose to avoid false positives
                exactly_tied_exclusive = probs == threshold_exclusive
                exactly_tied_inclusive = probs == threshold_inclusive

                if np.any(exactly_tied_exclusive):
                    # Items exactly tied to exclusive threshold should not be predicted positive
                    assert not pred_exclusive[exactly_tied_exclusive].any(), (
                        f"Items exactly tied to exclusive threshold should not be predicted positive with '>'. "
                        f"Threshold: {threshold_exclusive}, Tied items: {probs[exactly_tied_exclusive]}, "
                        f"Predictions: {pred_exclusive[exactly_tied_exclusive]}"
                    )

                if np.any(exactly_tied_inclusive):
                    # Items exactly tied to inclusive threshold should be predicted positive
                    assert pred_inclusive[exactly_tied_inclusive].all(), (
                        f"Items exactly tied to inclusive threshold should be predicted positive with '>='. "
                        f"Threshold: {threshold_inclusive}, Tied items: {probs[exactly_tied_inclusive]}, "
                        f"Predictions: {pred_inclusive[exactly_tied_inclusive]}"
                    )

            except Exception as e:
                # Some edge cases might be unsupported
                if "degenerate" in str(e).lower() or "empty" in str(e).lower():
                    continue
                raise

    def test_plateau_sensitivity_detailed(self):
        """Detailed test of behavior when optimal score plateau includes tied probabilities."""
        # Create a case where multiple thresholds achieve the same optimal score
        # but tie handling affects the actual predictions
        probs = np.array([0.1, 0.4, 0.4, 0.4, 0.8])
        labels = np.array([0, 1, 0, 1, 1])  # Mixed labels with ties at 0.4

        for metric in ["f1", "accuracy"]:
            threshold_exclusive = get_optimal_threshold(
                labels, probs, metric=metric, method="sort_scan", comparison=">"
            )
            threshold_inclusive = get_optimal_threshold(
                labels, probs, metric=metric, method="sort_scan", comparison=">="
            )

            # Apply thresholds
            pred_exclusive = probs > threshold_exclusive
            pred_inclusive = probs >= threshold_inclusive

            # If threshold is exactly at a tied value (0.4), behavior should differ
            if abs(threshold_exclusive - 0.4) < 1e-10:
                tied_indices = np.isclose(probs, 0.4, atol=1e-10)
                assert not pred_exclusive[tied_indices].any(), (
                    "Exclusive should not predict tied values as positive"
                )

            if abs(threshold_inclusive - 0.4) < 1e-10:
                tied_indices = np.isclose(probs, 0.4, atol=1e-10)
                assert pred_inclusive[tied_indices].all(), (
                    "Inclusive should predict tied values as positive"
                )

            # Verify scores are valid
            if metric == "f1":
                score_exclusive = f1_score(
                    *get_confusion_matrix(
                        labels, probs, threshold_exclusive, comparison=">"
                    )
                )
                score_inclusive = f1_score(
                    *get_confusion_matrix(
                        labels, probs, threshold_inclusive, comparison=">="
                    )
                )
            else:  # accuracy
                score_exclusive = np.mean(pred_exclusive == labels)
                score_inclusive = np.mean(pred_inclusive == labels)

            assert 0 <= score_exclusive <= 1
            assert 0 <= score_inclusive <= 1


class TestComparisonThreading:
    """Test that comparison operators are correctly threaded through algorithms."""

    def test_sort_scan_threading(self):
        """Test that sort_scan correctly threads comparison operator to kernel."""
        # Use a case where the operator makes a difference
        probs = np.array([0.2, 0.6, 0.6, 0.8])
        labels = np.array([0, 1, 0, 1])

        # Both should work without error
        threshold_exclusive = get_optimal_threshold(
            labels, probs, metric="f1", method="sort_scan", comparison=">"
        )
        threshold_inclusive = get_optimal_threshold(
            labels, probs, metric="f1", method="sort_scan", comparison=">="
        )

        assert 0 <= threshold_exclusive <= 1
        assert 0 <= threshold_inclusive <= 1

        # Verify that confusion matrices use the correct comparison
        tp_ex, tn_ex, fp_ex, fn_ex = get_confusion_matrix(
            labels, probs, threshold_exclusive, comparison=">"
        )
        tp_in, tn_in, fp_in, fn_in = get_confusion_matrix(
            labels, probs, threshold_inclusive, comparison=">="
        )

        # Totals should match
        assert tp_ex + tn_ex + fp_ex + fn_ex == len(labels)
        assert tp_in + tn_in + fp_in + fn_in == len(labels)

    def test_unique_scan_threading(self):
        """Test that unique_scan correctly handles comparison operators."""
        probs = np.array([0.3, 0.5, 0.5, 0.7])
        labels = np.array([0, 1, 0, 1])

        # Test both comparison operators
        for comparison in [">", ">="]:
            threshold = get_optimal_threshold(
                labels,
                probs,
                metric="accuracy",
                method="unique_scan",
                comparison=comparison,
            )

            assert 0 <= threshold <= 1

            # Verify the threshold works correctly with the specified comparison
            pred = (probs > threshold) if comparison == ">" else (probs >= threshold)
            accuracy = np.mean(pred == labels)

            assert 0 <= accuracy <= 1

    def test_minimize_method_threading(self):
        """Test that minimize method handles comparison operators."""
        probs = np.array([0.1, 0.3, 0.7, 0.9])
        labels = np.array([0, 0, 1, 1])

        for comparison in [">", ">="]:
            try:
                threshold = get_optimal_threshold(
                    labels, probs, metric="f1", method="minimize", comparison=comparison
                )

                assert 0 <= threshold <= 1

                # Verify confusion matrix uses correct comparison
                tp, tn, fp, fn = get_confusion_matrix(
                    labels, probs, threshold, comparison=comparison
                )
                assert tp + tn + fp + fn == len(labels)

            except Exception as e:
                # Some methods might not support all combinations
                if "not supported" in str(e).lower():
                    continue
                raise


class TestEdgeCasesWithComparison:
    """Test edge cases with both comparison operators."""

    def test_boundary_probabilities(self):
        """Test with probabilities exactly at 0.0 and 1.0."""
        probs = np.array([0.0, 0.5, 1.0])
        labels = np.array([0, 1, 1])

        for comparison in [">", ">="]:
            threshold = get_optimal_threshold(
                labels,
                probs,
                metric="accuracy",
                method="sort_scan",
                comparison=comparison,
            )

            # Should handle boundary values gracefully
            assert 0 <= threshold <= 1

            pred = (probs > threshold) if comparison == ">" else (probs >= threshold)

            # Specific boundary behavior
            if threshold == 0.0:
                if comparison == ">":
                    # 0.0 > 0.0 is False, others depend on their values vs 0
                    expected = probs > 0.0
                else:  # '>='
                    # All >= 0.0 should be True
                    expected = np.array([True, True, True])
                assert np.array_equal(pred, expected)

            if threshold == 1.0:
                if comparison == ">":
                    # Only values > 1.0 (none in [0,1]) should be True
                    expected = np.array([False, False, False])
                else:  # '>='
                    # Values >= 1.0 (only 1.0 itself)
                    expected = probs >= 1.0
                assert np.array_equal(pred, expected)

    def test_all_probabilities_tied_at_boundaries(self):
        """Test when all probabilities are tied at 0.0 or 1.0."""
        # All at 0.0
        probs_zero = np.zeros(4)
        labels_zero = np.array([0, 1, 0, 1])

        for comparison in [">", ">="]:
            threshold = get_optimal_threshold(
                labels_zero,
                probs_zero,
                metric="accuracy",
                method="sort_scan",
                comparison=comparison,
            )

            pred = (
                (probs_zero > threshold)
                if comparison == ">"
                else (probs_zero >= threshold)
            )

            # Predictions should be consistent based on threshold and comparison
            if threshold > 0:
                # No probability can be > or >= a positive threshold
                assert not pred.any(), (
                    f"All probs = 0, threshold = {threshold} > 0 should predict all negative"
                )
            elif threshold == 0:
                if comparison == ">":
                    # 0 > 0 is False
                    assert not pred.any()
                else:  # '>='
                    # 0 >= 0 is True
                    assert pred.all()

        # All at 1.0
        probs_one = np.ones(4)
        labels_one = np.array([1, 0, 1, 0])

        for comparison in [">", ">="]:
            threshold = get_optimal_threshold(
                labels_one,
                probs_one,
                metric="accuracy",
                method="sort_scan",
                comparison=comparison,
            )

            pred = (
                (probs_one > threshold)
                if comparison == ">"
                else (probs_one >= threshold)
            )

            # Predictions should be consistent
            if threshold < 1:
                # All probabilities should be > or >= a threshold < 1
                assert pred.all(), (
                    f"All probs = 1, threshold = {threshold} < 1 should predict all positive"
                )
            elif threshold == 1:
                if comparison == ">":
                    # 1 > 1 is False
                    assert not pred.any()
                else:  # '>='
                    # 1 >= 1 is True
                    assert pred.all()

    def test_single_probability_tied(self):
        """Test with a single probability exactly equal to potential threshold."""
        probs = np.array([0.2, 0.4, 0.6, 0.8])
        np.array([0, 0, 1, 1])

        # Force consideration of 0.6 as threshold by making it optimal
        test_threshold = 0.6

        pred_exclusive = probs > test_threshold  # [False, False, False, True]
        pred_inclusive = probs >= test_threshold  # [False, False, True, True]

        expected_exclusive = np.array([False, False, False, True])
        expected_inclusive = np.array([False, False, True, True])

        assert np.array_equal(pred_exclusive, expected_exclusive)
        assert np.array_equal(pred_inclusive, expected_inclusive)

        # Verify different predictions
        assert not np.array_equal(pred_exclusive, pred_inclusive), (
            "Should differ when probability equals threshold"
        )
