"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const sqs = require("aws-cdk-lib/aws-sqs");
const defaults = require("@aws-solutions-constructs/core");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
test('Test default behaviors', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'pipes-test')
        }
    };
    const construct = new lib_1.SqsToPipesToStepfunctions(stack, 'test-sqs-pipes-states', props);
    const template = assertions_1.Template.fromStack(stack);
    expect(construct.pipe).toBeDefined();
    expect(construct.pipeRole).toBeDefined();
    template.hasResourceProperties('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "pipes.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "sqs:ReceiveMessage",
                                "sqs:DeleteMessage",
                                "sqs:GetQueueAttributes"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::GetAtt": [
                                    assertions_1.Match.stringLikeRegexp(`testsqspipesstatesqueue.*`),
                                    "Arn"
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "sourcePolicy"
            },
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "states:StartExecution",
                            Effect: "Allow",
                            Resource: {
                                Ref: assertions_1.Match.stringLikeRegexp(`testsqspipesstatesStateMachine.*`)
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "targetPolicy"
            }
        ]
    });
    template.hasResourceProperties('AWS::Pipes::Pipe', {
        Source: {
            "Fn::GetAtt": [
                assertions_1.Match.stringLikeRegexp(`testsqspipesstatesqueue.*`),
                "Arn"
            ]
        },
        Target: {
            Ref: assertions_1.Match.stringLikeRegexp(`testsqspipesstatesStateMachine.*`),
        },
    });
    template.hasResourceProperties('AWS::Pipes::Pipe', {
        LogConfiguration: {
            CloudwatchLogsLogDestination: {
                LogGroupArn: {
                    "Fn::GetAtt": [
                        assertions_1.Match.stringLikeRegexp(`testsqspipesstatesLogGrouptestsqspipesstates.*`),
                        "Arn"
                    ]
                }
            },
            Level: "INFO"
        }
    });
    template.hasResourceProperties('AWS::Logs::LogGroup', {
        LogGroupName: {
            "Fn::Join": [
                "",
                [
                    assertions_1.Match.stringLikeRegexp('/aws/vendedlogs/pipes/constructs'),
                    {
                        "Fn::Select": [
                            2,
                            assertions_1.Match.anyValue()
                        ]
                    }
                ]
            ]
        },
    });
});
test('Test existing state machine', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const stateMachine = defaults.CreateTestStateMachine(stack, 'state-machine');
    const props = {
        existingStateMachineObj: stateMachine
    };
    const construct = new lib_1.SqsToPipesToStepfunctions(stack, 'test-sqs-pipes-states', props);
    const template = assertions_1.Template.fromStack(stack);
    expect(construct.pipe).toBeDefined();
    expect(construct.pipeRole).toBeDefined();
    template.resourceCountIs("AWS::StepFunctions::StateMachine", 1);
    template.hasResourceProperties('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "pipes.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "sqs:ReceiveMessage",
                                "sqs:DeleteMessage",
                                "sqs:GetQueueAttributes"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::GetAtt": [
                                    assertions_1.Match.stringLikeRegexp(`testsqspipesstatesqueue.*`),
                                    "Arn"
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "sourcePolicy"
            },
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "states:StartExecution",
                            Effect: "Allow",
                            Resource: {
                                Ref: assertions_1.Match.stringLikeRegexp(`statemachine.*`)
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "targetPolicy"
            }
        ]
    });
    template.hasResourceProperties('AWS::Pipes::Pipe', {
        Source: {
            "Fn::GetAtt": [
                assertions_1.Match.stringLikeRegexp(`testsqspipesstatesqueue.*`),
                "Arn"
            ]
        },
        Target: {
            Ref: assertions_1.Match.stringLikeRegexp(`statemachine.*`),
        },
    });
    template.hasResourceProperties('AWS::Pipes::Pipe', {
        LogConfiguration: {
            CloudwatchLogsLogDestination: {
                LogGroupArn: {
                    "Fn::GetAtt": [
                        assertions_1.Match.stringLikeRegexp(`testsqspipesstatesLogGrouptestsqspipesstates.*`),
                        "Arn"
                    ]
                }
            },
            Level: "INFO"
        }
    });
    template.hasResourceProperties('AWS::Logs::LogGroup', {
        LogGroupName: {
            "Fn::Join": [
                "",
                [
                    assertions_1.Match.stringLikeRegexp('/aws/vendedlogs/pipes/constructs'),
                    {
                        "Fn::Select": [
                            2,
                            assertions_1.Match.anyValue()
                        ]
                    }
                ]
            ]
        },
    });
});
test('Test queue and state machine aspects of construct', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'lamstp-test')
        }
    };
    const construct = new lib_1.SqsToPipesToStepfunctions(stack, 'test-sqs-pipes-states', props);
    const template = assertions_1.Template.fromStack(stack);
    expect(construct.sqsQueue).toBeDefined();
    expect(construct.deadLetterQueue).toBeDefined();
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.hasResourceProperties('AWS::SQS::Queue', {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    const stateMachine = construct.stateMachine;
    expect(stateMachine).toBeDefined();
    const cwAlarm = construct.cloudwatchAlarms;
    expect(cwAlarm).toBeDefined();
    expect(construct.stateMachineLogGroup).toBeDefined();
});
test('Test existing queue', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingQueue = new sqs.Queue(stack, 'existing-queue', {});
    const props = {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'lamstp-test')
        },
        existingQueueObj: existingQueue,
    };
    const construct = new lib_1.SqsToPipesToStepfunctions(stack, 'test-sqs-pipes-states', props);
    expect(construct.sqsQueue.queueArn).toEqual(existingQueue.queueArn);
});
test('Test filter', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testFilterPattern = `{
    "body": {
      "state": ["open"]
    }
  }`;
    const props = {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'lamstp-test')
        },
        pipeProps: {
            sourceParameters: {
                filterCriteria: {
                    filters: [{ pattern: testFilterPattern }],
                },
            }
        }
    };
    new lib_1.SqsToPipesToStepfunctions(stack, 'test-sqs-pipes-states', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Pipes::Pipe', {
        SourceParameters: {
            FilterCriteria: {
                Filters: [
                    {
                        Pattern: testFilterPattern
                    }
                ]
            }
        }
    });
});
test('Test target parameters?', () => {
    const testMessageGroupId = "test-id";
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'lamstp-test')
        },
        pipeProps: {
            // We realize that this would not launch because the target is not
            // a queue, but it tests the functionality
            targetParameters: {
                sqsQueueParameters: {
                    messageGroupId: testMessageGroupId,
                }
            }
        }
    };
    new lib_1.SqsToPipesToStepfunctions(stack, 'test-sqs-pipes-states', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Pipes::Pipe', {
        TargetParameters: {
            SqsQueueParameters: {
                MessageGroupId: testMessageGroupId,
            }
        }
    });
});
test('test state machine enrichment', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const enrichmentStateMachine = defaults.CreateTestStateMachine(stack, 'enrichment-state-machine');
    const props = {
        enrichmentStateMachine,
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'pipes-test')
        }
    };
    const construct = new lib_1.SqsToPipesToStepfunctions(stack, 'test', props);
    expect(construct.pipe).toBeDefined();
    expect(construct.pipeRole).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    // Look for additional enrichment permision
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: "states:StartSyncExecution",
                    Effect: "Allow",
                    Resource: {
                        Ref: assertions_1.Match.stringLikeRegexp('enrichmentstatemachine.*'),
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: assertions_1.Match.stringLikeRegexp(`testenrichmentpolicytest.*`),
        Roles: [
            {
                Ref: assertions_1.Match.stringLikeRegexp(`testPipeRoletest.*`)
            }
        ]
    });
    template.hasResourceProperties('AWS::Pipes::Pipe', {
        Enrichment: {
            Ref: assertions_1.Match.stringLikeRegexp("enrichmentstatemachine.*")
        },
    });
});
test('Test lambda function enrichment', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const enrichmentFunction = new lambda.Function(stack, 'enrichment-function', {
        runtime: lambda.Runtime.NODEJS_20_X,
        handler: 'index.handler',
        code: lambda.Code.fromInline(`exports.handler = async (event) => {return;}`)
    });
    const props = {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'pipes-test')
        },
        enrichmentFunction
    };
    const construct = new lib_1.SqsToPipesToStepfunctions(stack, 'enrichment-function-test', props);
    const template = assertions_1.Template.fromStack(stack);
    expect(construct.pipe).toBeDefined();
    expect(construct.pipeRole).toBeDefined();
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: "lambda:InvokeFunction",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            assertions_1.Match.stringLikeRegexp('enrichmentfunction.*'),
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: assertions_1.Match.stringLikeRegexp(`enrichmentfunctiontestenrichmentpolicyenrichmentfunctiontest.*`),
        Roles: [
            {
                Ref: assertions_1.Match.stringLikeRegexp(`enrichmentfunctiontestPipeRoleenrichmentfunctiontest.*`)
            }
        ]
    });
    template.hasResourceProperties('AWS::Pipes::Pipe', {
        Enrichment: {
            "Fn::GetAtt": [
                assertions_1.Match.stringLikeRegexp("enrichmentfunction.*"),
                "Arn"
            ]
        },
    });
});
test('Test custom log level', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'pipes-test')
        },
        logLevel: defaults.PipesLogLevel.TRACE
    };
    const construct = new lib_1.SqsToPipesToStepfunctions(stack, 'test-sqs-pipes-states', props);
    const template = assertions_1.Template.fromStack(stack);
    expect(construct.pipe).toBeDefined();
    expect(construct.pipeRole).toBeDefined();
    template.hasResourceProperties('AWS::Pipes::Pipe', {
        LogConfiguration: {
            Level: "TRACE",
        }
    });
});
test('Test log level OFF', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'pipes-test')
        },
        logLevel: defaults.PipesLogLevel.OFF
    };
    const construct = new lib_1.SqsToPipesToStepfunctions(stack, 'test-sqs-pipes-states', props);
    const template = assertions_1.Template.fromStack(stack);
    expect(construct.pipe).toBeDefined();
    expect(construct.pipeRole).toBeDefined();
    // Should be no LogConfiguration
    template.resourcePropertiesCountIs('AWS::Pipes::Pipe', { LogConfiguration: {} }, 0);
    // One log group for the state machine, none for the pipe
    template.resourceCountIs('AWS::Logs::LogGroup', 1);
});
test('Test custom pipe log props', () => {
    const testRetention = aws_logs_1.RetentionDays.FOUR_MONTHS;
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'pipes-test')
        },
        pipeLogProps: {
            retention: testRetention
        }
    };
    const construct = new lib_1.SqsToPipesToStepfunctions(stack, 'test-sqs-pipes-states', props);
    const template = assertions_1.Template.fromStack(stack);
    expect(construct.pipe).toBeDefined();
    expect(construct.pipeRole).toBeDefined();
    template.hasResourceProperties('AWS::Logs::LogGroup', {
        RetentionInDays: 120,
    });
});
test('Test setting source parameters like batchSize', () => {
    const testBatchSize = 123;
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'pipes-test')
        },
        pipeProps: {
            sourceParameters: {
                sqsQueueParameters: {
                    batchSize: testBatchSize,
                },
            }
        }
    };
    const construct = new lib_1.SqsToPipesToStepfunctions(stack, 'test-sqs-pipes-states', props);
    const template = assertions_1.Template.fromStack(stack);
    expect(construct.pipe).toBeDefined();
    expect(construct.pipeRole).toBeDefined();
    template.hasResourceProperties('AWS::Pipes::Pipe', {
        SourceParameters: {
            SqsQueueParameters: {
                BatchSize: testBatchSize,
            }
        }
    });
});
test('Test sending state machine props and existing state machine is an error', () => {
    // These are all tested in CheckStateMachineProps, so this is just checking that CheckStateMachineProps is called.
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'pipes-test')
        },
        existingStateMachineObj: { pretend: "I'm A State Machine :-)" },
    };
    const app = () => {
        new lib_1.SqsToPipesToStepfunctions(stack, 'test-sqs-pipes-states', props);
    };
    // Assertion
    expect(app).toThrowError('ERROR - If existingStateMachine is provided, no other state machine props are allowed\n');
});
//# sourceMappingURL=data:application/json;base64,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