"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ArtifactPath = exports.Artifact = void 0;
const core_1 = require("@aws-cdk/core");
const validation = require("./validation");
/**
 * An output artifact of an action. Artifacts can be used as input by some actions.
 */
class Artifact {
    constructor(artifactName) {
        this.metadata = {};
        validation.validateArtifactName(artifactName);
        this._artifactName = artifactName;
    }
    /**
     * A static factory method used to create instances of the Artifact class.
     * Mainly meant to be used from `decdk`.
     *
     * @param name the (required) name of the Artifact
     */
    static artifact(name) {
        return new Artifact(name);
    }
    get artifactName() {
        return this._artifactName;
    }
    /**
     * Returns an ArtifactPath for a file within this artifact.
     * CfnOutput is in the form "<artifact-name>::<file-name>"
     * @param fileName The name of the file
     */
    atPath(fileName) {
        return new ArtifactPath(this, fileName);
    }
    /**
     * The artifact attribute for the name of the S3 bucket where the artifact is stored.
     */
    get bucketName() {
        return artifactAttribute(this, 'BucketName');
    }
    /**
     * The artifact attribute for The name of the .zip file that contains the artifact that is
     * generated by AWS CodePipeline, such as 1ABCyZZ.zip.
     */
    get objectKey() {
        return artifactAttribute(this, 'ObjectKey');
    }
    /**
     * The artifact attribute of the Amazon Simple Storage Service (Amazon S3) URL of the artifact,
     * such as https://s3-us-west-2.amazonaws.com/artifactstorebucket-yivczw8jma0c/test/TemplateSo/1ABCyZZ.zip.
     */
    get url() {
        return artifactAttribute(this, 'URL');
    }
    /**
     * Returns a token for a value inside a JSON file within this artifact.
     * @param jsonFile The JSON file name.
     * @param keyName The hash key.
     */
    getParam(jsonFile, keyName) {
        return artifactGetParam(this, jsonFile, keyName);
    }
    /**
     * Returns the location of the .zip file in S3 that this Artifact represents.
     * Used by Lambda's `CfnParametersCode` when being deployed in a CodePipeline.
     */
    get s3Location() {
        return {
            bucketName: this.bucketName,
            objectKey: this.objectKey,
        };
    }
    /**
     * Add arbitrary extra payload to the artifact under a given key.
     * This can be used by CodePipeline actions to communicate data between themselves.
     * If metadata was already present under the given key,
     * it will be overwritten with the new value.
     */
    setMetadata(key, value) {
        this.metadata[key] = value;
    }
    /**
     * Retrieve the metadata stored in this artifact under the given key.
     * If there is no metadata stored under the given key,
     * null will be returned.
     */
    getMetadata(key) {
        return this.metadata[key];
    }
    toString() {
        return this.artifactName;
    }
    /** @internal */
    _setName(name) {
        if (this._artifactName) {
            throw new Error(`Artifact already has name '${this._artifactName}', cannot override it`);
        }
        else {
            this._artifactName = name;
        }
    }
}
exports.Artifact = Artifact;
/**
 * A specific file within an output artifact.
 *
 * The most common use case for this is specifying the template file
 * for a CloudFormation action.
 */
class ArtifactPath {
    constructor(artifact, fileName) {
        this.artifact = artifact;
        this.fileName = fileName;
    }
    static artifactPath(artifactName, fileName) {
        return new ArtifactPath(Artifact.artifact(artifactName), fileName);
    }
    get location() {
        const artifactName = this.artifact.artifactName
            ? this.artifact.artifactName
            : core_1.Lazy.stringValue({ produce: () => this.artifact.artifactName });
        return `${artifactName}::${this.fileName}`;
    }
}
exports.ArtifactPath = ArtifactPath;
function artifactAttribute(artifact, attributeName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetArtifactAtt': [lazyArtifactName, attributeName] });
}
function artifactGetParam(artifact, jsonFile, keyName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetParam': [lazyArtifactName, jsonFile, keyName] });
}
//# sourceMappingURL=data:application/json;base64,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