"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const ddb = require("aws-cdk-lib/aws-dynamodb");
const api = require("aws-cdk-lib/aws-apigateway");
const assertions_1 = require("aws-cdk-lib/assertions");
test("check properties", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiGatewayToDynamoDBProps = {};
    const construct = new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb-default", apiGatewayToDynamoDBProps);
    expect(construct.dynamoTable).toBeDefined();
    expect(construct.apiGateway).toBeDefined();
    expect(construct.apiGatewayRole).toBeDefined();
    expect(construct.apiGatewayCloudWatchRole).toBeDefined();
    expect(construct.apiGatewayLogGroup).toBeDefined();
});
test("check allow CRUD operations", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiGatewayToDynamoDBProps = {
        allowReadOperation: true,
        allowCreateOperation: true,
        createRequestTemplate: "{}",
        allowDeleteOperation: true,
        allowUpdateOperation: true,
        updateRequestTemplate: "{}",
    };
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb", apiGatewayToDynamoDBProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "dynamodb:PutItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
                {
                    Action: "dynamodb:Query",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
                {
                    Action: "dynamodb:UpdateItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
                {
                    Action: "dynamodb:DeleteItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
            ],
            Version: "2012-10-17",
        },
        PolicyName: "testapigatewaydynamodbapigatewayroleDefaultPolicy43AC565D",
        Roles: [
            {
                Ref: "testapigatewaydynamodbapigatewayrole961B19C4",
            },
        ],
    });
    template.hasResourceProperties("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM",
    });
    template.hasResourceProperties("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM",
    });
    template.hasResourceProperties("AWS::ApiGateway::Method", {
        HttpMethod: "PUT",
        AuthorizationType: "AWS_IAM",
    });
    template.hasResourceProperties("AWS::ApiGateway::Method", {
        HttpMethod: "DELETE",
        AuthorizationType: "AWS_IAM",
    });
    template.hasResourceProperties("AWS::ApiGateway::Resource", {
        PathPart: "{id}",
    });
});
test("check allow read and update only", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiGatewayToDynamoDBProps = {
        allowUpdateOperation: true,
        updateRequestTemplate: "{}",
    };
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb", apiGatewayToDynamoDBProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "dynamodb:Query",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
                {
                    Action: "dynamodb:UpdateItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
            ],
            Version: "2012-10-17",
        },
        PolicyName: "testapigatewaydynamodbapigatewayroleDefaultPolicy43AC565D",
        Roles: [
            {
                Ref: "testapigatewaydynamodbapigatewayrole961B19C4",
            },
        ],
    });
    template.hasResourceProperties("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM",
    });
});
test("check using custom partition key for dynamodb", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiGatewayToDynamoDBProps = {
        dynamoTableProps: {
            partitionKey: {
                name: "page_id",
                type: ddb.AttributeType.STRING,
            },
        },
    };
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb", apiGatewayToDynamoDBProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ApiGateway::Resource", {
        PathPart: "{page_id}",
    });
});
test("override apiGatewayProps for api gateway", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiGatewayToDynamoDBProps = {
        apiGatewayProps: {
            description: "This is a sample description for api gateway",
        },
    };
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb", apiGatewayToDynamoDBProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ApiGateway::RestApi", {
        Description: "This is a sample description for api gateway",
    });
});
test("Test deployment ApiGateway AuthorizationType override", () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, "api-gateway-dynamodb", {
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE,
            },
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "NONE",
    });
});
test("Test deployment with existing DynamoDB table", () => {
    const oddPartitionKeyName = 'oddName';
    const oddReadCapacity = 23;
    const stack = new aws_cdk_lib_1.Stack();
    const table = new ddb.Table(stack, "existing-table", {
        partitionKey: {
            name: oddPartitionKeyName,
            type: ddb.AttributeType.STRING,
        },
        readCapacity: oddReadCapacity
    });
    const apiGatewayToDynamoDBProps = {
        existingTableObj: table
    };
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb-default", apiGatewayToDynamoDBProps);
    // Confirm there is only the one table
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::DynamoDB::Table", 1);
    // Confirm that the one table is the one create here
    template.hasResourceProperties("AWS::DynamoDB::Table", {
        ProvisionedThroughput: {
            ReadCapacityUnits: oddReadCapacity,
        }
    });
    template.hasResourceProperties("AWS::ApiGateway::Resource", {
        PathPart: `{${oddPartitionKeyName}}`,
    });
});
test("check setting allowReadOperation=false for dynamodb", () => {
    const stack1 = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack1, "test-api-gateway-dynamodb1", {
        allowReadOperation: true,
        allowDeleteOperation: true
    });
    // Expect two APIG Methods (GET, DELETE) for allowReadOperation and allowDeleteOperation
    assertions_1.Template.fromStack(stack1).resourceCountIs("AWS::ApiGateway::Method", 2);
    const stack2 = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack2, "test-api-gateway-dynamodb2", {
        allowReadOperation: false,
        allowDeleteOperation: true
    });
    // Expect only one APIG Method (DELETE) for allowDeleteOperation
    assertions_1.Template.fromStack(stack2).resourceCountIs("AWS::ApiGateway::Method", 1);
});
test('Construct can override default create request template type', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowCreateOperation: true,
        createRequestTemplate: 'ok',
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            RequestTemplates: {
                'application/json': 'ok'
            }
        }
    });
});
test('Construct can override default read request template type', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowReadOperation: true,
        readRequestTemplate: 'ok',
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'GET',
        Integration: {
            RequestTemplates: {
                'application/json': 'ok'
            }
        }
    });
});
test('Construct can override default update request template type', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowUpdateOperation: true,
        updateRequestTemplate: 'ok',
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'PUT',
        Integration: {
            RequestTemplates: {
                'application/json': 'ok'
            }
        }
    });
});
test('Construct can override default delete request template type', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowDeleteOperation: true,
        deleteRequestTemplate: 'ok',
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'DELETE',
        Integration: {
            RequestTemplates: {
                'application/json': 'ok'
            }
        }
    });
});
test('Construct accepts additional create request templates', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowCreateOperation: true,
        createRequestTemplate: 'create-me',
        additionalCreateRequestTemplates: {
            'text/plain': 'Hello'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            RequestTemplates: {
                'application/json': 'create-me',
                'text/plain': 'Hello'
            }
        }
    });
});
test('Construct accepts additional read request templates', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowReadOperation: true,
        additionalReadRequestTemplates: {
            'text/plain': 'Hello'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'GET',
        Integration: {
            RequestTemplates: {
                'text/plain': 'Hello'
            }
        }
    });
});
test('Construct accepts additional update request templates', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowUpdateOperation: true,
        updateRequestTemplate: 'default-update-template',
        additionalUpdateRequestTemplates: {
            'text/plain': 'additional-update-template'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'PUT',
        Integration: {
            RequestTemplates: {
                'application/json': 'default-update-template',
                'text/plain': 'additional-update-template'
            }
        }
    });
});
test('Construct accepts additional delete request templates', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowDeleteOperation: true,
        additionalDeleteRequestTemplates: {
            'text/plain': 'DeleteMe'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'DELETE',
        Integration: {
            RequestTemplates: {
                'text/plain': 'DeleteMe'
            }
        }
    });
});
test('Construct can customize the api resourceName', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        resourceName: 'my-resource-name',
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ApiGateway::Resource", {
        PathPart: "{my-resource-name}",
    });
});
test('Construct uses default integration responses', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowCreateOperation: true,
        allowReadOperation: true,
        allowUpdateOperation: true,
        allowDeleteOperation: true,
        createRequestTemplate: 'create',
        updateRequestTemplate: 'update'
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            IntegrationResponses: [
                {
                    StatusCode: '200'
                },
                {
                    ResponseTemplates: {
                        'text/html': 'Error'
                    },
                    SelectionPattern: '500',
                    StatusCode: '500'
                }
            ]
        }
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'GET',
        Integration: {
            IntegrationResponses: [
                {
                    StatusCode: '200'
                },
                {
                    ResponseTemplates: {
                        'text/html': 'Error'
                    },
                    SelectionPattern: '500',
                    StatusCode: '500'
                }
            ]
        }
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'PUT',
        Integration: {
            IntegrationResponses: [
                {
                    StatusCode: '200'
                },
                {
                    ResponseTemplates: {
                        'text/html': 'Error'
                    },
                    SelectionPattern: '500',
                    StatusCode: '500'
                }
            ]
        }
    });
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'DELETE',
        Integration: {
            IntegrationResponses: [
                {
                    StatusCode: '200'
                },
                {
                    ResponseTemplates: {
                        'text/html': 'Error'
                    },
                    SelectionPattern: '500',
                    StatusCode: '500'
                }
            ]
        }
    });
});
test('Construct uses custom createIntegrationResponses property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowCreateOperation: true,
        createRequestTemplate: 'OK',
        createIntegrationResponses: [
            {
                statusCode: '200',
                responseTemplates: {
                    'text/html': 'OK'
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'text/html': 'OK'
                    },
                    StatusCode: '200'
                }
            ]
        }
    });
});
test('Construct uses custom readIntegrationResponses property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowReadOperation: true,
        readIntegrationResponses: [
            {
                statusCode: '200',
                responseTemplates: {
                    'text/html': 'OK'
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'GET',
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'text/html': 'OK'
                    },
                    StatusCode: '200'
                }
            ]
        }
    });
});
test('Construct uses custom updateIntegrationResponses property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowUpdateOperation: true,
        updateRequestTemplate: 'OK',
        updateIntegrationResponses: [
            {
                statusCode: '200',
                responseTemplates: {
                    'text/html': 'OK'
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'PUT',
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'text/html': 'OK'
                    },
                    StatusCode: '200'
                }
            ]
        }
    });
});
test('Construct uses custom deleteIntegrationResponses property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowDeleteOperation: true,
        deleteIntegrationResponses: [
            {
                statusCode: '200',
                responseTemplates: {
                    'text/html': 'OK'
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'DELETE',
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'text/html': 'OK'
                    },
                    StatusCode: '200'
                }
            ]
        }
    });
});
test('Construct throws error when createRequestTemplate is set and allowCreateOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        createRequestTemplate: '{}',
    });
    expect(app).toThrowError(/The 'allowCreateOperation' property must be set to true when setting any of the following: 'createRequestTemplate', 'additionalCreateRequestTemplates', 'createIntegrationResponses'/);
});
test('Construct throws error when additionalCreateRequestTemplates is set and allowCreateOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        additionalCreateRequestTemplates: {}
    });
    expect(app).toThrowError(/The 'allowCreateOperation' property must be set to true when setting any of the following: 'createRequestTemplate', 'additionalCreateRequestTemplates', 'createIntegrationResponses'/);
});
test('Construct throws error when createIntegrationResponses is set and allowCreateOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        createIntegrationResponses: []
    });
    expect(app).toThrowError(/The 'allowCreateOperation' property must be set to true when setting any of the following: 'createRequestTemplate', 'additionalCreateRequestTemplates', 'createIntegrationResponses'/);
});
test('Construct throws error when readRequestTemplate is set and allowReadOperation is false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowReadOperation: false,
        readRequestTemplate: '{}',
    });
    expect(app).toThrowError(/The 'allowReadOperation' property must be set to true or undefined when setting any of the following: 'readRequestTemplate', 'additionalReadRequestTemplates', 'readIntegrationResponses'/);
});
test('Construct throws error when additionalReadRequestTemplates is set and allowReadOperation is false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowReadOperation: false,
        additionalReadRequestTemplates: {},
    });
    expect(app).toThrowError(/The 'allowReadOperation' property must be set to true or undefined when setting any of the following: 'readRequestTemplate', 'additionalReadRequestTemplates', 'readIntegrationResponses'/);
});
test('Construct throws error when readIntegrationResponses is set and allowReadOperation is false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        allowReadOperation: false,
        readIntegrationResponses: [],
    });
    expect(app).toThrowError(/The 'allowReadOperation' property must be set to true or undefined when setting any of the following: 'readRequestTemplate', 'additionalReadRequestTemplates', 'readIntegrationResponses'/);
});
test('Construct throws error when updateRequestTemplate is set and allowUpdateOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        updateRequestTemplate: '{}',
    });
    expect(app).toThrowError(/The 'allowUpdateOperation' property must be set to true when setting any of the following: 'updateRequestTemplate', 'additionalUpdateRequestTemplates', 'updateIntegrationResponses'/);
});
test('Construct throws error when additionalUpdateRequestTemplates is set and allowUpdateOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        additionalUpdateRequestTemplates: {}
    });
    expect(app).toThrowError(/The 'allowUpdateOperation' property must be set to true when setting any of the following: 'updateRequestTemplate', 'additionalUpdateRequestTemplates', 'updateIntegrationResponses'/);
});
test('Construct throws error when updateIntegrationResponses is set and allowUpdateOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        updateIntegrationResponses: []
    });
    expect(app).toThrowError(/The 'allowUpdateOperation' property must be set to true when setting any of the following: 'updateRequestTemplate', 'additionalUpdateRequestTemplates', 'updateIntegrationResponses'/);
});
test('Construct throws error when deleteRequestTemplate is set and allowDeleteOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        deleteRequestTemplate: '{}',
    });
    expect(app).toThrowError(/The 'allowDeleteOperation' property must be set to true when setting any of the following: 'deleteRequestTemplate', 'additionalDeleteRequestTemplates', 'deleteIntegrationResponses'/);
});
test('Construct throws error when additionalDeleteRequestTemplates is set and allowDeleteOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        additionalDeleteRequestTemplates: {}
    });
    expect(app).toThrowError(/The 'allowDeleteOperation' property must be set to true when setting any of the following: 'deleteRequestTemplate', 'additionalDeleteRequestTemplates', 'deleteIntegrationResponses'/);
});
test('Construct throws error when deleteIntegrationResponses is set and allowDeleteOperation is not true', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => new lib_1.ApiGatewayToDynamoDB(stack, 'api-gateway-dynamodb', {
        deleteIntegrationResponses: []
    });
    expect(app).toThrowError(/The 'allowDeleteOperation' property must be set to true when setting any of the following: 'deleteRequestTemplate', 'additionalDeleteRequestTemplates', 'deleteIntegrationResponses'/);
});
test('Test that CheckDynamoDBProps is getting called', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const tableName = 'custom-table-name';
    const existingTable = new ddb.Table(stack, 'MyTablet', {
        tableName,
        partitionKey: {
            name: 'id',
            type: ddb.AttributeType.STRING
        }
    });
    const props = {
        existingTableObj: existingTable,
        dynamoTableProps: {
            tableName,
            partitionKey: {
                name: 'id',
                type: ddb.AttributeType.STRING
            },
        },
    };
    const app = () => {
        new lib_1.ApiGatewayToDynamoDB(stack, 'test-apigateway-dynamodb-stack', props);
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide existingTableObj or dynamoTableProps, but not both.\n/);
});
//# sourceMappingURL=data:application/json;base64,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