"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToDynamoDB = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
const core_1 = require("@aws-solutions-constructs/core");
/**
 * @summary The ApiGatewayToDynamoDB class.
 */
class ApiGatewayToDynamoDB extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToDynamoDB class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {CloudFrontToApiGatewayToLambdaProps} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckDynamoDBProps(props);
        if (this.CheckCreateRequestProps(props)) {
            throw new Error(`The 'allowCreateOperation' property must be set to true when setting any of the following: ` +
                `'createRequestTemplate', 'additionalCreateRequestTemplates', 'createIntegrationResponses'`);
        }
        if (this.CheckReadRequestProps(props)) {
            throw new Error(`The 'allowReadOperation' property must be set to true or undefined when setting any of the following: ` +
                `'readRequestTemplate', 'additionalReadRequestTemplates', 'readIntegrationResponses'`);
        }
        if (this.CheckUpdateRequestProps(props)) {
            throw new Error(`The 'allowUpdateOperation' property must be set to true when setting any of the following: ` +
                `'updateRequestTemplate', 'additionalUpdateRequestTemplates', 'updateIntegrationResponses'`);
        }
        if (this.CheckDeleteRequestProps(props)) {
            throw new Error(`The 'allowDeleteOperation' property must be set to true when setting any of the following: ` +
                `'deleteRequestTemplate', 'additionalDeleteRequestTemplates', 'deleteIntegrationResponses'`);
        }
        // Set the default props for DynamoDB table
        const dynamoTableProps = defaults.consolidateProps(defaults.DefaultTableProps, props.dynamoTableProps);
        let partitionKeyName = dynamoTableProps.partitionKey.name;
        if (props.existingTableObj) {
            partitionKeyName = core_1.getPartitionKeyNameFromTable(props.existingTableObj);
        }
        const resourceName = props.resourceName ?? partitionKeyName;
        // Since we are only invoking this function with an existing Table or tableProps,
        // (not a table interface), we know that the implementation will always return
        // a Table object and we can force assignment to the dynamoTable property.
        const buildDynamoDBTableResponse = defaults.buildDynamoDBTable(this, {
            existingTableObj: props.existingTableObj,
            dynamoTableProps: props.dynamoTableProps
        });
        this.dynamoTable = buildDynamoDBTableResponse.tableObject;
        // Setup the API Gateway
        const globalRestApiResponse = defaults.GlobalRestApi(this, props.apiGatewayProps, props.logGroupProps);
        this.apiGateway = globalRestApiResponse.api;
        this.apiGatewayCloudWatchRole = globalRestApiResponse.role;
        this.apiGatewayLogGroup = globalRestApiResponse.logGroup;
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Setup the API Gateway Resource
        const apiGatewayResource = this.apiGateway.root.addResource("{" + resourceName + "}");
        // Setup API Gateway Method
        // Create
        if (this.ImplementCreateOperation(props)) {
            // ImplementCreateOperation has confirmed that createRequestTemplate exists)
            const createRequestTemplate = props.createRequestTemplate.replace("${Table}", this.dynamoTable.tableName);
            this.addActionToPolicy("dynamodb:PutItem");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "PutItem",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "POST",
                apiResource: this.apiGateway.root,
                requestTemplate: createRequestTemplate,
                additionalRequestTemplates: props.additionalCreateRequestTemplates,
                integrationResponses: props.createIntegrationResponses
            });
        }
        // Read
        if (this.ImplementReaOperation(props)) {
            const readRequestTemplate = props.readRequestTemplate ??
                `{ \
          "TableName": "${this.dynamoTable.tableName}", \
          "KeyConditionExpression": "${partitionKeyName} = :v1", \
          "ExpressionAttributeValues": { \
            ":v1": { \
              "S": "$input.params('${resourceName}')" \
            } \
          } \
        }`;
            this.addActionToPolicy("dynamodb:Query");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "Query",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "GET",
                apiResource: apiGatewayResource,
                requestTemplate: readRequestTemplate,
                additionalRequestTemplates: props.additionalReadRequestTemplates,
                integrationResponses: props.readIntegrationResponses
            });
        }
        // Update
        if (this.ImplementUpdateOperation(props)) {
            // ImplementUpdateOperation confirmed the existence of updateRequestTemplate
            const updateRequestTemplate = props.updateRequestTemplate.replace("${Table}", this.dynamoTable.tableName);
            this.addActionToPolicy("dynamodb:UpdateItem");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "UpdateItem",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "PUT",
                apiResource: apiGatewayResource,
                requestTemplate: updateRequestTemplate,
                additionalRequestTemplates: props.additionalUpdateRequestTemplates,
                integrationResponses: props.updateIntegrationResponses
            });
        }
        // Delete
        if (this.ImplementDeleteOperation(props)) {
            const deleteRequestTemplate = props.deleteRequestTemplate ??
                `{ \
          "TableName": "${this.dynamoTable.tableName}", \
          "Key": { \
            "${partitionKeyName}": { \
              "S": "$input.params('${resourceName}')" \
              } \
            }, \
          "ReturnValues": "ALL_OLD" \
        }`;
            this.addActionToPolicy("dynamodb:DeleteItem");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "DeleteItem",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "DELETE",
                apiResource: apiGatewayResource,
                requestTemplate: deleteRequestTemplate,
                additionalRequestTemplates: props.additionalDeleteRequestTemplates,
                integrationResponses: props.deleteIntegrationResponses
            });
        }
    }
    CheckReadRequestProps(props) {
        if ((props.readRequestTemplate || props.additionalReadRequestTemplates || props.readIntegrationResponses)
            && props.allowReadOperation === false) {
            return true;
        }
        return false;
    }
    CheckUpdateRequestProps(props) {
        if ((props.updateRequestTemplate || props.additionalUpdateRequestTemplates || props.updateIntegrationResponses)
            && props.allowUpdateOperation !== true) {
            return true;
        }
        return false;
    }
    CheckDeleteRequestProps(props) {
        if ((props.deleteRequestTemplate || props.additionalDeleteRequestTemplates || props.deleteIntegrationResponses)
            && props.allowDeleteOperation !== true) {
            return true;
        }
        return false;
    }
    CheckCreateRequestProps(props) {
        if ((props.createRequestTemplate || props.additionalCreateRequestTemplates || props.createIntegrationResponses)
            && props.allowCreateOperation !== true) {
            return true;
        }
        return false;
    }
    ImplementCreateOperation(props) {
        if (props.allowCreateOperation && props.allowCreateOperation === true && props.createRequestTemplate) {
            return true;
        }
        return false;
    }
    ImplementReaOperation(props) {
        if (props.allowReadOperation === undefined || props.allowReadOperation === true) {
            return true;
        }
        return false;
    }
    ImplementUpdateOperation(props) {
        if (props.allowUpdateOperation && props.allowUpdateOperation === true && props.updateRequestTemplate) {
            return true;
        }
        return false;
    }
    ImplementDeleteOperation(props) {
        if (props.allowDeleteOperation && props.allowDeleteOperation === true) {
            return true;
        }
        return false;
    }
    addActionToPolicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.dynamoTable.tableArn
            ],
            actions: [`${action}`]
        }));
    }
}
exports.ApiGatewayToDynamoDB = ApiGatewayToDynamoDB;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToDynamoDB[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-dynamodb.ApiGatewayToDynamoDB", version: "2.50.0" };
//# sourceMappingURL=data:application/json;base64,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