/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import * as api from 'aws-cdk-lib/aws-apigateway';
import * as iam from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
import * as dynamodb from 'aws-cdk-lib/aws-dynamodb';
import * as logs from 'aws-cdk-lib/aws-logs';
/**
 * @summary The properties for the ApiGatewayToDynamoDB class.
 */
export interface ApiGatewayToDynamoDBProps {
    /**
     * Optional user provided props to override the default props
     *
     * @default - Default props are used
     */
    readonly dynamoTableProps?: dynamodb.TableProps;
    /**
     * Existing instance of DynamoDB table object, providing both this and `dynamoTableProps` will cause an error.
     *
     * @default - None
     */
    readonly existingTableObj?: dynamodb.Table;
    /**
     * Optional user-provided props to override the default props for the API Gateway.
     *
     * @default - Default properties are used.
     */
    readonly apiGatewayProps?: api.RestApiProps;
    /**
     * Optional resource name on the API
     * This property is useful if your integration does not directly use the partition key name
     *
     * @default - partition key name, retrieved from the DynamoDB table object
     */
    readonly resourceName?: string;
    /**
     * Whether to deploy an API Gateway Method for POST HTTP operations on the DynamoDB table (i.e. dynamodb:PutItem).
     *
     * @default - false
     */
    readonly allowCreateOperation?: boolean;
    /**
     * API Gateway Request Template for the create method for the default `application/json` content-type.
     * This property is required if the `allowCreateOperation` property is set to true.
     *
     * @default - None
     */
    readonly createRequestTemplate?: string;
    /**
     * Optional Create Request Templates for content-types other than `application/json`.
     * Use the `createRequestTemplate` property to set the request template for the `application/json` content-type.
     * This property can only be specified if the `allowCreateOperation` property is set to true.
     *
     * @default - None
     */
    readonly additionalCreateRequestTemplates?: {
        [contentType: string]: string;
    };
    /**
     * Optional, custom API Gateway Integration Response for the create method.
     * This property can only be specified if the `allowCreateOperation` property is set to true.
     *
     * @default - [{statusCode:"200"},{statusCode:"500",responseTemplates:{"text/html":"Error"},selectionPattern:"500"}]
     */
    readonly createIntegrationResponses?: api.IntegrationResponse[];
    /**
     * Whether to deploy an API Gateway Method for GET HTTP operations on DynamoDB table (i.e. dynamodb:Query).
     *
     * @default - true
     */
    readonly allowReadOperation?: boolean;
    /**
     * API Gateway Request Template for the read method for the default `application/json` content-type.
     *
     * The default template only supports a partition key and not partition + sort keys.
     *
     * @default - `{ \
     *       "TableName": "DYNAMODB_TABLE_NAME", \
     *       "KeyConditionExpression": "PARTITION_KEY_NAME = :v1", \
     *       "ExpressionAttributeValues": { \
     *         ":v1": { \
     *           "S": "$input.params('PARTITION_KEY_NAME')" \
     *         } \
     *       } \
     *     }`
     */
    readonly readRequestTemplate?: string;
    /**
     * Optional Read Request Templates for content-types other than `application/json`.
     * Use the `readRequestTemplate` property to set the request template for the `application/json` content-type.
     *
     * @default - None
     */
    readonly additionalReadRequestTemplates?: {
        [contentType: string]: string;
    };
    /**
     * Optional, custom API Gateway Integration Response for the read method.
     *
     * @default - [{statusCode:"200"},{statusCode:"500",responseTemplates:{"text/html":"Error"},selectionPattern:"500"}]
     */
    readonly readIntegrationResponses?: api.IntegrationResponse[];
    /**
     * Whether to deploy API Gateway Method for PUT HTTP operations on DynamoDB table (i.e. dynamodb:UpdateItem).
     *
     * @default - false
     */
    readonly allowUpdateOperation?: boolean;
    /**
     * API Gateway Request Template for the update method.
     * This property is required if the `allowUpdateOperation` property is set to true.
     *
     * @default - None
     */
    readonly updateRequestTemplate?: string;
    /**
     * Optional Update Request Templates for content-types other than `application/json`.
     * Use the `updateRequestTemplate` property to set the request template for the `application/json` content-type.
     * This property can only be specified if the `allowUpdateOperation` property is set to true.
     *
     * @default - None
     */
    readonly additionalUpdateRequestTemplates?: {
        [contentType: string]: string;
    };
    /**
     * Optional, custom API Gateway Integration Response for the update method.
     * This property can only be specified if the `allowUpdateOperation` property is set to true.
     *
     * @default - [{statusCode:"200"},{statusCode:"500",responseTemplates:{"text/html":"Error"},selectionPattern:"500"}]
     */
    readonly updateIntegrationResponses?: api.IntegrationResponse[];
    /**
     * Whether to deploy API Gateway Method for DELETE HTTP operations on DynamoDB table (i.e. dynamodb:DeleteItem).
     *
     * @default - false
     */
    readonly allowDeleteOperation?: boolean;
    /**
     * API Gateway Request Template for the delete method for the default `application/json` content-type.
     * This property can only be specified if the `allowDeleteOperation` property is set to true.
     *
     * @default - `{ \
     *       "TableName": "DYNAMODB_TABLE_NAME", \
     *       "Key": { \
     *         "${partitionKeyName}": { \
     *           "S": "$input.params('PARTITION_KEY_NAME')" \
     *           } \
     *         }, \
     *       "ReturnValues": "ALL_OLD" \
     *     }`
     */
    readonly deleteRequestTemplate?: string;
    /**
     * Optional Delete request templates for content-types other than `application/json`.
     * Use the `deleteRequestTemplate` property to set the request template for the `application/json` content-type.
     * This property can only be specified if the `allowDeleteOperation` property is set to true.
     *
     * @default - None
     */
    readonly additionalDeleteRequestTemplates?: {
        [contentType: string]: string;
    };
    /**
     * Optional, custom API Gateway Integration Response for the delete method.
     * This property can only be specified if the `allowDeleteOperation` property is set to true.
     *
     * @default - [{statusCode:"200"},{statusCode:"500",responseTemplates:{"text/html":"Error"},selectionPattern:"500"}]
     */
    readonly deleteIntegrationResponses?: api.IntegrationResponse[];
    /**
     * User provided props to override the default props for the CloudWatchLogs LogGroup.
     *
     * @default - Default props are used
     */
    readonly logGroupProps?: logs.LogGroupProps;
}
/**
 * @summary The ApiGatewayToDynamoDB class.
 */
export declare class ApiGatewayToDynamoDB extends Construct {
    readonly dynamoTable: dynamodb.Table;
    readonly apiGatewayRole: iam.Role;
    readonly apiGateway: api.RestApi;
    readonly apiGatewayCloudWatchRole?: iam.Role;
    readonly apiGatewayLogGroup: logs.LogGroup;
    /**
     * @summary Constructs a new instance of the ApiGatewayToDynamoDB class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {CloudFrontToApiGatewayToLambdaProps} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope: Construct, id: string, props: ApiGatewayToDynamoDBProps);
    private CheckReadRequestProps;
    private CheckUpdateRequestProps;
    private CheckDeleteRequestProps;
    private CheckCreateRequestProps;
    private ImplementCreateOperation;
    private ImplementReaOperation;
    private ImplementUpdateOperation;
    private ImplementDeleteOperation;
    private addActionToPolicy;
}
