"""
AI/ML Project Template
This template creates a basic AI/ML project structure with:
- Data processing pipelines
- Model training and evaluation
- Configuration management
- Experiment tracking
- Jupyter notebooks
"""
import os
from pathlib import Path

def create_ai_ml_project(project_name: str):
    """Create a basic AI/ML project structure."""
    project_path = Path(project_name)
    
    # Create project directories
    (project_path / "data").mkdir(parents=True, exist_ok=True)
    (project_path / "data" / "raw").mkdir(parents=True, exist_ok=True)
    (project_path / "data" / "processed").mkdir(parents=True, exist_ok=True)
    (project_path / "data" / "external").mkdir(parents=True, exist_ok=True)
    (project_path / "data" / "interim").mkdir(parents=True, exist_ok=True)
    (project_path / "notebooks").mkdir(parents=True, exist_ok=True)
    (project_path / "src").mkdir(parents=True, exist_ok=True)
    (project_path / "src" / "data").mkdir(parents=True, exist_ok=True)
    (project_path / "src" / "features").mkdir(parents=True, exist_ok=True)
    (project_path / "src" / "models").mkdir(parents=True, exist_ok=True)
    (project_path / "src" / "visualization").mkdir(parents=True, exist_ok=True)
    (project_path / "src" / "utils").mkdir(parents=True, exist_ok=True)
    (project_path / "reports").mkdir(parents=True, exist_ok=True)
    (project_path / "reports" / "figures").mkdir(parents=True, exist_ok=True)
    (project_path / "tests").mkdir(parents=True, exist_ok=True)
    (project_path / "models").mkdir(parents=True, exist_ok=True)
    (project_path / "docs").mkdir(parents=True, exist_ok=True)
    
    # Create main configuration file
    with open(project_path / "config.yaml", "w") as f:
        f.write(f'''# Configuration for {project_name}

# Data Configuration
data:
  raw_dir: "data/raw"
  processed_dir: "data/processed"
  interim_dir: "data/interim"
  external_dir: "data/external"
  train_file: "train.csv"
  test_file: "test.csv"
  target_column: "target"

# Model Configuration
model:
  name: "RandomForestClassifier"
  hyperparameters:
    n_estimators: 100
    max_depth: 10
    random_state: 42
  save_path: "models/"

# Training Configuration
training:
  test_size: 0.2
  random_state: 42
  cv_folds: 5
  scoring_metric: "accuracy"

# Features Configuration
features:
  categorical_features: []
  numerical_features: []
  text_features: []
  target_transform: false

# Experiment Tracking
experiment:
  name: "{project_name}"
  tracking_uri: "sqlite:///mlruns.db"
  artifact_location: "mlruns"
''')
    
    # Create requirements.txt
    with open(project_path / "requirements.txt", "w") as f:
        f.write('''# Core libraries
numpy==1.26.4
pandas==2.2.2
matplotlib==3.9.0
seaborn==0.13.2
scikit-learn==1.5.0
scipy==1.14.0

# ML/DL Libraries
torch==2.3.1
torchvision==0.18.1
tensorflow==2.16.2
transformers==4.41.2
datasets==2.20.0
accelerate==0.31.0

# Experiment Tracking & Model Management
mlflow==2.13.1
wandb==0.17.4
hydra-core==1.5.0

# Data Processing
pyarrow==16.1.0
polars==0.20.31
dask==2024.6.1

# Utilities
tqdm==4.66.4
click==8.1.7
python-dotenv==1.0.1
requests==2.32.3
yacs==0.1.8
pyyaml==6.0.1

# Development
jupyter==1.0.0
notebook==7.2.1
black==24.4.2
flake8==7.0.0
pytest==8.2.2
pandas-profiling==3.6.6
''')
    
    # Create main src module
    with open(project_path / "src" / "__init__.py", "w") as f:
        f.write(f'''"""
AI/ML Project for {project_name}
Generated with Codeius AI
"""
__version__ = "1.0.0"
__author__ = "Generated by Codeius AI"
''')
    
    # Create data loading module
    with open(project_path / "src" / "data" / "data_loader.py", "w") as f:
        f.write('''"""
Data loading and preprocessing module
"""
import pandas as pd
import numpy as np
import yaml
from pathlib import Path
from typing import Tuple, Dict, Any
import logging

logger = logging.getLogger(__name__)

class DataLoader:
    """Class for loading and preprocessing data."""
    
    def __init__(self, config_path: str = "config.yaml"):
        """Initialize with configuration."""
        with open(config_path, 'r') as f:
            self.config = yaml.safe_load(f)
    
    def load_train_data(self) -> pd.DataFrame:
        """Load training data."""
        data_path = Path(self.config['data']['raw_dir']) / self.config['data']['train_file']
        logger.info(f"Loading training data from {data_path}")
        df = pd.read_csv(data_path)
        logger.info(f"Training data shape: {df.shape}")
        return df
    
    def load_test_data(self) -> pd.DataFrame:
        """Load test data."""
        data_path = Path(self.config['data']['raw_dir']) / self.config['data']['test_file']
        logger.info(f"Loading test data from {data_path}")
        df = pd.read_csv(data_path)
        logger.info(f"Test data shape: {df.shape}")
        return df
    
    def preprocess_data(self, df: pd.DataFrame) -> pd.DataFrame:
        """Apply preprocessing steps to data."""
        logger.info("Starting data preprocessing...")
        
        # Fill missing values
        df = df.fillna(df.mean(numeric_only=True))
        
        # Remove duplicates
        initial_shape = df.shape
        df = df.drop_duplicates()
        logger.info(f"Removed {initial_shape[0] - df.shape[0]} duplicate rows")
        
        # Basic feature transformations
        for col in df.select_dtypes(include=[object]).columns:
            if df[col].nunique() < 0.5 * len(df):
                # Label encode categorical features
                df[col] = df[col].astype('category').cat.codes
        
        logger.info("Data preprocessing completed")
        return df
''')
    
    # Create model trainer module
    with open(project_path / "src" / "models" / "trainer.py", "w") as f:
        f.write('''"""
Model training module
"""
import mlflow
import mlflow.sklearn
import pandas as pd
import numpy as np
from sklearn.model_selection import train_test_split, cross_val_score, GridSearchCV
from sklearn.ensemble import RandomForestClassifier
from sklearn.linear_model import LogisticRegression
from sklearn.metrics import accuracy_score, classification_report, confusion_matrix
from sklearn.preprocessing import StandardScaler
from pathlib import Path
import joblib
import yaml
import logging

logger = logging.getLogger(__name__)

class ModelTrainer:
    """Class for training and evaluating ML models."""
    
    def __init__(self, config_path: str = "config.yaml"):
        """Initialize with configuration."""
        with open(config_path, 'r') as f:
            self.config = yaml.safe_load(f)
        
        self.scaler = StandardScaler()
        self.model = None
    
    def prepare_features(self, df: pd.DataFrame) -> Tuple[np.ndarray, np.ndarray]:
        """Prepare features and target variables."""
        target_col = self.config['data']['target_column']
        
        X = df.drop(columns=[target_col]).values
        y = df[target_col].values
        
        return X, y
    
    def train_model(self, X: np.ndarray, y: np.ndarray):
        """Train the model."""
        logger.info("Starting model training...")
        
        # Scale features
        X_scaled = self.scaler.fit_transform(X)
        
        # Get model configuration
        model_config = self.config['model']
        model_name = model_config['name']
        
        # Initialize model based on config
        if model_name == "RandomForestClassifier":
            self.model = RandomForestClassifier(**model_config['hyperparameters'])
        elif model_name == "LogisticRegression":
            self.model = LogisticRegression(**model_config['hyperparameters'])
        else:
            raise ValueError(f"Unsupported model: {model_name}")
        
        # Start MLflow run
        with mlflow.start_run(run_name=self.config['experiment']['name']):
            # Log parameters
            mlflow.log_params(model_config['hyperparameters'])
            
            # Train model
            self.model.fit(X_scaled, y)
            
            # Log model
            mlflow.sklearn.log_model(self.model, "model")
            
            # Cross-validation scores
            cv_scores = cross_val_score(
                self.model, 
                X_scaled, 
                y, 
                cv=self.config['training']['cv_folds'],
                scoring=self.config['training']['scoring_metric']
            )
            
            # Log metrics
            mlflow.log_metric("cv_mean_score", np.mean(cv_scores))
            mlflow.log_metric("cv_std_score", np.std(cv_scores))
            
            logger.info(f"Model trained successfully. CV Score: {np.mean(cv_scores):.4f} (+/- {np.std(cv_scores) * 2:.4f})")
    
    def evaluate_model(self, X: np.ndarray, y: np.ndarray) -> Dict[str, float]:
        """Evaluate the trained model."""
        if self.model is None:
            raise ValueError("Model not trained yet. Call train_model() first.")
        
        # Scale features
        X_scaled = self.scaler.transform(X)
        
        # Predictions
        y_pred = self.model.predict(X_scaled)
        
        # Metrics
        accuracy = accuracy_score(y, y_pred)
        
        # Log metrics to MLflow
        with mlflow.start_run():
            mlflow.log_metric("test_accuracy", accuracy)
        
        logger.info(f"Model evaluated. Accuracy: {accuracy:.4f}")
        
        return {
            "accuracy": accuracy,
            "classification_report": classification_report(y, y_pred, output_dict=True),
            "confusion_matrix": confusion_matrix(y, y_pred)
        }
    
    def save_model(self):
        """Save the trained model."""
        model_path = Path(self.config['model']['save_path'])
        model_path.mkdir(exist_ok=True)
        
        joblib.dump(self.model, model_path / "model.pkl")
        joblib.dump(self.scaler, model_path / "scaler.pkl")
        logger.info(f"Model saved to {model_path}")
    
    def load_model(self, model_path: str):
        """Load a trained model."""
        self.model = joblib.load(Path(model_path) / "model.pkl")
        self.scaler = joblib.load(Path(model_path) / "scaler.pkl")
        logger.info(f"Model loaded from {model_path}")
''')
    
    # Create main training script
    with open(project_path / "train.py", "w") as f:
        f.write('''"""
Main training script
"""
import logging
import mlflow
import yaml
from src.data.data_loader import DataLoader
from src.models.trainer import ModelTrainer

# Set up logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

def main():
    """Main training function."""
    logger.info("Starting AI/ML project training...")
    
    # Set up MLflow tracking
    mlflow.set_tracking_uri("sqlite:///mlruns.db")
    
    # Initialize components
    data_loader = DataLoader()
    trainer = ModelTrainer()
    
    # Load and preprocess data
    df = data_loader.load_train_data()
    df = data_loader.preprocess_data(df)
    
    # Prepare features
    X, y = trainer.prepare_features(df)
    
    # Split data
    test_size = trainer.config['training']['test_size']
    random_state = trainer.config['training']['random_state']
    X_train, X_test, y_train, y_test = train_test_split(
        X, y, test_size=test_size, random_state=random_state
    )
    
    logger.info(f"Training set size: {X_train.shape[0]}")
    logger.info(f"Test set size: {X_test.shape[0]}")
    
    # Train model
    trainer.train_model(X_train, y_train)
    
    # Evaluate model
    metrics = trainer.evaluate_model(X_test, y_test)
    logger.info(f"Test accuracy: {metrics['accuracy']:.4f}")
    
    # Save model
    trainer.save_model()
    
    logger.info("Training completed successfully!")

if __name__ == "__main__":
    main()
''')
    
    # Create a simple Jupyter notebook example
    with open(project_path / "notebooks" / "exploratory_analysis.ipynb", "w") as f:
        f.write('''{
 "cells": [
  {
   "cell_type": "markdown",
   "metadata": {},
   "source": [
    "# Exploratory Analysis\\\\n\\\\nThis notebook explores the dataset for the ''' + project_name + ''' project."
   ]
  },
  {
   "cell_type": "code",
   "execution_count": null,
   "metadata": {},
   "outputs": [],
   "source": [
    "# Import required libraries\\\\nimport pandas as pd\\\\nimport numpy as np\\\\nimport matplotlib.pyplot as plt\\\\nimport seaborn as sns\\\\nfrom pathlib import Path\\\\nimport warnings\\\\nwarnings.filterwarnings(\'ignore\')\\\\n\\\\n# Set style\\\\nplt.style.use(\'seaborn-v0_8\')\\\\nsns.set_palette(\"husl\")"
   ]
  },
  {
   "cell_type": "code",
   "execution_count": null,
   "metadata": {},
   "outputs": [],
   "source": [
    "# Load data\\\\ndf = pd.read_csv(\"../data/raw/train.csv\")  # Adjust path as needed\\\\nprint(f\"Dataset shape: {df.shape}\")\\\\ndf.head()"
   ]
  },
  {
   "cell_type": "code",
   "execution_count": null,
   "metadata": {},
   "outputs": [],
   "source": [
    "# Basic statistics\\\\ndf.describe()"
   ]
  },
  {
   "cell_type": "code",
   "execution_count": null,
   "metadata": {},
   "outputs": [],
   "source": [
    "# Missing values heatmap\\\\nplt.figure(figsize=(12, 6))\\\\nsns.heatmap(df.isnull(), yticklabels=False, cbar=True, cmap=\'viridis\')\\\\nplt.title(\'Missing Values Heatmap\')\\\\nplt.show()"
   ]
  },
  {
   "cell_type": "code",
   "execution_count": null,
   "metadata": {},
   "outputs": [],
   "source": [
    "# Correlation matrix\\\\ncorr_matrix = df.select_dtypes(include=[np.number]).corr()\\\\n\\\\nplt.figure(figsize=(12, 10))\\\\nsns.heatmap(corr_matrix, annot=True, cmap=\'coolwarm\', center=0)\\\\nplt.title(\'Correlation Matrix\')\\\\nplt.show()"
   ]
  },
  {
   "cell_type": "code",
   "execution_count": null,
   "metadata": {},
   "outputs": [],
   "source": [
    "# Distribution of numerical features\\\\nnumerical_cols = df.select_dtypes(include=[np.number]).columns\\\\n\\\\nn_cols = len(numerical_cols)\\\\nfig, axes = plt.subplots(n_cols//3 + 1, 3, figsize=(15, 5*(n_cols//3 + 1)))\\\\naxes = axes.flatten() if n_cols > 1 else [axes]\\\\n\\\\nfor i, col in enumerate(numerical_cols):\\\\n    axes[i].hist(df[col].dropna(), bins=30, edgecolor=\'black\')\\\\n    axes[i].set_title(f\'Distribution of {col}\')\\\\n    axes[i].set_xlabel(col)\\\\n    axes[i].set_ylabel(\'Frequency\')\\\\n\\\\n# Hide unused subplots\\\\nfor i in range(len(numerical_cols), len(axes)):\\\\n    axes[i].set_visible(False)\\\\n\\\\nplt.tight_layout()\\\\nplt.show()"
   ]
  },
  {
   "cell_type": "code",
   "execution_count": null,
   "metadata": {},
   "outputs": [],
   "source": [
    "# Save processed data\\\\n# Uncomment the following lines when ready to save\\\\n# df.to_csv(\"../data/processed/train_processed.csv\", index=False)\\\\n# print(\"Processed data saved to ../data/processed/train_processed.csv\")"
   ]
  }
 ],
 "metadata": {
  "kernelspec": {
   "display_name": "Python 3",
   "language": "python",
   "name": "python3"
  },
  "language_info": {
   "codemirror_mode": {
    "name": "ipython",
    "version": 3
   },
   "file_extension": ".py",
   "mimetype": "text/x-python",
   "name": "python",
   "nbconvert_exporter": "python",
   "pygments_lexer": "ipython3",
   "version": "3.9.0"
  }
 },
 "nbformat": 4,
 "nbformat_minor": 4
}
''')
    
    # Create .gitignore
    with open(project_path / ".gitignore", "w") as f:
        f.write('''# Python
__pycache__/
*.py[cod]
*$py.class
*.so
.Python
build/
develop-eggs/
dist/
downloads/
eggs/
.eggs/
lib/
lib64/
parts/
sdist/
var/
wheels/
*.egg-info/
.installed.cfg
*.egg

# Virtual environments
venv/
env/
ENV/

# Jupyter
.ipynb_checkpoints
*/.ipynb_checkpoints/*

# Data
data/processed/*
!data/processed/.gitkeep
data/interim/*
!data/interim/.gitkeep
data/external/*
!data/external/.gitkeep

# Models
models/*
!models/.gitkeep

# MLflow
mlruns/

# IDE
.vscode/
.idea/
*.swp
*.swo
*~

# OS
.DS_Store
Thumbs.db
''')
    
    # Create README.md
    with open(project_path / "README.md", "w") as f:
        f.write(f'''# {project_name} - AI/ML Project

An AI/ML project template generated with Codeius AI.

## Project Structure

```
{project_name}/
├── data/
│   ├── raw/           # Raw data files
│   ├── processed/     # Processed/cleaned data
│   ├── interim/       # Intermediate data
│   └── external/      # External data sources
├── notebooks/         # Jupyter notebooks for exploration
├── src/               # Source code for project
│   ├── data/          # Data loading and preprocessing
│   ├── features/      # Feature engineering
│   ├── models/        # Model definitions and training
│   ├── visualization/ # Visualization utilities
│   └── utils/         # Common utilities
├── models/            # Saved model files
├── reports/           # Generated analysis reports
├── tests/             # Unit tests
├── config.yaml        # Configuration file
├── requirements.txt   # Python dependencies
└── train.py           # Main training script
```

## Setup

1. Install dependencies:
```bash
pip install -r requirements.txt
```

2. Set up your data in the `data/` directory

3. Update `config.yaml` with your specific parameters

4. Run the training script:
```bash
python train.py
```

## Features

- Data loading and preprocessing pipeline
- Model training with experiment tracking (MLflow)
- Hyperparameter tuning capabilities
- Cross-validation and evaluation metrics
- Jupyter notebooks for exploratory analysis
- Configuration management with YAML
- Model serialization and loading
- Clean project structure following best practices

## Experiment Tracking

The project uses MLflow for experiment tracking. View your experiments with:

```bash
mlflow ui
```
''')
    
    # Create a simple test file
    with open(project_path / "tests" / "test_data_loader.py", "w") as f:
        f.write('''"""
Unit tests for data loader module
"""
import unittest
import pandas as pd
import tempfile
import os
from pathlib import Path
from src.data.data_loader import DataLoader

class TestDataLoader(unittest.TestCase):
    """Test cases for DataLoader class."""
    
    def setUp(self):
        """Set up test fixtures."""
        # Create a temporary config file
        self.temp_config = tempfile.NamedTemporaryFile(mode='w', delete=False, suffix='.yaml')
        self.temp_config.write("""
data:
  raw_dir: "data/raw"
  processed_dir: "data/processed"
  train_file: "train.csv"
  test_file: "test.csv"
  target_column: "target"

model:
  name: "RandomForestClassifier"
  hyperparameters:
    n_estimators: 100
    max_depth: 10
    random_state: 42
  save_path: "models/"
""")
        self.temp_config.close()
        
        # Create temporary data directory and file
        self.temp_dir = tempfile.mkdtemp()
        self.temp_data_dir = Path(self.temp_dir) / "data" / "raw"
        self.temp_data_dir.mkdir(parents=True)
        
        # Create a dummy CSV file for testing
        self.temp_csv = self.temp_data_dir / "train.csv"
        test_data = pd.DataFrame({
            "feature1": [1, 2, 3, 4, 5],
            "feature2": [10, 20, 30, 40, 50],
            "target": [0, 1, 0, 1, 0]
        })
        test_data.to_csv(self.temp_csv, index=False)
    
    def tearDown(self):
        """Clean up test fixtures."""
        import shutil
        shutil.rmtree(Path(self.temp_dir))
        os.unlink(self.temp_config.name)
    
    def test_load_train_data(self):
        """Test loading training data."""
        # Temporarily modify config to use our test directory
        original_config = "config.yaml"
        os.rename(original_config, f"{original_config}.bak")
        
        # Create a config with test paths
        test_config = {
            "data": {
                "raw_dir": str(self.temp_data_dir),
                "train_file": "train.csv",
                "target_column": "target"
            }
        }
        
        with open("config.yaml", "w") as f:
            import yaml
            yaml.dump(test_config, f)
        
        try:
            loader = DataLoader("config.yaml")
            df = loader.load_train_data()
            
            self.assertIsInstance(df, pd.DataFrame)
            self.assertEqual(df.shape[0], 5)
            self.assertIn("feature1", df.columns)
        finally:
            # Restore original config
            os.rename(f"{original_config}.bak", original_config)

if __name__ == '__main__':
    unittest.main()
''')
    
    print(f"AI/ML project '{project_name}' created successfully!")