# src/apathetic_utils/text.py
"""Text manipulation utilities."""

from __future__ import annotations

import re
from pathlib import Path
from typing import Any


class ApatheticUtils_Internal_Text:  # noqa: N801  # pyright: ignore[reportUnusedClass]
    """Mixin class that provides text manipulation functionality.

    This class contains utilities for text processing and formatting.
    When mixed into apathetic_utils, it provides text manipulation methods.
    """

    @staticmethod
    def plural(obj: Any) -> str:
        """Return 's' if obj represents a plural count.

        Accepts ints, floats, and any object implementing __len__().
        Returns '' for singular or zero.
        """
        count: int | float
        try:
            count = len(obj)
        except TypeError:
            # fallback for numbers or uncountable types
            count = obj if isinstance(obj, (int, float)) else 0
        return "s" if count != 1 else ""

    @staticmethod
    def remove_path_in_error_message(inner_msg: str, path: Path) -> str:
        """Remove redundant file path mentions (and nearby filler)
        from error messages.

        Useful when wrapping a lower-level exception that already
        embeds its own file reference, so the higher-level message
        can use its own path without duplication.

        Example:
            "Invalid JSONC syntax in /abs/path/config.jsonc: Expecting value"
            → "Invalid JSONC syntax: Expecting value"

        """
        # Normalize both path and name for flexible matching
        full_path = str(path)
        filename = path.name

        # Common redundant phrases we might need to remove
        candidates = [
            f"in {full_path}",
            f"in '{full_path}'",
            f'in "{full_path}"',
            f"in {filename}",
            f"in '{filename}'",
            f'in "{filename}"',
            full_path,
            filename,
        ]

        clean_msg = inner_msg
        for pattern in candidates:
            clean_msg = clean_msg.replace(pattern, "").strip(": ").strip()

        # Normalize leftover spaces and colons
        clean_msg = re.sub(r"\s{2,}", " ", clean_msg)
        clean_msg = re.sub(r"\s*:\s*", ": ", clean_msg)

        return clean_msg
