# src/apathetic_utils/paths.py
"""Path manipulation utilities."""

from __future__ import annotations

import re
from itertools import zip_longest
from pathlib import Path

from apathetic_logging import getLogger


class ApatheticUtils_Internal_Paths:  # noqa: N801  # pyright: ignore[reportUnusedClass]
    """Mixin class that provides path manipulation functionality.

    This class contains utilities for path normalization and glob handling.
    When mixed into apathetic_utils, it provides path manipulation methods.
    """

    @staticmethod
    def normalize_path_string(raw: str) -> str:
        r"""Normalize a user-supplied path string for cross-platform use.

        Industry-standard (Git/Node/Python) rules:
          - Treat both '/' and '\\' as valid separators and normalize all to '/'.
          - Replace escaped spaces ('\\ ') with real spaces.
          - Collapse redundant slashes (preserve protocol prefixes like 'file://').
          - Never resolve '.' or '..' or touch the filesystem.
          - Never raise for syntax; normalization is always possible.

        This is the pragmatic cross-platform normalization strategy used by
        Git, Node.js, and Python build tools.
        This function is purely lexical — it normalizes syntax, not filesystem state.
        """
        logger = getLogger()
        if not raw:
            return ""

        path = raw.strip()

        # Handle escaped spaces (common shell copy-paste)
        if "\\ " in path:
            fixed = path.replace("\\ ", " ")
            logger.warning("Normalizing escaped spaces in path: %r → %s", path, fixed)
            path = fixed

        # Normalize all backslashes to forward slashes
        path = path.replace("\\", "/")

        # Collapse redundant slashes (keep protocol //)
        collapsed_slashes = re.sub(r"(?<!:)//+", "/", path)
        if collapsed_slashes != path:
            logger.trace(
                "Collapsed redundant slashes: %r → %r", path, collapsed_slashes
            )
            path = collapsed_slashes

        return path

    @staticmethod
    def has_glob_chars(s: str) -> bool:
        return any(c in s for c in "*?[]")

    @staticmethod
    def get_glob_root(pattern: str) -> Path:
        """Return the non-glob portion of a path like 'src/**/*.txt'.

        Normalizes paths to cross-platform.
        """
        if not pattern:
            return Path()

        # Normalize backslashes to forward slashes
        normalized = ApatheticUtils_Internal_Paths.normalize_path_string(pattern)

        parts: list[str] = []
        for part in Path(normalized).parts:
            if re.search(r"[*?\[\]]", part):
                break
            parts.append(part)
        return Path(*parts) if parts else Path()

    @staticmethod
    def strip_common_prefix(path: str | Path, base: str | Path) -> str:
        """Return `path` relative to `base`'s common prefix.

        Example:
            strip_common_prefix(
                "/home/user/code/serger/src/serger/logs.py",
                "/home/user/code/serger/tests/utils/patch_everywhere.py"
            )
            → "src/serger/logs.py"

        Args:
            path: Path to make relative
            base: Base path to find common prefix with

        Returns:
            Path relative to common prefix, or "." if no common prefix
        """
        p = Path(path).resolve()
        b = Path(base).resolve()

        # Split both into parts and find the longest shared prefix
        common_len = 0
        for a, c in zip_longest(p.parts, b.parts):
            if a != c:
                break
            common_len += 1

        # Slice off the shared prefix
        remaining = Path(*p.parts[common_len:])
        return str(remaining) or "."
