"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
/* eslint-disable quote-props */
// CDK parameters
const CONSTRUCT_NAME = 'aws-cdk-dynamodb-global';
// DynamoDB table parameters
const TABLE_NAME = 'GlobalTable';
const TABLE_PARTITION_KEY = { name: 'hashKey', type: aws_dynamodb_1.AttributeType.STRING };
const STACK_PROPS = {
    partitionKey: TABLE_PARTITION_KEY,
    tableName: TABLE_NAME,
    regions: ['us-east-1', 'us-east-2', 'us-west-2'],
};
module.exports = {
    'Default Global DynamoDB stack': {
        'global dynamo'(test) {
            const stack = new core_1.Stack();
            new lib_1.GlobalTable(stack, CONSTRUCT_NAME, STACK_PROPS);
            const topStack = stack.node.findChild(CONSTRUCT_NAME);
            for (const reg of STACK_PROPS.regions) {
                const tableStack = topStack.node.findChild(CONSTRUCT_NAME + '-' + reg);
                assert_1.expect(tableStack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                    'KeySchema': [
                        {
                            'AttributeName': 'hashKey',
                            'KeyType': 'HASH',
                        },
                    ],
                    'AttributeDefinitions': [
                        {
                            'AttributeName': 'hashKey',
                            'AttributeType': 'S',
                        },
                    ],
                    'StreamSpecification': {
                        'StreamViewType': 'NEW_AND_OLD_IMAGES',
                    },
                    'TableName': 'GlobalTable',
                }));
            }
            const customResourceStack = stack.node.findChild(CONSTRUCT_NAME + '-CustomResource');
            assert_1.expect(customResourceStack).to(assert_1.haveResource('AWS::Lambda::Function', {
                Description: 'Lambda to make DynamoDB a global table',
                Handler: 'index.handler',
                Timeout: 300,
            }));
            assert_1.expect(customResourceStack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
                Regions: STACK_PROPS.regions,
                ResourceType: 'Custom::DynamoGlobalTableCoordinator',
                TableName: TABLE_NAME,
            }));
            test.done();
        },
    },
    'GlobalTable generated stacks inherit their account from the parent stack'(test) {
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'GlobalTableStack', { env: { account: '123456789012', region: 'us-east-1' } });
        const globalTable = new lib_1.GlobalTable(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            partitionKey: TABLE_PARTITION_KEY,
            regions: ['us-east-1', 'us-west-2'],
            stream: aws_dynamodb_1.StreamViewType.NEW_AND_OLD_IMAGES,
        });
        new core_1.CfnOutput(stack, 'DynamoDbOutput', {
            // this works, because both `stack` and `regionTables[0]` stack are in the same account & region
            value: globalTable.regionalTables[0].tableStreamArn,
        });
        assert_1.expect(stack).toMatch({
            'Outputs': {
                'DynamoDbOutput': {
                    'Value': {
                        'Fn::ImportValue': 'GlobalTableStackawscdkdynamodbglobalawscdkdynamodbglobaluseast19C1C8A14:awscdkdynamodbglobalawscdkdynamodbglobaluseast1ExportsOutputFnGetAttawscdkdynamodbglobalGlobalTableuseast1FC03DD69StreamArn28E90DB8',
                    },
                },
            },
        });
        test.done();
    },
    'Enforce StreamSpecification': {
        'global dynamo should only allow NEW_AND_OLD_IMAGES'(test) {
            const stack = new core_1.Stack();
            test.throws(() => {
                new lib_1.GlobalTable(stack, CONSTRUCT_NAME, {
                    tableName: TABLE_NAME,
                    stream: aws_dynamodb_1.StreamViewType.KEYS_ONLY,
                    partitionKey: TABLE_PARTITION_KEY,
                    regions: ['us-east-1', 'us-east-2', 'us-west-2'],
                });
            }, /dynamoProps.stream MUST be set to dynamodb.StreamViewType.NEW_AND_OLD_IMAGES/);
            test.done();
        },
    },
    'Check getting tables': {
        'global dynamo should only allow NEW_AND_OLD_IMAGES'(test) {
            const stack = new core_1.Stack();
            const regTables = new lib_1.GlobalTable(stack, CONSTRUCT_NAME, {
                tableName: TABLE_NAME,
                partitionKey: TABLE_PARTITION_KEY,
                regions: ['us-east-1', 'us-east-2', 'us-west-2'],
            });
            test.equal(regTables.regionalTables.length, 3);
            for (const table of regTables.regionalTables) {
                test.ok(table instanceof aws_dynamodb_1.Table);
            }
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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