"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const glue = require("@aws-cdk/aws-glue");
const iam = require("@aws-cdk/aws-iam");
const assets = require("@aws-cdk/aws-s3-assets");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const start_job_run_1 = require("../../lib/glue/start-job-run");
/*
 * Stack verification steps:
 * * aws stepfunctions start-execution --state-machine-arn <deployed state machine arn>
 * * aws stepfunctions describe-execution --execution-arn <execution arn created above>
 * The "describe-execution" call should eventually return status "SUCCEEDED".
 * NOTE: It will take up to 15 minutes for the step function to complete due to the cold start time
 * for AWS Glue, which as of 02/2020, is around 10-15 minutes.
 */
const app = new cdk.App();
const stack = new cdk.Stack(app, 'aws-stepfunctions-integ');
const codeAsset = new assets.Asset(stack, 'Glue Job Script', {
    path: path.join(__dirname, 'my-glue-script/job.py'),
});
const jobRole = new iam.Role(stack, 'Glue Job Role', {
    assumedBy: new iam.ServicePrincipal('glue'),
    managedPolicies: [
        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole'),
    ],
});
codeAsset.grantRead(jobRole);
const job = new glue.CfnJob(stack, 'Glue Job', {
    name: 'My Glue Job',
    glueVersion: '1.0',
    command: {
        name: 'glueetl',
        pythonVersion: '3',
        scriptLocation: `s3://${codeAsset.s3BucketName}/${codeAsset.s3ObjectKey}`,
    },
    role: jobRole.roleArn,
});
const jobTask = new start_job_run_1.GlueStartJobRun(stack, 'Glue Job Task', {
    glueJobName: job.name,
    integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    arguments: sfn.TaskInput.fromObject({
        '--enable-metrics': 'true',
    }),
});
const startTask = new sfn.Pass(stack, 'Start Task');
const endTask = new sfn.Pass(stack, 'End Task');
const stateMachine = new sfn.StateMachine(stack, 'State Machine', {
    definition: sfn.Chain.start(startTask).next(jobTask).next(endTask),
});
new cdk.CfnOutput(stack, 'State Machine ARN Output', {
    value: stateMachine.stateMachineArn,
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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