import copy
import json
from functools import lru_cache
from logging import Logger
from typing import List, Type, cast, Dict

from log_with_context import add_logging_context
from presidio_anonymizer import AnonymizerEngine
from presidio_anonymizer.entities import RecognizerResult, OperatorConfig
from pydantic import Field, BaseModel
from pymultirole_plugins.v1.processor import ProcessorParameters, ProcessorBase
from pymultirole_plugins.v1.schema import Document, AltText

from .fakerop import FakerOp

logger = Logger("pymultirole")

OPERATOR_MASK = {
    "type": "mask",
    "masking_char": "*",
    "chars_to_mask": 12,
    "from_end": True,
}
OPERATOR_MASK_STR = json.dumps(OPERATOR_MASK, indent=2)

OPERATOR_REPLACE = {
    "type": "replace",
    "new_value": "_ANONYMIZED_"
}
OPERATOR_REPLACE_STR = json.dumps(OPERATOR_REPLACE, indent=2)

OPERATOR_REDACT = {
    "type": "redact"
}
OPERATOR_REDACT_STR = json.dumps(OPERATOR_REDACT, indent=2)

OPERATOR_LABEL = {
    "type": "label"
}
OPERATOR_LABEL_STR = json.dumps(OPERATOR_LABEL, indent=2)

OPERATOR_IDENTITY = {
    "type": "identity"
}
OPERATOR_IDENTITY_STR = json.dumps(OPERATOR_IDENTITY, indent=2)

CUSTOM_OPERATORS = {
    "identity": lambda x: x,
}

OPERATOR_FAKER = {
    "type": "faker",
    "provider": "name",
    "locale": "fr_FR"
}
OPERATOR_FAKER_STR = json.dumps(OPERATOR_FAKER, indent=2)


class PseudonimizerParameters(ProcessorParameters):
    mapping: Dict[str, str] = Field(None,
                                    description="""List of anonymization/pseudonymization [operators](https://microsoft.github.io/presidio/tutorial/10_simple_anonymization/) like this ones:<br/>"""
                                                + "<li>mask: replaces the annotation with a sequence of a given character. ```" + OPERATOR_MASK_STR + "```"
                                                + "<li>replace: replaces the annotation with desired value. ```" + OPERATOR_REPLACE_STR + "```"
                                                + "<li>redact: redact: removes the annotation completely from text. ```" + OPERATOR_REDACT_STR + "```"
                                                + "<li>identity: leaves the annotation as it is. ```" + OPERATOR_IDENTITY_STR + "```"
                                                + "<li>label: replaces the annotation with its label. ```" + OPERATOR_LABEL_STR + "```"
                                                + "<li>faker: replaces the annotation by a fake value generated by a [faker provider](https://faker.readthedocs.io/en/master/providers.html). ```" + OPERATOR_FAKER_STR + "```",
                                    extra="key:label,val:json")
    default_operator: str = Field(OPERATOR_LABEL_STR,
                                  description="""Default anonymization/pseudonymization operator to use if no explicit mapping is provided""",
                                  extra="json")
    as_altText: str = Field(
        "pseudonimization",
        description="""If defined generate the pseudonimization as an alternative text of the input document,
    if not replace the text of the input document.""",
    )


def operator_from_annotation(a, definitions):
    default_def = definitions.get(None, None)
    label = a.labelName or a.label
    if label in definitions:
        definition = definitions[label]
        return operator_from_definition(a, definition)
    elif default_def is not None:
        return operator_from_definition(a, default_def)
    return None


def operator_from_definition(a, definition):
    defi = copy.deepcopy(definition)
    if 'type' in defi:
        type = defi['type']
        if type == 'label':
            replacement = f"<{a.label or a.labelName}>"
            return OperatorConfig("replace",
                                  {"new_value": replacement})
        elif type == "faker":
            defi.pop("type")
            provider = defi.pop("provider")
            fakerop = FakerOp(provider, **defi)
            return OperatorConfig("custom", {"lambda": fakerop.evaluate})
        elif type in CUSTOM_OPERATORS:
            return OperatorConfig("custom", {"lambda": CUSTOM_OPERATORS[type]})
        else:
            return OperatorConfig.from_json(defi)
    return None


class PseudonimizerProcessor(ProcessorBase):
    __doc__ = """Pseudonimizer based on [Presidio](https://microsoft.github.io/presidio/anonymizer/)."""

    def process(
            self, documents: List[Document], parameters: ProcessorParameters
    ) -> List[Document]:
        params: PseudonimizerParameters = cast(PseudonimizerParameters, parameters)
        mapping = frozenset(params.mapping.items()) if params.mapping else None
        definitions = get_definitions(mapping, params.default_operator)
        # logger.warning(f"Anonymizing with: {str(definitions)}")
        engine = get_engine()
        operators: Dict[str, OperatorConfig] = {}
        try:
            for document in documents:
                with add_logging_context(docid=document.identifier):
                    result_text = document.text
                    if document.annotations:
                        analyzer_results = []
                        for a in document.annotations:
                            if a.status != "KO":
                                analyzer_results.append(
                                    RecognizerResult(entity_type=a.labelName, start=a.start, end=a.end, score=a.score or 1.0))
                                if a.labelName not in operators:
                                    op = operator_from_annotation(a, definitions)
                                    if op is not None:
                                        operators[a.labelName] = op
                        result = engine.anonymize(
                            text=document.text,
                            analyzer_results=analyzer_results,
                            operators=operators,
                        )
                        result_text = result.text
                    if params.as_altText is not None and len(params.as_altText.strip()):
                        as_altText = params.as_altText.strip()
                        document.altTexts = document.altTexts or []
                        altTexts = [
                            alt
                            for alt in document.altTexts
                            if alt.name != as_altText
                        ]
                        altTexts.append(AltText(name=as_altText, text=result_text))
                        document.altTexts = altTexts
                    elif document.text != result_text:
                        document.text = result_text
                        document.annotations = None
                        document.sentences = None
        except BaseException as err:
            raise err
        return documents

    @classmethod
    def get_model(cls) -> Type[BaseModel]:
        return PseudonimizerParameters


@lru_cache(maxsize=None)
def get_engine():
    return AnonymizerEngine()


@lru_cache(maxsize=None)
def get_definitions(mapping_items, default):
    definitions = {None: None}
    if default is not None and default.strip() != "":
        default_def = json.loads(default)
        definitions[None] = default_def
    if mapping_items:
        for pname, pvalue in mapping_items:
            if pname and pvalue is not None and pvalue.strip() != "":
                op_def = json.loads(pvalue)
                definitions[pname] = op_def
    return definitions
