import contextlib
from typing import AsyncIterator

import uvicorn
from fastapi import FastAPI

from algo_flow.app.main import make_app
from algo_flow.cores.config import settings
from algo_flow.cores.log import LOG
from algo_flow.cores.nacos_client import deregister_instance, heartbeat, register_instance


@contextlib.asynccontextmanager
async def lifespan(_app: FastAPI) -> AsyncIterator[None]:
    LOG.info("Starting application lifespan...")

    # nacos注册服务
    try:
        register_instance()
        heartbeat()
        LOG.info("已注册到Nacos服务中心，并启动心跳线程")
    except Exception as e:
        LOG.exception(e)
        LOG.error(f"注册到Nacos失败: {e}")

    # 通过 yield 将控制权交给 FastAPI
    yield

    try:
        deregister_instance()
        LOG.info("已从Nacos注销服务")
    except Exception as e:
        LOG.error(f"从Nacos注销失败: {e}")


if __name__ == "__main__":
    app = make_app(lifespan_func=lifespan)

    try:
        uvicorn.run(
            app,
            host=settings.app.host,
            port=settings.app.port,
            reload=settings.app.debug,
            log_config=None,
        )
    except KeyboardInterrupt:
        LOG.info("Shutdown requested")
