from typing import AsyncGenerator

import pytest
from fastapi import FastAPI
from httpx import AsyncClient
from tortoise import Tortoise

from algo_flow.app.system.models import Permission, Role, User
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.app.system.views.user import user_router
from algo_flow.cores.pwd import get_password_hash

# 测试数据
test_admin_data = {
    "username": "admin",
    "email": "admin@example.com",
    "password": "adminpass123",
}

test_user_data = {
    "username": "testuser",
    "email": "test@example.com",
    "password": "testpass123",
    "avatar": "https://example.com/avatar.jpg",
    "is_active": True,
    "is_superuser": False,
}

test_role_data = {
    "name": "测试角色",
    "identifier": "test_role",
    "description": "这是一个测试角色",
}

test_permission_data = {
    "name": "system:role:create",
    "identifier": "system:role:create",
    "module": "api",
    "description": "Create role permission",
}


async def mock_get_current_active_user():
    """模拟认证用户"""
    return await User.get(username=test_admin_data["username"])


@pytest.fixture
def app() -> FastAPI:
    """创建测试应用"""
    app = FastAPI()
    app.include_router(user_router, prefix="/users")
    app.dependency_overrides[get_current_active_user] = mock_get_current_active_user
    return app


@pytest.fixture(autouse=True)
async def setup_database() -> AsyncGenerator:
    """设置测试数据库"""
    await Tortoise.init(
        db_url="sqlite://:memory:",
        modules={"models": ["app.system.models"]},
    )
    await Tortoise.generate_schemas()

    # 创建测试权限和角色
    permission = await Permission.create(**test_permission_data)
    role = await Role.create(**test_role_data)
    await role.permissions.add(permission)

    # 创建管理员用户
    admin = await User.create(
        username=test_admin_data["username"],
        email=test_admin_data["email"],
        hashed_password=get_password_hash(test_admin_data["password"]),
        is_superuser=True,
    )
    await admin.roles.add(role)

    yield

    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_create_user(app: FastAPI):
    """测试创建用户"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.post("/users", json=test_user_data)
        assert response.status_code == 201
        data = response.json()
        assert data["username"] == test_user_data["username"]
        assert data["email"] == test_user_data["email"]
        assert data["avatar"] == test_user_data["avatar"]
        assert data["is_active"] == test_user_data["is_active"]
        assert data["is_superuser"] == test_user_data["is_superuser"]


@pytest.mark.asyncio
async def test_create_user_with_duplicate_username(app: FastAPI):
    """测试创建重复用户名的用户"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个用户
        await client.post("/users", json=test_user_data)

        # 尝试创建同名用户
        response = await client.post("/users", json=test_user_data)
        assert response.status_code == 409


@pytest.mark.asyncio
async def test_get_user(app: FastAPI):
    """测试获取用户详情"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个用户
        create_response = await client.post("/users", json=test_user_data)
        user_id = create_response.json()["id"]

        # 获取用户详情
        response = await client.get(f"/users/{user_id}")
        assert response.status_code == 200
        data = response.json()
        assert data["id"] == user_id
        assert data["username"] == test_user_data["username"]
        assert data["email"] == test_user_data["email"]


@pytest.mark.asyncio
async def test_list_users(app: FastAPI):
    """测试获取用户列表"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 创建多个用户
        await client.post("/users", json=test_user_data)
        await client.post(
            "/users",
            json={
                **test_user_data,
                "username": "testuser2",
                "email": "test2@example.com",
            },
        )

        # 测试无过滤条件
        response = await client.get("/users")
        assert response.status_code == 200
        data = response.json()
        assert data["total"] == 3  # 包括管理员用户
        assert len(data["items"]) == 3

        # 测试关键字搜索
        response = await client.get("/users", params={"keyword": "testuser"})
        assert response.status_code == 200
        data = response.json()
        assert data["total"] == 2
        assert len(data["items"]) == 2

        # 测试分页
        response = await client.get("/users", params={"limit": 1})
        assert response.status_code == 200
        data = response.json()
        assert data["total"] == 3
        assert len(data["items"]) == 1


@pytest.mark.asyncio
async def test_update_user(app: FastAPI):
    """测试更新用户"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个用户
        create_response = await client.post("/users", json=test_user_data)
        user_id = create_response.json()["id"]

        # 更新用户信息
        update_data = {
            "avatar": "https://example.com/avatar1.jpg",
        }
        response = await client.patch(f"/users/{user_id}", json=update_data)
        assert response.status_code == 200
        data = response.json()
        assert data["id"] == user_id
        assert data["avatar"] == update_data["avatar"]


@pytest.mark.asyncio
async def test_delete_user(app: FastAPI):
    """测试删除用户"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个用户
        create_response = await client.post("/users", json=test_user_data)
        user_id = create_response.json()["id"]

        # 删除用户
        response = await client.delete(f"/users/{user_id}")
        assert response.status_code == 204

        # 确认用户已被删除
        response = await client.get(f"/users/{user_id}")
        assert response.status_code == 404
