from typing import AsyncGenerator

import pytest
from fastapi import FastAPI
from httpx import AsyncClient
from tortoise import Tortoise

from algo_flow.app.system.models import Role, User
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.app.system.views.permission import permission_router
from algo_flow.app.system.views.role import role_router
from algo_flow.cores.pwd import get_password_hash

# 测试数据
test_admin_data = {
    "username": "admin",
    "email": "admin@example.com",
    "password": "adminpass123",
}

test_role_data = {
    "name": "测试角色",
    "identifier": "test_role",
    "description": "这是一个测试角色",
}

test_permission_data = {
    "name": "system:role:create",
    "identifier": "system:role:create",
    "module": "api",
    "description": "Create role permission",
}


async def mock_get_current_active_user():
    """模拟认证用户"""
    return await User.get(username=test_admin_data["username"])


@pytest.fixture
def app() -> FastAPI:
    """创建测试应用"""
    app = FastAPI()
    app.include_router(role_router, prefix="/roles")
    app.include_router(permission_router, prefix="/permissions")
    app.dependency_overrides[get_current_active_user] = mock_get_current_active_user
    return app


@pytest.fixture(autouse=True)
async def setup_database() -> AsyncGenerator:
    """设置测试数据库"""
    await Tortoise.init(
        db_url="sqlite://:memory:",
        modules={"models": ["app.system.models"]},
    )
    await Tortoise.generate_schemas()

    # 创建管理员用户和角色
    admin_role = await Role.create(
        name="admin",
        identifier="admin",
        description="系统管理员角色",
        is_default=True,
    )

    admin = await User.create(
        username=test_admin_data["username"],
        email=test_admin_data["email"],
        hashed_password=get_password_hash(test_admin_data["password"]),
        is_superuser=True,
    )
    await admin.roles.add(admin_role)

    yield

    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_create_role(app: FastAPI):
    """测试创建角色"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.post("/roles", json=test_role_data)
        assert response.status_code == 201
        data = response.json()
        assert data["name"] == test_role_data["name"]
        assert data["identifier"] == test_role_data["identifier"]
        assert data["description"] == test_role_data["description"]


@pytest.mark.asyncio
async def test_create_role_with_duplicate_identifier(app: FastAPI):
    """测试创建重复标识符的角色"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个角色
        await client.post("/roles", json=test_role_data)

        # 尝试创建相同标识符的角色
        response = await client.post("/roles", json=test_role_data)
        assert response.status_code == 409


@pytest.mark.asyncio
async def test_get_role(app: FastAPI):
    """测试获取角色详情"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个角色
        create_response = await client.post("/roles", json=test_role_data)
        role_id = create_response.json()["id"]

        # 获取角色详情
        response = await client.get(f"/roles/{role_id}")
        assert response.status_code == 200
        data = response.json()
        assert data["id"] == role_id
        assert data["name"] == test_role_data["name"]
        assert data["identifier"] == test_role_data["identifier"]


@pytest.mark.asyncio
async def test_list_roles(app: FastAPI):
    """测试获取角色列表"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 创建多个角色
        await client.post("/roles", json=test_role_data)
        await client.post(
            "/roles",
            json={
                **test_role_data,
                "name": "测试角色2",
                "identifier": "test_role2",
            },
        )

        # 测试无过滤条件
        response = await client.get("/roles")
        assert response.status_code == 200
        data = response.json()
        assert data["total"] == 3  # 包括管理员角色
        assert len(data["items"]) == 3

        # 测试分页
        response = await client.get("/roles", params={"limit": 1})
        assert response.status_code == 200
        data = response.json()
        assert data["total"] == 3
        assert len(data["items"]) == 1


@pytest.mark.asyncio
async def test_update_role(app: FastAPI):
    """测试更新角色"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个角色
        create_response = await client.post("/roles", json=test_role_data)
        role_id = create_response.json()["id"]

        # 更新角色信息
        update_data = {
            "name": "更新后的角色",
            "description": "这是更新后的角色描述",
        }
        response = await client.patch(f"/roles/{role_id}", json=update_data)
        assert response.status_code == 200
        data = response.json()
        assert data["id"] == role_id
        assert data["name"] == update_data["name"]
        assert data["description"] == update_data["description"]


@pytest.mark.asyncio
async def test_delete_role(app: FastAPI):
    """测试删除角色"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个角色
        create_response = await client.post("/roles", json=test_role_data)
        role_id = create_response.json()["id"]

        # 删除角色
        response = await client.delete(f"/roles/{role_id}")
        assert response.status_code == 204

        # 确认角色已被删除
        response = await client.get(f"/roles/{role_id}")
        assert response.status_code == 404


@pytest.mark.asyncio
async def test_assign_role_permissions(app: FastAPI):
    """测试分配角色权限"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个权限
        permission_response = await client.post("/permissions", json=test_permission_data)
        assert permission_response.status_code == 201
        permission_id = permission_response.json()["id"]

        # 创建一个角色
        create_response = await client.post("/roles", json=test_role_data)
        role_id = create_response.json()["id"]

        # 分配权限
        permissions_data = {"permission_ids": [permission_id]}
        response = await client.post(f"/roles/{role_id}/permissions", json=permissions_data)
        assert response.status_code == 200
        data = response.json()
        assert len(data["permissions"]) == 1


@pytest.mark.asyncio
async def test_get_role_permissions(app: FastAPI):
    """测试获取角色权限"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个角色
        create_response = await client.post("/roles", json=test_role_data)
        role_id = create_response.json()["id"]

        # 获取角色权限
        response = await client.get(f"/roles/{role_id}/permissions")
        assert response.status_code == 200
        data = response.json()
        assert "permissions" in data
