from typing import AsyncGenerator

import pytest
from fastapi import FastAPI
from httpx import AsyncClient
from tortoise import Tortoise

from algo_flow.app.system.models import Permission, Role, User
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.app.system.views.permission import permission_router
from algo_flow.cores.pwd import get_password_hash

# 测试数据
test_admin_data = {
    "username": "admin",
    "email": "admin@example.com",
    "password": "adminpass123",
}

test_permission_data = {
    "name": "测试权限",
    "identifier": "test:permission",
    "module": "api",
    "description": "这是一个测试权限",
}

test_role_data = {
    "name": "admin",
    "code": "admin",
    "identifier": "system:admin",
    "description": "Admin role",
}


async def mock_get_current_active_user():
    """模拟认证用户"""
    return await User.get(username=test_admin_data["username"])


@pytest.fixture
def app() -> FastAPI:
    """创建测试应用"""
    app = FastAPI()
    app.include_router(permission_router, prefix="/permissions")
    app.dependency_overrides[get_current_active_user] = mock_get_current_active_user
    return app


@pytest.fixture(autouse=True)
async def setup_database() -> AsyncGenerator:
    """设置测试数据库"""
    await Tortoise.init(
        db_url="sqlite://:memory:",
        modules={"models": ["app.system.models"]},
    )
    await Tortoise.generate_schemas()

    # 创建管理员权限
    admin_permission = await Permission.create(
        name="系统权限管理",
        identifier="system:permission:create",
        module="api",
        description="系统权限管理",
    )

    # 创建管理员角色
    admin_role = await Role.create(**test_role_data)
    await admin_role.permissions.add(admin_permission)

    # 创建管理员用户
    admin = await User.create(
        username=test_admin_data["username"],
        email=test_admin_data["email"],
        hashed_password=get_password_hash(test_admin_data["password"]),
        is_superuser=True,
    )
    await admin.roles.add(admin_role)

    yield

    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_create_permission(app: FastAPI):
    """测试创建权限"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.post("/permissions", json=test_permission_data)
        assert response.status_code == 201
        data = response.json()
        assert data["name"] == test_permission_data["name"]
        assert data["identifier"] == test_permission_data["identifier"]
        assert data["module"] == test_permission_data["module"]
        assert data["description"] == test_permission_data["description"]


@pytest.mark.asyncio
async def test_create_permission_with_duplicate_identifier(app: FastAPI):
    """测试创建重复标识符的权限"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个权限
        await client.post("/permissions", json=test_permission_data)

        # 尝试创建相同标识符的权限
        response = await client.post("/permissions", json=test_permission_data)
        assert response.status_code == 409


@pytest.mark.asyncio
async def test_get_permission(app: FastAPI):
    """测试获取权限详情"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个权限
        create_response = await client.post("/permissions", json=test_permission_data)
        permission_id = create_response.json()["id"]

        # 获取权限详情
        response = await client.get(f"/permissions/{permission_id}")
        assert response.status_code == 200
        data = response.json()
        assert data["id"] == permission_id
        assert data["name"] == test_permission_data["name"]
        assert data["identifier"] == test_permission_data["identifier"]
        assert data["module"] == test_permission_data["module"]


@pytest.mark.asyncio
async def test_list_permissions(app: FastAPI):
    """测试获取权限列表"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 创建多个权限
        await client.post("/permissions", json=test_permission_data)
        await client.post(
            "/permissions",
            json={
                **test_permission_data,
                "name": "测试权限2",
                "identifier": "test:permission2",
            },
        )

        # 测试无过滤条件
        response = await client.get("/permissions")
        assert response.status_code == 200
        data = response.json()
        assert data["total"] == 3  # 包括管理员权限
        assert len(data["items"]) == 3

        # 测试按模块过滤
        response = await client.get("/permissions", params={"module": "api"})
        assert response.status_code == 200
        data = response.json()
        assert data["total"] == 3  # 所有权限都是 api 类型
        assert len(data["items"]) == 3

        # 测试分页
        response = await client.get("/permissions", params={"limit": 1})
        assert response.status_code == 200
        data = response.json()
        assert data["total"] == 3
        assert len(data["items"]) == 1


@pytest.mark.asyncio
async def test_update_permission(app: FastAPI):
    """测试更新权限"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个权限
        create_response = await client.post("/permissions", json=test_permission_data)
        permission_id = create_response.json()["id"]

        # 更新权限信息
        update_data = {
            "name": "更新后的权限",
            "description": "这是更新后的权限描述",
        }
        response = await client.patch(f"/permissions/{permission_id}", json=update_data)
        assert response.status_code == 200
        data = response.json()
        assert data["id"] == permission_id
        assert data["name"] == update_data["name"]
        assert data["description"] == update_data["description"]


@pytest.mark.asyncio
async def test_delete_permission(app: FastAPI):
    """测试删除权限"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个权限
        create_response = await client.post("/permissions", json=test_permission_data)
        permission_id = create_response.json()["id"]

        # 删除权限
        response = await client.delete(f"/permissions/{permission_id}")
        assert response.status_code == 204

        # 确认权限已被删除
        response = await client.get(f"/permissions/{permission_id}")
        assert response.status_code == 404
