from typing import AsyncGenerator

import pytest
from fastapi import FastAPI
from httpx import AsyncClient
from tortoise import Tortoise

from algo_flow.app.system.models import Permission, Role, User
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.app.system.views.menu import menu_router
from algo_flow.cores.pwd import get_password_hash

# 测试数据
test_admin_data = {
    "username": "admin",
    "email": "admin@example.com",
    "password": "adminpass123",
}

test_menu_data = {
    "name": "系统管理",
    "path": "/system",
    "component": "Layout",
    "order": 1,
    "icon": "setting",
    "hide_in_menu": False,
    "ignore_cache": False,
}

test_submenu_data = {
    "name": "菜单管理",
    "path": "menu",
    "component": "system/menu/index",
    "order": 1,
    "icon": "menu",
    "parent_id": None,  # 将在测试中设置
    "hide_in_menu": False,
    "ignore_cache": False,
}

test_role_data = {
    "name": "admin",
    "identifier": "admin",
    "description": "Admin role",
}

test_permission_data = {
    "name": "菜单管理",
    "identifier": "system:menu:create",
    "module": "system",
}


async def mock_get_current_active_user():
    """模拟认证用户"""
    return await User.get(username=test_admin_data["username"])


@pytest.fixture
def app() -> FastAPI:
    """创建测试应用"""
    app = FastAPI()
    app.include_router(menu_router, prefix="/menus")
    app.dependency_overrides[get_current_active_user] = mock_get_current_active_user
    return app


@pytest.fixture(autouse=True)
async def setup_database() -> AsyncGenerator:
    """设置测试数据库"""
    await Tortoise.init(
        db_url="sqlite://:memory:",
        modules={"models": ["app.system.models"]},
    )
    await Tortoise.generate_schemas()

    # 创建管理员权限
    admin_permission = await Permission.create(**test_permission_data)

    # 创建管理员角色
    admin_role = await Role.create(**test_role_data)
    await admin_role.permissions.add(admin_permission)

    # 创建管理员用户
    admin = await User.create(
        username=test_admin_data["username"],
        email=test_admin_data["email"],
        hashed_password=get_password_hash(test_admin_data["password"]),
        is_superuser=True,
    )
    await admin.roles.add(admin_role)

    yield

    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_create_menu(app: FastAPI):
    """测试创建菜单"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.post("/menus", json=test_menu_data)
        assert response.status_code == 201
        data = response.json()
        assert data["name"] == test_menu_data["name"]
        assert data["path"] == test_menu_data["path"]
        assert data["component"] == test_menu_data["component"]
        assert data["order"] == test_menu_data["order"]
        assert data["icon"] == test_menu_data["icon"]
        assert data["hide_in_menu"] == test_menu_data["hide_in_menu"]
        assert data["ignore_cache"] == test_menu_data["ignore_cache"]


@pytest.mark.asyncio
async def test_create_submenu(app: FastAPI):
    """测试创建子菜单"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建父菜单
        parent_response = await client.post("/menus", json=test_menu_data)
        parent_id = parent_response.json()["id"]

        # 创建子菜单
        submenu_data = {**test_submenu_data, "parent_id": parent_id}
        response = await client.post("/menus", json=submenu_data)
        assert response.status_code == 201
        data = response.json()
        assert data["name"] == submenu_data["name"]
        assert data["parent_id"] == parent_id


@pytest.mark.asyncio
async def test_get_menu(app: FastAPI):
    """测试获取菜单详情"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个菜单
        create_response = await client.post("/menus", json=test_menu_data)
        menu_id = create_response.json()["id"]

        # 获取菜单详情
        response = await client.get(f"/menus/{menu_id}")
        assert response.status_code == 200
        data = response.json()
        assert data["id"] == menu_id
        assert data["name"] == test_menu_data["name"]
        assert data["path"] == test_menu_data["path"]


@pytest.mark.asyncio
async def test_list_menus(app: FastAPI):
    """测试获取菜单列表"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 创建父菜单
        parent_response = await client.post("/menus", json=test_menu_data)
        parent_id = parent_response.json()["id"]

        # 创建子菜单
        submenu_data = {**test_submenu_data, "parent_id": parent_id}
        await client.post("/menus", json=submenu_data)

        # 测试获取所有菜单
        response = await client.get("/menus")
        assert response.status_code == 200
        data = response.json()
        assert data["total"] >= 2  # 至少包含一个父菜单和一个子菜单
        assert len(data["items"]) >= 2

        # 测试获取树形结构
        response = await client.get("/menus/tree")
        if response.status_code != 200:
            print("Error response:", response.json())
        assert response.status_code == 200


@pytest.mark.asyncio
async def test_update_menu(app: FastAPI):
    """测试更新菜单"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个菜单
        create_response = await client.post("/menus", json=test_menu_data)
        menu_id = create_response.json()["id"]

        # 更新菜单信息
        update_data = {
            "name": "更新后的菜单",
            "path": "/updated",
            "icon": "new-icon",
            "order": 2,
        }
        response = await client.patch(f"/menus/{menu_id}", json=update_data)
        assert response.status_code == 200
        data = response.json()
        assert data["id"] == menu_id
        assert data["name"] == update_data["name"]
        assert data["path"] == update_data["path"]
        assert data["icon"] == update_data["icon"]
        assert data["order"] == update_data["order"]


@pytest.mark.asyncio
async def test_delete_menu(app: FastAPI):
    """测试删除菜单"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个菜单
        create_response = await client.post("/menus", json=test_menu_data)
        menu_id = create_response.json()["id"]

        # 删除菜单
        response = await client.delete(f"/menus/{menu_id}")
        assert response.status_code == 204

        # 确认菜单已被删除
        response = await client.get(f"/menus/{menu_id}")
        assert response.status_code == 404


@pytest.mark.asyncio
async def test_delete_menu_with_children(app: FastAPI):
    """测试删除带有子菜单的菜单"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 创建父菜单
        parent_response = await client.post("/menus", json=test_menu_data)
        parent_id = parent_response.json()["id"]

        # 创建子菜单
        submenu_data = {**test_submenu_data, "parent_id": parent_id}
        submenu_response = await client.post("/menus", json=submenu_data)
        submenu_id = submenu_response.json()["id"]

        # 删除父菜单（应该同时删除子菜单）
        response = await client.delete(f"/menus/{parent_id}")
        assert response.status_code == 204  # 成功删除

        # 验证父菜单和子菜单都已被删除
        parent_response = await client.get(f"/menus/{parent_id}")
        assert parent_response.status_code == 404
        submenu_response = await client.get(f"/menus/{submenu_id}")
        assert submenu_response.status_code == 404
