from typing import AsyncGenerator

import pytest
from fastapi import FastAPI
from httpx import AsyncClient
from tortoise import Tortoise

from algo_flow.app.system.models import Permission, Role, User
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.app.system.views.config import config_router
from algo_flow.cores.pwd import get_password_hash

# 测试数据
test_admin_data = {
    "username": "admin",
    "email": "admin@example.com",
    "password": "adminpass123",
}

test_config_data = {
    "name": "测试配置",
    "description": "这是一个测试配置",
    "value": {"key": "value"},
    "type": "json",
}

test_role_data = {
    "name": "admin",
    "identifier": "admin",
    "description": "Admin role",
}

test_permission_data = {
    "name": "系统配置管理",
    "identifier": "system:config:create",
    "module": "system",
}


async def mock_get_current_active_user():
    """模拟认证用户"""
    return await User.get(username=test_admin_data["username"])


@pytest.fixture
def app() -> FastAPI:
    """创建测试应用"""
    app = FastAPI()
    app.include_router(config_router, prefix="/configs")
    app.dependency_overrides[get_current_active_user] = mock_get_current_active_user
    return app


@pytest.fixture(autouse=True)
async def setup_database() -> AsyncGenerator:
    """设置测试数据库"""
    await Tortoise.init(
        db_url="sqlite://:memory:",
        modules={"models": ["app.system.models"]},
    )
    await Tortoise.generate_schemas()

    # 创建管理员权限
    admin_permission = await Permission.create(**test_permission_data)

    # 创建管理员角色
    admin_role = await Role.create(**test_role_data)
    await admin_role.permissions.add(admin_permission)

    # 创建管理员用户
    admin = await User.create(
        username=test_admin_data["username"],
        email=test_admin_data["email"],
        hashed_password=get_password_hash(test_admin_data["password"]),
        is_superuser=True,
    )
    await admin.roles.add(admin_role)

    yield

    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_create_config(app: FastAPI):
    """测试创建配置"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.post("/configs", json=test_config_data)
        assert response.status_code == 201
        data = response.json()
        assert data["name"] == test_config_data["name"]
        assert data["description"] == test_config_data["description"]
        assert data["value"] == test_config_data["value"]
        assert data["type"] == test_config_data["type"]


@pytest.mark.asyncio
async def test_create_config_with_duplicate_name(app: FastAPI):
    """测试创建重复名称的配置"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个配置
        await client.post("/configs", json=test_config_data)

        # 尝试创建同名配置
        response = await client.post("/configs", json=test_config_data)
        assert response.status_code == 409


@pytest.mark.asyncio
async def test_get_config(app: FastAPI):
    """测试获取配置详情"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个配置
        create_response = await client.post("/configs", json=test_config_data)
        config_id = create_response.json()["id"]

        # 获取配置详情
        response = await client.get(f"/configs/{config_id}")
        assert response.status_code == 200
        data = response.json()
        assert data["id"] == config_id
        assert data["name"] == test_config_data["name"]
        assert data["value"] == test_config_data["value"]
        assert data["type"] == test_config_data["type"]


@pytest.mark.asyncio
async def test_list_configs(app: FastAPI):
    """测试获取配置列表"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 创建多个配置
        await client.post("/configs", json=test_config_data)
        await client.post(
            "/configs",
            json={
                **test_config_data,
                "name": "测试配置2",
                "value": {"key2": "value2"},
            },
        )

        # 测试无过滤条件
        response = await client.get("/configs")
        assert response.status_code == 200
        data = response.json()
        assert data["total"] == 2
        assert len(data["items"]) == 2

        # 测试按类型过滤
        response = await client.get("/configs", params={"type": "json"})
        assert response.status_code == 200
        data = response.json()
        assert data["total"] == 2
        assert len(data["items"]) == 2

        # 测试分页
        response = await client.get("/configs", params={"limit": 1})
        assert response.status_code == 200
        data = response.json()
        assert data["total"] == 2
        assert len(data["items"]) == 1


@pytest.mark.asyncio
async def test_update_config(app: FastAPI):
    """测试更新配置"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个配置
        create_response = await client.post("/configs", json=test_config_data)
        config_id = create_response.json()["id"]

        # 更新配置信息
        update_data = {
            "name": "更新后的配置",
            "description": "这是更新后的配置描述",
            "value": {"updated_key": "updated_value"},
        }
        response = await client.patch(f"/configs/{config_id}", json=update_data)
        assert response.status_code == 200
        data = response.json()
        assert data["id"] == config_id
        assert data["name"] == update_data["name"]
        assert data["description"] == update_data["description"]
        assert data["value"] == update_data["value"]


@pytest.mark.asyncio
async def test_delete_config(app: FastAPI):
    """测试删除配置"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个配置
        create_response = await client.post("/configs", json=test_config_data)
        config_id = create_response.json()["id"]

        # 删除配置
        response = await client.delete(f"/configs/{config_id}")
        assert response.status_code == 204

        # 确认配置已被删除
        response = await client.get(f"/configs/{config_id}")
        assert response.status_code == 404
