from typing import AsyncGenerator

import pytest
from fastapi import FastAPI
from httpx import AsyncClient
from passlib.context import CryptContext
from tortoise import Tortoise

from algo_flow.app.system.models import Permission, Role, User
from algo_flow.app.system.views.auth import auth_router
from algo_flow.cores.pwd import get_password_hash

# 测试数据
test_user_data = {
    "username": "testuser",
    "email": "test@example.com",
    "password": "testpassword123",
}

test_role_data = {
    "name": "测试角色",
    "identifier": "test_role",
    "description": "这是一个测试角色",
}

test_permission_data = {
    "name": "test:permission",
    "identifier": "test:permission",
    "module": "api",
    "description": "Test permission",
}

pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")


@pytest.fixture
def app() -> FastAPI:
    """创建测试应用"""
    app = FastAPI()
    app.include_router(auth_router, prefix="/auth")
    return app


@pytest.fixture(autouse=True)
async def setup_database() -> AsyncGenerator:
    """设置测试数据库"""
    await Tortoise.init(
        db_url="sqlite://:memory:",
        modules={"models": ["app.system.models"]},
    )
    await Tortoise.generate_schemas()

    # 创建测试用户、角色和权限
    permission = await Permission.create(**test_permission_data)
    role = await Role.create(**test_role_data)
    await role.permissions.add(permission)

    user = await User.create(
        username=test_user_data["username"],
        email=test_user_data["email"],
        hashed_password=get_password_hash(test_user_data["password"]),
    )
    await user.roles.add(role)

    yield

    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_login_with_password(app: FastAPI):
    """测试密码登录"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.post(
            "/auth/password",
            data={
                "username": test_user_data["username"],
                "password": test_user_data["password"],
            },
        )
        assert response.status_code == 200
        data = response.json()
        assert "access_token" in data
        assert data["token_type"] == "bearer"
        assert "test:permission" in data["scopes"]


@pytest.mark.asyncio
async def test_login_with_wrong_password(app: FastAPI):
    """测试错误密码登录"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.post(
            "/auth/password",
            data={
                "username": test_user_data["username"],
                "password": "wrongpassword",
            },
        )
        assert response.status_code == 400


@pytest.mark.asyncio
async def test_login_with_nonexistent_user(app: FastAPI):
    """测试不存在用户登录"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.post(
            "/auth/password",
            data={
                "username": "nonexistent",
                "password": test_user_data["password"],
            },
        )
        assert response.status_code == 400


@pytest.mark.asyncio
async def test_oauth2_password_login(app: FastAPI):
    """测试OAuth2密码模式登录"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.post(
            "/auth/oauth2/password",
            data={
                "username": test_user_data["username"],
                "password": test_user_data["password"],
                "scope": "test:permission",
            },
        )
        assert response.status_code == 200
        data = response.json()
        assert "access_token" in data
        assert data["token_type"] == "bearer"
        assert "test:permission" in data["scopes"]


@pytest.mark.asyncio
async def test_oauth2_password_login_with_invalid_scope(app: FastAPI):
    """测试OAuth2密码模式使用无效权限登录"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.post(
            "/auth/oauth2/password",
            data={
                "username": test_user_data["username"],
                "password": test_user_data["password"],
                "scope": "invalid:permission",
            },
        )
        assert response.status_code == 400
