import uuid
from typing import AsyncGenerator

import pytest
from tortoise import Tortoise

from algo_flow.app.system.crud.user import (
    assign_roles,
    get_user_roles,
)
from algo_flow.app.system.models import Role, User
from algo_flow.cores.exceptions import ResourceNotFoundError
from algo_flow.cores.pwd import get_password_hash

# 测试数据
test_user_data = {
    "username": "testuser",
    "password": "testpassword",
    "email": "test@example.com",
    "avatar": None,
    "nickname": "测试用户",
}

test_role_data = {
    "name": "测试角色",
    "identifier": "test_role",
    "description": "这是一个测试角色",
}


@pytest.fixture(autouse=True)
async def setup_database() -> AsyncGenerator:
    """设置测试数据库"""
    await Tortoise.init(
        db_url="sqlite://:memory:",
        modules={"models": ["app.system.models"]},
    )
    await Tortoise.generate_schemas()

    yield

    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_assign_roles():
    """测试分配用户角色"""
    # 创建用户和角色
    user = await User.create(
        username=test_user_data["username"],
        hashed_password=get_password_hash(test_user_data["password"]),
        email=test_user_data["email"],
        avatar=test_user_data["avatar"],
    )
    role = await Role.create(**test_role_data)

    # 分配角色
    updated_user = await assign_roles(user.id, [role.id])

    # 验证角色分配
    assert len(updated_user.roles) == 1
    assert updated_user.roles[0].id == role.id


@pytest.mark.asyncio
async def test_assign_multiple_roles():
    """测试分配多个角色"""
    # 创建用户和多个角色
    user = await User.create(
        username=test_user_data["username"],
        hashed_password=get_password_hash(test_user_data["password"]),
        email=test_user_data["email"],
        avatar=test_user_data["avatar"],
    )
    role1 = await Role.create(**test_role_data)
    role2 = await Role.create(
        **{
            **test_role_data,
            "name": "测试角色2",
            "identifier": "test_role2",
        }
    )

    # 分配多个角色
    updated_user = await assign_roles(user.id, [role1.id, role2.id])

    # 验证角色分配
    assert len(updated_user.roles) == 2
    role_ids = {role.id for role in updated_user.roles}
    assert role1.id in role_ids
    assert role2.id in role_ids


@pytest.mark.asyncio
async def test_reassign_roles():
    """测试重新分配角色"""
    # 创建用户和角色
    user = await User.create(
        username=test_user_data["username"],
        hashed_password=get_password_hash(test_user_data["password"]),
        email=test_user_data["email"],
        avatar=test_user_data["avatar"],
    )
    role1 = await Role.create(**test_role_data)
    role2 = await Role.create(
        **{
            **test_role_data,
            "name": "测试角色2",
            "identifier": "test_role2",
        }
    )

    # 先分配角色1
    await assign_roles(user.id, [role1.id])

    # 重新分配为角色2
    updated_user = await assign_roles(user.id, [role2.id])

    # 验证角色分配
    assert len(updated_user.roles) == 1
    assert updated_user.roles[0].id == role2.id


@pytest.mark.asyncio
async def test_clear_roles():
    """测试清除用户角色"""
    # 创建用户和角色
    user = await User.create(
        username=test_user_data["username"],
        hashed_password=get_password_hash(test_user_data["password"]),
        email=test_user_data["email"],
        avatar=test_user_data["avatar"],
    )
    role = await Role.create(**test_role_data)

    # 先分配角色
    await assign_roles(user.id, [role.id])

    # 清除角色
    updated_user = await assign_roles(user.id, [])

    # 验证角色已清除
    assert len(updated_user.roles) == 0


@pytest.mark.asyncio
async def test_assign_roles_to_nonexistent_user():
    """测试给不存在的用户分配角色"""
    # 创建角色
    role = await Role.create(**test_role_data)

    # 尝试给不存在的用户分配角色
    with pytest.raises(ResourceNotFoundError):
        await assign_roles(uuid.uuid4(), [role.id])


@pytest.mark.asyncio
async def test_assign_nonexistent_roles():
    """测试分配不存在的角色"""
    # 创建用户
    user = await User.create(
        username=test_user_data["username"],
        hashed_password=get_password_hash(test_user_data["password"]),
        email=test_user_data["email"],
        avatar=test_user_data["avatar"],
    )

    # 尝试分配不存在的角色
    with pytest.raises(ResourceNotFoundError):
        await assign_roles(user.id, [uuid.uuid4()])


@pytest.mark.asyncio
async def test_get_user_roles():
    """测试获取用户角色"""
    # 创建用户和角色
    user = await User.create(
        username=test_user_data["username"],
        hashed_password=get_password_hash(test_user_data["password"]),
        email=test_user_data["email"],
        avatar=test_user_data["avatar"],
    )
    role = await Role.create(**test_role_data)

    # 分配角色
    await assign_roles(user.id, [role.id])

    # 获取用户角色
    user_with_roles = await get_user_roles(user.id)

    # 验证角色信息
    assert len(user_with_roles.roles) == 1
    assert user_with_roles.roles[0].id == role.id
    assert user_with_roles.roles[0].name == role.name
    assert user_with_roles.roles[0].identifier == role.identifier


@pytest.mark.asyncio
async def test_get_roles_of_nonexistent_user():
    """测试获取不存在用户的角色"""
    with pytest.raises(ResourceNotFoundError):
        await get_user_roles(uuid.uuid4())
