import uuid
from typing import AsyncGenerator

import pytest
from tortoise import Tortoise

from algo_flow.app.system.crud.user import (
    count_users,
    create_user,
    delete_user,
    get_user,
    list_users,
    update_user,
)
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

# 测试数据
test_user_data = {
    "username": "testuser",
    "password": "testpassword",
    "email": "test@example.com",
    "is_active": True,
    "avatar": None,
}

# 测试数据
test_role_data = {
    "name": "测试角色",
    "identifier": "test_role",
    "description": "这是一个测试角色",
}


@pytest.fixture(autouse=True)
async def setup_database() -> AsyncGenerator:
    """设置测试数据库"""
    await Tortoise.init(
        db_url="sqlite://:memory:",
        modules={"models": ["app.system.models"]},
    )
    await Tortoise.generate_schemas()

    yield

    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_create_user():
    """测试创建用户"""
    # 创建用户
    user = await create_user(**test_user_data)

    # 验证用户信息
    assert user.username == test_user_data["username"]
    assert user.email == test_user_data["email"]


@pytest.mark.asyncio
async def test_create_user_with_duplicate_username():
    """测试创建重复用户名的用户"""
    # 先创建一个用户
    await create_user(**test_user_data)

    # 尝试创建同名用户
    with pytest.raises(ResourceConflictError):
        await create_user(**test_user_data)


@pytest.mark.asyncio
async def test_get_user():
    """测试获取用户"""
    # 先创建一个用户
    created_user = await create_user(**test_user_data)

    # 获取用户
    user = await get_user(created_user.id)

    # 验证用户信息
    assert user.id == created_user.id
    assert user.username == test_user_data["username"]
    assert user.email == test_user_data["email"]


@pytest.mark.asyncio
async def test_get_nonexistent_user():
    """测试获取不存在的用户"""
    with pytest.raises(ResourceNotFoundError):
        await get_user(uuid.uuid4())


@pytest.mark.asyncio
async def test_list_users():
    """测试获取用户列表"""
    # 创建多个用户
    await create_user(**test_user_data)
    await create_user(
        **{
            **test_user_data,
            "username": "testuser2",
            "email": "test2@example.com",
        }
    )

    # 测试无过滤条件
    users = await list_users()
    assert len(users) == 2

    # 测试分页
    users = await list_users(limit=1)
    assert len(users) == 1


@pytest.mark.asyncio
async def test_update_user():
    """测试更新用户"""
    # 先创建一个用户
    user = await create_user(**test_user_data)

    # 更新用户
    updated_data = {
        "email": "updated@example.com",
    }
    updated_user = await update_user(user.id, **updated_data)

    # 验证更新后的信息
    assert updated_user.id == user.id
    assert updated_user.email == updated_data["email"]


@pytest.mark.asyncio
async def test_update_nonexistent_user():
    """测试更新不存在的用户"""
    with pytest.raises(ResourceNotFoundError):
        await update_user(
            uuid.uuid4(),
            email="new@example.com",
        )


@pytest.mark.asyncio
async def test_update_user_with_duplicate_username():
    """测试更新用户时使用重复的用户名"""
    # 创建两个用户
    user1 = await create_user(**test_user_data)
    user2 = await create_user(
        **{
            **test_user_data,
            "username": "testuser2",
            "email": "test2@example.com",
        }
    )

    # 尝试将 user2 的用户名更新为 user1 的用户名
    with pytest.raises(ResourceConflictError):
        await update_user(
            user2.id,
            username=user1.username,
        )


@pytest.mark.asyncio
async def test_delete_user():
    """测试删除用户"""
    # 先创建一个用户
    user = await create_user(**test_user_data)

    # 删除用户
    await delete_user(user.id)

    # 验证用户已被删除
    with pytest.raises(ResourceNotFoundError):
        await get_user(user.id)


@pytest.mark.asyncio
async def test_delete_nonexistent_user():
    """测试删除不存在的用户"""
    with pytest.raises(ResourceNotFoundError):
        await delete_user(uuid.uuid4())


@pytest.mark.asyncio
async def test_count_users():
    """测试用户计数"""
    # 初始状态
    count = await count_users()
    assert count == 0

    # 创建用户后
    await create_user(**test_user_data)
    count = await count_users()
    assert count == 1
