import uuid
from typing import AsyncGenerator

import pytest
from tortoise import Tortoise

from algo_flow.app.system.crud.role import (
    assign_permissions,
    count_roles,
    create_role,
    delete_role,
    get_role,
    get_role_permissions,
    list_roles,
    update_role,
)
from algo_flow.app.system.models import Permission
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

# 测试数据
test_role_data = {
    "name": "测试角色",
    "identifier": "test_role",
    "description": "这是一个测试角色",
}

test_permission_data = {
    "name": "测试权限",
    "identifier": "test:permission",
    "module": "test",
    "description": "这是一个测试权限",
}


@pytest.fixture(autouse=True)
async def setup_database() -> AsyncGenerator:
    """设置测试数据库"""
    await Tortoise.init(
        db_url="sqlite://:memory:",
        modules={"models": ["app.system.models"]},
    )
    await Tortoise.generate_schemas()

    yield

    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_create_role():
    """测试创建角色"""
    # 创建角色
    role = await create_role(**test_role_data)

    # 验证角色信息
    assert role.name == test_role_data["name"]
    assert role.identifier == test_role_data["identifier"]
    assert role.description == test_role_data["description"]


@pytest.mark.asyncio
async def test_create_role_with_duplicate_identifier():
    """测试创建重复标识符的角色"""
    # 先创建一个角色
    await create_role(**test_role_data)

    # 尝试创建相同标识符的角色
    with pytest.raises(ResourceConflictError):
        await create_role(**test_role_data)


@pytest.mark.asyncio
async def test_get_role():
    """测试获取角色"""
    # 先创建一个角色
    created_role = await create_role(**test_role_data)

    # 获取角色
    role = await get_role(created_role.id)

    # 验证角色信息
    assert role.id == created_role.id
    assert role.name == test_role_data["name"]
    assert role.identifier == test_role_data["identifier"]


@pytest.mark.asyncio
async def test_get_nonexistent_role():
    """测试获取不存在的角色"""
    with pytest.raises(ResourceNotFoundError):
        await get_role(uuid.uuid4())


@pytest.mark.asyncio
async def test_list_roles():
    """测试获取角色列表"""
    # 创建多个角色
    await create_role(**test_role_data)
    await create_role(
        **{
            **test_role_data,
            "name": "测试角色2",
            "identifier": "test_role2",
        }
    )

    # 测试无过滤条件
    roles = await list_roles()
    assert len(roles) == 2

    # 测试分页
    roles = await list_roles(limit=1)
    assert len(roles) == 1


@pytest.mark.asyncio
async def test_update_role():
    """测试更新角色"""
    # 先创建一个角色
    role = await create_role(**test_role_data)

    # 更新角色
    updated_data = {
        "name": "更新后的角色",
        "identifier": "updated_role",
        "description": "这是更新后的描述",
    }
    updated_role = await update_role(role.id, **updated_data)

    # 验证更新后的信息
    assert updated_role.id == role.id
    assert updated_role.name == updated_data["name"]
    assert updated_role.identifier == updated_data["identifier"]
    assert updated_role.description == updated_data["description"]


@pytest.mark.asyncio
async def test_update_nonexistent_role():
    """测试更新不存在的角色"""
    with pytest.raises(ResourceNotFoundError):
        await update_role(
            uuid.uuid4(),
            name="新名称",
        )


@pytest.mark.asyncio
async def test_update_role_with_duplicate_identifier():
    """测试更新角色时使用重复的标识符"""
    # 创建两个角色
    role1 = await create_role(**test_role_data)
    role2 = await create_role(
        **{
            **test_role_data,
            "name": "测试角色2",
            "identifier": "test_role2",
        }
    )

    # 尝试将 role2 的标识符更新为 role1 的标识符
    with pytest.raises(ResourceConflictError):
        await update_role(
            role2.id,
            identifier=role1.identifier,
        )


@pytest.mark.asyncio
async def test_delete_role():
    """测试删除角色"""
    # 先创建一个角色
    role = await create_role(**test_role_data)

    # 删除角色
    await delete_role(role.id)

    # 验证角色已被删除
    with pytest.raises(ResourceNotFoundError):
        await get_role(role.id)


@pytest.mark.asyncio
async def test_delete_nonexistent_role():
    """测试删除不存在的角色"""
    with pytest.raises(ResourceNotFoundError):
        await delete_role(uuid.uuid4())


@pytest.mark.asyncio
async def test_count_roles():
    """测试角色计数"""
    # 初始状态
    count = await count_roles()
    assert count == 0

    # 创建角色后
    await create_role(**test_role_data)
    count = await count_roles()
    assert count == 1


@pytest.mark.asyncio
async def test_assign_permissions():
    """测试分配权限"""
    # 创建角色和权限
    role = await create_role(**test_role_data)
    permission = await Permission.create(**test_permission_data)

    # 分配权限
    updated_role = await assign_permissions(role.id, [permission.id])

    # 验证权限分配
    assert len(updated_role.permissions) == 1
    assert updated_role.permissions[0].id == permission.id


@pytest.mark.asyncio
async def test_assign_permissions_to_nonexistent_role():
    """测试给不存在的角色分配权限"""
    with pytest.raises(ResourceNotFoundError):
        await assign_permissions(uuid.uuid4(), [uuid.uuid4()])


@pytest.mark.asyncio
async def test_assign_nonexistent_permissions():
    """测试分配不存在的权限"""
    # 创建角色
    role = await create_role(**test_role_data)

    # 尝试分配不存在的权限
    with pytest.raises(ResourceNotFoundError):
        await assign_permissions(role.id, [uuid.uuid4()])


@pytest.mark.asyncio
async def test_get_role_permissions():
    """测试获取角色权限"""
    # 创建角色和权限
    role = await create_role(**test_role_data)
    permission = await Permission.create(**test_permission_data)

    # 分配权限
    await assign_permissions(role.id, [permission.id])

    # 获取角色权限
    role_with_permissions = await get_role_permissions(role.id)

    # 验证权限信息
    assert len(role_with_permissions.permissions) == 1
    assert role_with_permissions.permissions[0].id == permission.id
