import uuid
from typing import AsyncGenerator

import pytest
from tortoise import Tortoise

from algo_flow.app.system.crud.permission import (
    count_permissions,
    create_permission,
    delete_permission,
    get_permission,
    list_permissions,
    update_permission,
)
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

# 测试数据
test_permission_data = {
    "name": "测试权限",
    "identifier": "test:permission",
    "module": "test",
    "description": "这是一个测试权限",
}


@pytest.fixture(autouse=True)
async def setup_database() -> AsyncGenerator:
    """设置测试数据库"""
    await Tortoise.init(
        db_url="sqlite://:memory:",
        modules={"models": ["app.system.models"]},
    )
    await Tortoise.generate_schemas()

    yield

    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_create_permission():
    """测试创建权限"""
    # 创建权限
    permission = await create_permission(**test_permission_data)

    # 验证权限信息
    assert permission.name == test_permission_data["name"]
    assert permission.identifier == test_permission_data["identifier"]
    assert permission.module == test_permission_data["module"]
    assert permission.description == test_permission_data["description"]


@pytest.mark.asyncio
async def test_create_permission_with_duplicate_identifier():
    """测试创建重复标识符的权限"""
    # 先创建一个权限
    await create_permission(**test_permission_data)

    # 尝试创建相同标识符的权限
    with pytest.raises(ResourceConflictError):
        await create_permission(**test_permission_data)


@pytest.mark.asyncio
async def test_get_permission():
    """测试获取权限"""
    # 先创建一个权限
    created_permission = await create_permission(**test_permission_data)

    # 获取权限
    permission = await get_permission(created_permission.id)

    # 验证权限信息
    assert permission.id == created_permission.id
    assert permission.name == test_permission_data["name"]
    assert permission.identifier == test_permission_data["identifier"]
    assert permission.module == test_permission_data["module"]


@pytest.mark.asyncio
async def test_get_nonexistent_permission():
    """测试获取不存在的权限"""
    with pytest.raises(ResourceNotFoundError):
        await get_permission(uuid.uuid4())


@pytest.mark.asyncio
async def test_list_permissions():
    """测试获取权限列表"""
    # 创建多个权限
    await create_permission(**test_permission_data)
    await create_permission(
        **{
            **test_permission_data,
            "name": "测试权限2",
            "identifier": "test:permission2",
            "module": "test2",
        }
    )

    # 测试无过滤条件
    permissions = await list_permissions()
    assert len(permissions) == 2

    # 测试按模块过滤
    permissions = await list_permissions(module="test")
    assert len(permissions) == 1
    assert permissions[0].module == "test"

    # 测试分页
    permissions = await list_permissions(limit=1)
    assert len(permissions) == 1


@pytest.mark.asyncio
async def test_update_permission():
    """测试更新权限"""
    # 先创建一个权限
    permission = await create_permission(**test_permission_data)

    # 更新权限
    updated_data = {
        "name": "更新后的权限",
        "identifier": "updated:permission",
        "module": "updated",
        "description": "这是更新后的描述",
    }
    updated_permission = await update_permission(permission.id, **updated_data)

    # 验证更新后的信息
    assert updated_permission.id == permission.id
    assert updated_permission.name == updated_data["name"]
    assert updated_permission.identifier == updated_data["identifier"]
    assert updated_permission.module == updated_data["module"]
    assert updated_permission.description == updated_data["description"]


@pytest.mark.asyncio
async def test_update_nonexistent_permission():
    """测试更新不存在的权限"""
    with pytest.raises(ResourceNotFoundError):
        await update_permission(
            uuid.uuid4(),
            name="新名称",
        )


@pytest.mark.asyncio
async def test_update_permission_with_duplicate_identifier():
    """测试更新权限时使用重复的标识符"""
    # 创建两个权限
    permission1 = await create_permission(**test_permission_data)
    permission2 = await create_permission(
        **{
            **test_permission_data,
            "name": "测试权限2",
            "identifier": "test:permission2",
        }
    )

    # 尝试将 permission2 的标识符更新为 permission1 的标识符
    with pytest.raises(ResourceConflictError):
        await update_permission(
            permission2.id,
            identifier=permission1.identifier,
        )


@pytest.mark.asyncio
async def test_delete_permission():
    """测试删除权限"""
    # 先创建一个权限
    permission = await create_permission(**test_permission_data)

    # 删除权限
    await delete_permission(permission.id)

    # 验证权限已被删除
    with pytest.raises(ResourceNotFoundError):
        await get_permission(permission.id)


@pytest.mark.asyncio
async def test_delete_nonexistent_permission():
    """测试删除不存在的权限"""
    with pytest.raises(ResourceNotFoundError):
        await delete_permission(uuid.uuid4())


@pytest.mark.asyncio
async def test_count_permissions():
    """测试权限计数"""
    # 初始状态
    count = await count_permissions()
    assert count == 0

    # 创建权限后
    await create_permission(**test_permission_data)
    count = await count_permissions()
    assert count == 1

    # 按模块统计
    count = await count_permissions(module="test")
    assert count == 1
    count = await count_permissions(module="other")
    assert count == 0
