import uuid
from typing import AsyncGenerator

import pytest
from tortoise import Tortoise

from algo_flow.app.system.crud.menu import (
    count_menus,
    create_menu,
    delete_menu,
    get_menu,
    list_menus,
    update_menu,
)
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

# 测试数据
test_menu_data = {
    "name": "测试菜单",
    "path": "/test",
    "component": "Test",
    "locale": "menu.test",
    "icon": "test-icon",
    "requires_auth": True,
    "order": 1,
    "hide_in_menu": False,
    "hide_children_in_menu": False,
    "no_affix": False,
    "ignore_cache": False,
}


@pytest.fixture(autouse=True)
async def setup_database() -> AsyncGenerator:
    """设置测试数据库"""
    await Tortoise.init(
        db_url="sqlite://:memory:",
        modules={"models": ["app.system.models"]},
    )
    await Tortoise.generate_schemas()

    yield

    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_create_menu():
    """测试创建菜单"""
    # 创建菜单
    menu = await create_menu(**test_menu_data)

    # 验证菜单信息
    assert menu.name == test_menu_data["name"]
    assert menu.path == test_menu_data["path"]
    assert menu.component == test_menu_data["component"]
    assert menu.locale == test_menu_data["locale"]
    assert menu.icon == test_menu_data["icon"]
    assert menu.requires_auth == test_menu_data["requires_auth"]
    assert menu.order == test_menu_data["order"]
    assert menu.hide_in_menu == test_menu_data["hide_in_menu"]
    assert menu.hide_children_in_menu == test_menu_data["hide_children_in_menu"]
    assert menu.no_affix == test_menu_data["no_affix"]
    assert menu.ignore_cache == test_menu_data["ignore_cache"]
    assert menu.parent_id is None


@pytest.mark.asyncio
async def test_create_duplicate_menu():
    """测试创建重复菜单"""
    # 先创建一个菜单
    await create_menu(**test_menu_data)

    # 尝试创建同名菜单
    with pytest.raises(ResourceConflictError):
        await create_menu(**test_menu_data)


@pytest.mark.asyncio
async def test_create_submenu():
    """测试创建子菜单"""
    # 先创建父菜单
    parent = await create_menu(**test_menu_data)

    # 创建子菜单
    submenu_data = {
        **test_menu_data,
        "name": "子菜单",
        "path": "/test/sub",
        "parent_id": parent.id,
    }
    submenu = await create_menu(**submenu_data)

    # 验证子菜单信息
    assert submenu.name == submenu_data["name"]
    assert submenu.path == submenu_data["path"]
    assert submenu.parent_id == parent.id


@pytest.mark.asyncio
async def test_create_menu_with_nonexistent_parent():
    """测试使用不存在的父菜单创建菜单"""
    with pytest.raises(ResourceNotFoundError):
        await create_menu(**{**test_menu_data, "parent_id": uuid.uuid4()})


@pytest.mark.asyncio
async def test_get_menu():
    """测试获取菜单"""
    # 先创建一个菜单
    created_menu = await create_menu(**test_menu_data)

    # 获取菜单
    menu = await get_menu(created_menu.id)

    # 验证菜单信息
    assert menu.id == created_menu.id
    assert menu.name == test_menu_data["name"]
    assert menu.path == test_menu_data["path"]
    assert menu.component == test_menu_data["component"]
    assert menu.locale == test_menu_data["locale"]
    assert menu.icon == test_menu_data["icon"]
    assert menu.requires_auth == test_menu_data["requires_auth"]
    assert menu.order == test_menu_data["order"]
    assert menu.hide_in_menu == test_menu_data["hide_in_menu"]
    assert menu.hide_children_in_menu == test_menu_data["hide_children_in_menu"]
    assert menu.no_affix == test_menu_data["no_affix"]
    assert menu.ignore_cache == test_menu_data["ignore_cache"]


@pytest.mark.asyncio
async def test_get_nonexistent_menu():
    """测试获取不存在的菜单"""
    with pytest.raises(ResourceNotFoundError):
        await get_menu(uuid.uuid4())


@pytest.mark.asyncio
async def test_list_menus():
    """测试获取菜单列表"""
    # 创建父菜单
    parent = await create_menu(**test_menu_data)

    # 创建子菜单
    await create_menu(
        **{
            **test_menu_data,
            "name": "子菜单",
            "path": "/test/child",
            "parent_id": parent.id,
        }
    )

    # 测试无过滤条件
    menus = await list_menus()
    assert len(menus) == 2

    # 测试按父菜单过滤
    menus = await list_menus(parent_id=parent.id)
    assert len(menus) == 1
    assert menus[0].parent_id == parent.id

    # 测试分页
    menus = await list_menus(limit=1)
    assert len(menus) == 1


@pytest.mark.asyncio
async def test_update_menu():
    """测试更新菜单"""
    # 先创建一个菜单
    menu = await create_menu(**test_menu_data)

    # 更新菜单
    update_data = {
        "name": "更新的菜单",
        "path": "/updated",
        "component": "Updated",
        "locale": "menu.updated",
        "icon": "updated-icon",
        "requires_auth": False,
        "order": 2,
        "hide_in_menu": True,
        "hide_children_in_menu": True,
        "no_affix": True,
        "ignore_cache": True,
    }
    updated_menu = await update_menu(menu.id, **update_data)

    # 验证更新后的信息
    assert updated_menu.name == update_data["name"]
    assert updated_menu.path == update_data["path"]
    assert updated_menu.component == update_data["component"]
    assert updated_menu.locale == update_data["locale"]
    assert updated_menu.icon == update_data["icon"]
    assert updated_menu.requires_auth == update_data["requires_auth"]
    assert updated_menu.order == update_data["order"]
    assert updated_menu.hide_in_menu == update_data["hide_in_menu"]
    assert updated_menu.hide_children_in_menu == update_data["hide_children_in_menu"]
    assert updated_menu.no_affix == update_data["no_affix"]
    assert updated_menu.ignore_cache == update_data["ignore_cache"]


@pytest.mark.asyncio
async def test_update_menu_parent():
    """测试更新菜单的父菜单"""
    # 创建两个菜单
    menu1 = await create_menu(**test_menu_data)
    menu2 = await create_menu(**{**test_menu_data, "name": "菜单2", "path": "/test2"})

    # 将menu2设置为menu1的子菜单
    updated_menu = await update_menu(menu2.id, parent_id=menu1.id)
    assert updated_menu.parent_id == menu1.id


@pytest.mark.asyncio
async def test_update_menu_invalid_parent():
    """测试使用无效的父菜单更新菜单"""
    # 创建菜单
    menu = await create_menu(**test_menu_data)

    # 尝试设置不存在的父菜单
    with pytest.raises(ResourceNotFoundError):
        await update_menu(menu.id, parent_id=uuid.uuid4())


@pytest.mark.asyncio
async def test_update_menu_circular_reference():
    """测试更新菜单时的循环引用"""
    # 创建父子菜单
    parent = await create_menu(**test_menu_data)
    child = await create_menu(
        **{
            **test_menu_data,
            "name": "子菜单",
            "path": "/test/child",
            "parent_id": parent.id,
        }
    )

    # 尝试将父菜单设置为子菜单的子菜单
    with pytest.raises(ResourceConflictError):
        await update_menu(parent.id, parent_id=child.id)


@pytest.mark.asyncio
async def test_delete_menu():
    """测试删除菜单"""
    # 先创建一个父菜单
    parent = await create_menu(**test_menu_data)

    # 创建子菜单
    await create_menu(
        **{
            **test_menu_data,
            "name": "子菜单",
            "path": "/test/child",
            "parent_id": parent.id,
        }
    )

    # 删除父菜单（应该同时删除子菜单）
    await delete_menu(parent.id)

    # 验证父菜单和子菜单都已被删除
    with pytest.raises(ResourceNotFoundError):
        await get_menu(parent.id)

    # 验证子菜单数量为0
    count = await count_menus()
    assert count == 0


@pytest.mark.asyncio
async def test_delete_nonexistent_menu():
    """测试删除不存在的菜单"""
    with pytest.raises(ResourceNotFoundError):
        await delete_menu(uuid.uuid4())


@pytest.mark.asyncio
async def test_count_menus():
    """测试菜单计数"""
    # 初始状态
    count = await count_menus()
    assert count == 0

    # 创建父菜单
    parent = await create_menu(**test_menu_data)
    count = await count_menus()
    assert count == 1

    # 创建子菜单后
    await create_menu(
        **{
            **test_menu_data,
            "name": "子菜单",
            "path": "/test/child",
            "parent_id": parent.id,
        }
    )
    count = await count_menus()
    assert count == 2

    # 按父菜单统计
    count = await count_menus(parent_id=parent.id)
    assert count == 1
