import hashlib
import hmac
from unittest.mock import patch

import pytest
from fastapi import FastAPI
from httpx import AsyncClient

from algo_flow.app.common.views import common_router


@pytest.fixture(autouse=True)
def mock_settings():
    """Mock settings for testing"""
    with patch("app.common.views.settings") as mock_settings:
        mock_settings.github.webhook_secret = "test_secret"
        yield mock_settings


@pytest.fixture
def app() -> FastAPI:
    """创建测试应用"""
    app = FastAPI()
    app.include_router(common_router, prefix="/common")
    return app


@pytest.mark.asyncio
async def test_healthy(app: FastAPI):
    """测试健康检查接口"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.post("/common/healthy")
        assert response.status_code == 200
        assert response.json() is None


@pytest.mark.asyncio
async def test_github_webhook_invalid_signature(app: FastAPI):
    """测试 GitHub Webhook 接口 - 无效签名"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.post(
            "/common/github-webhook",
            headers={"X-Hub-Signature-256": "invalid_signature"},
            json={"event": "push"},
        )
        assert response.status_code == 400
        assert response.json()["detail"] == "Invalid signature"


@pytest.mark.asyncio
async def test_github_webhook_valid_signature(app: FastAPI):
    """测试 GitHub Webhook 接口 - 有效签名"""
    # 准备测试数据
    payload = b'{"event": "push"}'
    signature = "sha256=" + hmac.new(b"test_secret", payload, hashlib.sha256).hexdigest()

    # Mock subprocess.run
    with patch("subprocess.run") as mock_run:
        async with AsyncClient(app=app, base_url="http://test") as client:
            response = await client.post(
                "/common/github-webhook",
                headers={"X-Hub-Signature-256": signature},
                content=payload,
            )
            assert response.status_code == 200
            assert response.json() == {"status": "success", "message": "Code pulled successfully"}

            # 验证是否调用了 git pull
            mock_run.assert_called_once_with(["git", "pull"], check=True)
