from unittest.mock import AsyncMock, patch
from uuid import UUID

import httpx
import pytest
from fastapi import FastAPI
from httpx import AsyncClient
from tortoise import Tortoise

from algo_flow.app.algo.models import Project
from algo_flow.app.algo.views.project import router
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.constant.algo import ProjectStatus
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError


# Mock the authentication dependency
async def mock_get_current_active_user():
    return {
        "id": "test-user-id",
        "scopes": [
            "algo:project:create",
            "algo:project:read",
            "algo:project:update",
            "algo:project:delete",
        ],
    }


test_project_id = UUID("12345678-1234-5678-1234-567812345678")
test_project_data = {"name": "测试项目", "description": "这是一个测试项目"}


@pytest.fixture
def app() -> FastAPI:
    """创建测试应用"""
    app = FastAPI()
    app.include_router(router, prefix="/projects")
    app.dependency_overrides[get_current_active_user] = mock_get_current_active_user
    return app


@pytest.fixture(autouse=True)
async def init_db():
    """初始化测试数据库"""
    await Tortoise.init(db_url="sqlite://:memory:", modules={"models": ["app.algo.models"]})
    await Tortoise.generate_schemas()
    yield
    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_create_project(app: FastAPI):
    """测试创建项目"""
    with patch("app.algo.views.project.create_project", new_callable=AsyncMock) as mock_create:
        mock_project = Project(
            id=test_project_id,
            name=test_project_data["name"],
            description=test_project_data["description"],
            status=ProjectStatus.ACTIVE,
        )
        mock_create.return_value = mock_project

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.post("/projects", json=test_project_data)

        assert response.status_code == 201
        response_data = response.json()
        assert response_data["name"] == test_project_data["name"]
        assert response_data["description"] == test_project_data["description"]
        assert response_data["status"] == ProjectStatus.ACTIVE
        mock_create.assert_called_once_with(
            name=test_project_data["name"], description=test_project_data["description"]
        )


@pytest.mark.asyncio
async def test_create_project_conflict(app: FastAPI):
    """测试创建项目 - 名称冲突"""
    with patch("app.algo.views.project.create_project", new_callable=AsyncMock) as mock_create:
        mock_create.side_effect = ResourceConflictError("项目名称已存在")

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.post("/projects", json=test_project_data)

        assert response.status_code == 400
        assert response.json()["detail"] == "项目名称已存在"


@pytest.mark.asyncio
async def test_get_project(app: FastAPI):
    """测试获取项目详情"""
    with patch("app.algo.views.project.get_project", new_callable=AsyncMock) as mock_get:
        mock_project = Project(
            id=test_project_id,
            name=test_project_data["name"],
            description=test_project_data["description"],
            status=ProjectStatus.ACTIVE,
        )
        mock_get.return_value = mock_project

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.get(f"/projects/{test_project_id}")

        assert response.status_code == 200
        response_data = response.json()
        assert response_data["id"] == str(test_project_id)
        assert response_data["name"] == test_project_data["name"]
        assert response_data["description"] == test_project_data["description"]
        assert response_data["status"] == ProjectStatus.ACTIVE
        mock_get.assert_called_once_with(test_project_id)


@pytest.mark.asyncio
async def test_get_project_not_found(app: FastAPI):
    """测试获取项目详情 - 项目不存在"""
    with patch("app.algo.views.project.get_project", new_callable=AsyncMock) as mock_get:
        mock_get.side_effect = ResourceNotFoundError(f"项目 {test_project_id} 不存在")

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.get(f"/projects/{test_project_id}")

        assert response.status_code == 404
        assert response.json()["detail"] == f"项目 {test_project_id} 不存在"


@pytest.mark.asyncio
async def test_list_projects(app: FastAPI):
    """测试获取项目列表"""
    with (
        patch("app.algo.views.project.list_projects", new_callable=AsyncMock) as mock_list,
        patch("app.algo.views.project.count_projects", new_callable=AsyncMock) as mock_count,
    ):
        mock_project = Project(
            id=test_project_id,
            name=test_project_data["name"],
            description=test_project_data["description"],
            status=ProjectStatus.ACTIVE,
        )
        mock_list.return_value = [mock_project]
        mock_count.return_value = 1

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.get("/projects")

        assert response.status_code == 200
        response_data = response.json()
        assert response_data["total"] == 1
        assert len(response_data["items"]) == 1
        item = response_data["items"][0]
        assert item["id"] == str(test_project_id)
        assert item["name"] == test_project_data["name"]
        assert item["description"] == test_project_data["description"]
        assert item["status"] == ProjectStatus.ACTIVE
        mock_list.assert_called_once_with(status=None, offset=0, limit=10)
        mock_count.assert_called_once_with(status=None)


@pytest.mark.asyncio
async def test_update_project(app: FastAPI):
    """测试更新项目"""
    update_data = {"name": "更新的项目名称", "description": "更新的项目描述"}

    with patch("app.algo.views.project.update_project", new_callable=AsyncMock) as mock_update:
        mock_project = Project(
            id=test_project_id,
            name=update_data["name"],
            description=update_data["description"],
            status=ProjectStatus.ACTIVE,
        )
        mock_update.return_value = mock_project

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.patch(f"/projects/{test_project_id}", json=update_data)

        assert response.status_code == 200
        response_data = response.json()
        assert response_data["id"] == str(test_project_id)
        assert response_data["name"] == update_data["name"]
        assert response_data["description"] == update_data["description"]
        assert response_data["status"] == ProjectStatus.ACTIVE
        mock_update.assert_called_once_with(
            project_id=test_project_id,
            name=update_data["name"],
            description=update_data["description"],
            status=None,
        )


@pytest.mark.asyncio
async def test_delete_project(app: FastAPI):
    """测试删除项目"""
    with patch("app.algo.views.project.delete_project", new_callable=AsyncMock) as mock_delete:
        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.delete(f"/projects/{test_project_id}")

        assert response.status_code == 204
        mock_delete.assert_called_once_with(test_project_id)


@pytest.mark.asyncio
async def test_delete_project_not_found(app: FastAPI):
    """测试删除项目 - 项目不存在"""
    with patch("app.algo.views.project.delete_project", new_callable=AsyncMock) as mock_delete:
        mock_delete.side_effect = ResourceNotFoundError(f"项目 {test_project_id} 不存在")

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.delete(f"/projects/{test_project_id}")

        assert response.status_code == 404
        assert response.json()["detail"] == f"项目 {test_project_id} 不存在"
