from unittest.mock import AsyncMock, patch
from uuid import UUID

import httpx
import pytest
from fastapi import FastAPI
from httpx import AsyncClient
from tortoise import Tortoise

from algo_flow.app.algo.models import ModelInfo
from algo_flow.app.algo.views.model import router as model_router
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.constant.algo import ModelFormat, ModelStatus, ModelType
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError


# Mock the authentication dependency
async def mock_get_current_active_user():
    return {
        "id": UUID("00000000-0000-0000-0000-000000000000"),
        "username": "test-user",
        "email": "test@example.com",
        "is_active": True,
        "scopes": [
            "algo:model:create",
            "algo:model:read",
            "algo:model:update",
            "algo:model:delete",
        ],
    }


test_project_id = UUID("12345678-1234-5678-1234-567812345678")
test_model_id = UUID("87654321-8765-4321-8765-432187654321")
test_model_data = {
    "project_id": str(test_project_id),
    "name": "测试模型",
    "description": "这是一个测试模型",
    "version": "1.0.0",
    "type": ModelType.CLASSIFICATION,
    "format": ModelFormat.PYTORCH,
    "status": ModelStatus.DRAFT,
    "storage_path": "/path/to/model",
    "config": {"key": "value"},
    "metrics": {"accuracy": 0.95},
}


@pytest.fixture
def app() -> FastAPI:
    """创建测试应用"""
    app = FastAPI()
    app.include_router(model_router, prefix="/models")
    app.dependency_overrides[get_current_active_user] = mock_get_current_active_user
    return app


@pytest.fixture(autouse=True)
async def init_db():
    """初始化测试数据库"""
    await Tortoise.init(
        db_url="sqlite://:memory:", modules={"models": ["app.algo.models", "app.system.models"]}
    )
    await Tortoise.generate_schemas()
    yield
    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_create_model(app: FastAPI):
    """测试创建模型"""
    with patch("app.algo.views.model.create_model", new_callable=AsyncMock) as mock_create:
        mock_model = ModelInfo(
            id=test_model_id,
            project_id=test_project_id,
            name=test_model_data["name"],
            description=test_model_data["description"],
            version=test_model_data["version"],
            type=test_model_data["type"],
            format=test_model_data["format"],
            status=test_model_data["status"],
            storage_path=test_model_data["storage_path"],
            config=test_model_data["config"],
            metrics=test_model_data["metrics"],
        )
        mock_create.return_value = mock_model

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.post("/models", json=test_model_data)

        assert response.status_code == 201
        response_data = response.json()
        assert response_data["name"] == test_model_data["name"]
        assert response_data["description"] == test_model_data["description"]
        assert response_data["version"] == test_model_data["version"]
        assert response_data["type"] == test_model_data["type"]
        assert response_data["format"] == test_model_data["format"]
        assert response_data["status"] == test_model_data["status"]
        assert response_data["storage_path"] == test_model_data["storage_path"]
        assert response_data["config"] == test_model_data["config"]
        assert response_data["metrics"] == test_model_data["metrics"]

        # 使用 model_dump 来获取实际传递的参数
        actual_call_args = mock_create.call_args[1]
        expected_args = {
            "project_id": test_project_id,
            "name": test_model_data["name"],
            "description": test_model_data["description"],
            "version": test_model_data["version"],
            "type": test_model_data["type"],
            "format": test_model_data["format"],
            "status": test_model_data["status"],
            "storage_path": test_model_data["storage_path"],
            "config": test_model_data["config"],
            "metrics": test_model_data["metrics"],
        }
        assert actual_call_args == expected_args


@pytest.mark.asyncio
async def test_create_model_conflict(app: FastAPI):
    """测试创建模型 - 名称冲突"""
    with patch("app.algo.views.model.create_model", new_callable=AsyncMock) as mock_create:
        mock_create.side_effect = ResourceConflictError("模型名称已存在")

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.post("/models", json=test_model_data)

        assert response.status_code == 400
        assert response.json()["detail"] == "模型名称已存在"


@pytest.mark.asyncio
async def test_get_model(app: FastAPI):
    """测试获取模型详情"""
    with patch("app.algo.views.model.get_model", new_callable=AsyncMock) as mock_get:
        mock_model = ModelInfo(
            id=test_model_id,
            project_id=test_project_id,
            name=test_model_data["name"],
            description=test_model_data["description"],
            version=test_model_data["version"],
            type=test_model_data["type"],
            format=test_model_data["format"],
            status=test_model_data["status"],
            storage_path=test_model_data["storage_path"],
            config=test_model_data["config"],
            metrics=test_model_data["metrics"],
        )
        mock_get.return_value = mock_model

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.get(f"/models/{test_model_id}")

        assert response.status_code == 200
        response_data = response.json()
        assert response_data["id"] == str(test_model_id)
        assert response_data["project_id"] == str(test_project_id)
        assert response_data["name"] == test_model_data["name"]
        assert response_data["description"] == test_model_data["description"]
        assert response_data["version"] == test_model_data["version"]
        assert response_data["type"] == test_model_data["type"]
        assert response_data["format"] == test_model_data["format"]
        assert response_data["status"] == test_model_data["status"]
        assert response_data["storage_path"] == test_model_data["storage_path"]
        assert response_data["config"] == test_model_data["config"]
        assert response_data["metrics"] == test_model_data["metrics"]
        mock_get.assert_called_once_with(test_model_id)


@pytest.mark.asyncio
async def test_get_model_not_found(app: FastAPI):
    """测试获取模型详情 - 模型不存在"""
    with patch("app.algo.views.model.get_model", new_callable=AsyncMock) as mock_get:
        mock_get.side_effect = ResourceNotFoundError(f"模型 {test_model_id} 不存在")

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.get(f"/models/{test_model_id}")

        assert response.status_code == 404
        assert response.json()["detail"] == f"模型 {test_model_id} 不存在"


@pytest.mark.asyncio
async def test_list_models(app: FastAPI):
    """测试获取模型列表"""
    with (
        patch("app.algo.views.model.list_models", new_callable=AsyncMock) as mock_list,
        patch("app.algo.views.model.count_models", new_callable=AsyncMock) as mock_count,
    ):
        mock_model = ModelInfo(
            id=test_model_id,
            project_id=test_project_id,
            name=test_model_data["name"],
            description=test_model_data["description"],
            version=test_model_data["version"],
            type=test_model_data["type"],
            format=test_model_data["format"],
            status=test_model_data["status"],
            storage_path=test_model_data["storage_path"],
            config=test_model_data["config"],
            metrics=test_model_data["metrics"],
        )
        mock_list.return_value = [mock_model]
        mock_count.return_value = 1

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.get("/models")

        assert response.status_code == 200
        response_data = response.json()
        assert response_data["total"] == 1
        assert len(response_data["items"]) == 1
        item = response_data["items"][0]
        assert item["id"] == str(test_model_id)
        assert item["project_id"] == str(test_project_id)
        assert item["name"] == test_model_data["name"]
        assert item["description"] == test_model_data["description"]
        assert item["version"] == test_model_data["version"]
        assert item["type"] == test_model_data["type"]
        assert item["format"] == test_model_data["format"]
        assert item["status"] == test_model_data["status"]
        assert item["storage_path"] == test_model_data["storage_path"]
        assert item["config"] == test_model_data["config"]
        assert item["metrics"] == test_model_data["metrics"]
        mock_list.assert_called_once_with(
            project_id=None, status=None, type=None, offset=0, limit=10
        )
        mock_count.assert_called_once_with(project_id=None, status=None, type=None)


@pytest.mark.asyncio
async def test_update_model(app: FastAPI):
    """测试更新模型"""
    update_data = {
        "name": "更新的模型名称",
        "description": "更新的模型描述",
        "version": "2.0.0",
        "status": ModelStatus.DRAFT,
        "metrics": {"accuracy": 0.98},
    }

    with patch("app.algo.views.model.update_model", new_callable=AsyncMock) as mock_update:
        mock_model = ModelInfo(
            id=test_model_id,
            project_id=test_project_id,
            name=update_data["name"],
            description=update_data["description"],
            version=update_data["version"],
            type=test_model_data["type"],
            format=test_model_data["format"],
            status=update_data["status"],
            storage_path=test_model_data["storage_path"],
            config=test_model_data["config"],
            metrics=update_data["metrics"],
        )
        mock_update.return_value = mock_model

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.patch(f"/models/{test_model_id}", json=update_data)

        assert response.status_code == 200
        response_data = response.json()
        assert response_data["id"] == str(test_model_id)
        assert response_data["name"] == update_data["name"]
        assert response_data["description"] == update_data["description"]
        assert response_data["version"] == update_data["version"]
        assert response_data["status"] == update_data["status"]
        assert response_data["metrics"] == update_data["metrics"]
        mock_update.assert_called_once_with(
            model_id=test_model_id,
            name=update_data["name"],
            description=update_data["description"],
            version=update_data["version"],
            status=update_data["status"],
            metrics=update_data["metrics"],
        )


@pytest.mark.asyncio
async def test_delete_model(app: FastAPI):
    """测试删除模型"""
    with patch("app.algo.views.model.delete_model", new_callable=AsyncMock) as mock_delete:
        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.delete(f"/models/{test_model_id}")

        assert response.status_code == 204
        mock_delete.assert_called_once_with(test_model_id)


@pytest.mark.asyncio
async def test_delete_model_not_found(app: FastAPI):
    """测试删除模型 - 模型不存在"""
    with patch("app.algo.views.model.delete_model", new_callable=AsyncMock) as mock_delete:
        mock_delete.side_effect = ResourceNotFoundError(f"模型 {test_model_id} 不存在")

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.delete(f"/models/{test_model_id}")

        assert response.status_code == 404
        assert response.json()["detail"] == f"模型 {test_model_id} 不存在"
