from unittest.mock import AsyncMock, patch
from uuid import UUID

import httpx
import pytest
from fastapi import FastAPI
from httpx import AsyncClient
from tortoise import Tortoise

from algo_flow.app.algo.models import Dataset
from algo_flow.app.algo.views.dataset import router as dataset_router
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.constant.algo import DatasetStatus
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError


# Mock the authentication dependency
async def mock_get_current_active_user():
    return {
        "id": UUID("00000000-0000-0000-0000-000000000000"),
        "username": "test-user",
        "email": "test@example.com",
        "is_active": True,
        "scopes": [
            "algo:dataset:create",
            "algo:dataset:read",
            "algo:dataset:update",
            "algo:dataset:delete",
        ],
    }


test_dataset_id = UUID("12345678-1234-5678-1234-567812345678")
test_dataset_data = {
    "name": "测试数据集",
    "description": "这是一个测试数据集",
    "storage_path": "/path/to/dataset",
    "metadata": {"format": "csv", "size": 1024, "rows": 1000, "columns": ["id", "name", "value"]},
}


@pytest.fixture
def app() -> FastAPI:
    """创建测试应用"""
    app = FastAPI()
    app.include_router(dataset_router, prefix="/datasets")
    app.dependency_overrides[get_current_active_user] = mock_get_current_active_user
    return app


@pytest.fixture(autouse=True)
async def init_db():
    """初始化测试数据库"""
    await Tortoise.init(db_url="sqlite://:memory:", modules={"models": ["app.algo.models"]})
    await Tortoise.generate_schemas()
    yield
    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_create_dataset(app: FastAPI):
    """测试创建数据集"""
    with patch("app.algo.views.dataset.create_dataset", new_callable=AsyncMock) as mock_create:
        mock_dataset = Dataset(
            id=test_dataset_id,
            name=test_dataset_data["name"],
            description=test_dataset_data["description"],
            status=DatasetStatus.PROCESSING,
            storage_path=test_dataset_data["storage_path"],
            metadata=test_dataset_data["metadata"],
        )
        mock_create.return_value = mock_dataset

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.post("/datasets", json=test_dataset_data)

        assert response.status_code == 201
        response_data = response.json()
        assert response_data["name"] == test_dataset_data["name"]
        assert response_data["description"] == test_dataset_data["description"]
        assert response_data["status"] == DatasetStatus.PROCESSING
        assert response_data["storage_path"] == test_dataset_data["storage_path"]
        assert response_data["metadata"] == test_dataset_data["metadata"]

        # 使用 model_dump 来获取实际传递的参数
        actual_call_args = mock_create.call_args[1]
        expected_args = {
            "name": test_dataset_data["name"],
            "description": test_dataset_data["description"],
            "status": DatasetStatus.PROCESSING,  # 包含默认状态
            "storage_path": test_dataset_data["storage_path"],
            "metadata": test_dataset_data["metadata"],
        }
        assert actual_call_args == expected_args


@pytest.mark.asyncio
async def test_create_dataset_conflict(app: FastAPI):
    """测试创建数据集 - 名称冲突"""
    with patch("app.algo.views.dataset.create_dataset", new_callable=AsyncMock) as mock_create:
        mock_create.side_effect = ResourceConflictError("数据集名称已存在")

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.post("/datasets", json=test_dataset_data)

        assert response.status_code == 400
        assert response.json()["detail"] == "数据集名称已存在"


@pytest.mark.asyncio
async def test_get_dataset(app: FastAPI):
    """测试获取数据集详情"""
    with patch("app.algo.views.dataset.get_dataset", new_callable=AsyncMock) as mock_get:
        mock_dataset = Dataset(
            id=test_dataset_id,
            name=test_dataset_data["name"],
            description=test_dataset_data["description"],
            status=DatasetStatus.PROCESSING,
            storage_path=test_dataset_data["storage_path"],
            metadata=test_dataset_data["metadata"],
        )
        mock_get.return_value = mock_dataset

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.get(f"/datasets/{test_dataset_id}")

        assert response.status_code == 200
        response_data = response.json()
        assert response_data["id"] == str(test_dataset_id)
        assert response_data["name"] == test_dataset_data["name"]
        assert response_data["description"] == test_dataset_data["description"]
        assert response_data["status"] == DatasetStatus.PROCESSING
        assert response_data["storage_path"] == test_dataset_data["storage_path"]
        assert response_data["metadata"] == test_dataset_data["metadata"]
        mock_get.assert_called_once_with(test_dataset_id)


@pytest.mark.asyncio
async def test_get_dataset_not_found(app: FastAPI):
    """测试获取数据集详情 - 数据集不存在"""
    with patch("app.algo.views.dataset.get_dataset", new_callable=AsyncMock) as mock_get:
        mock_get.side_effect = ResourceNotFoundError(f"数据集 {test_dataset_id} 不存在")

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.get(f"/datasets/{test_dataset_id}")

        assert response.status_code == 404
        assert response.json()["detail"] == f"数据集 {test_dataset_id} 不存在"


@pytest.mark.asyncio
async def test_list_datasets(app: FastAPI):
    """测试获取数据集列表"""
    with (
        patch("app.algo.views.dataset.list_datasets", new_callable=AsyncMock) as mock_list,
        patch("app.algo.views.dataset.count_datasets", new_callable=AsyncMock) as mock_count,
    ):
        mock_dataset = Dataset(
            id=test_dataset_id,
            name=test_dataset_data["name"],
            description=test_dataset_data["description"],
            status=DatasetStatus.PROCESSING,
            storage_path=test_dataset_data["storage_path"],
            metadata=test_dataset_data["metadata"],
        )
        mock_list.return_value = [mock_dataset]
        mock_count.return_value = 1

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.get("/datasets")

        assert response.status_code == 200
        response_data = response.json()
        assert response_data["total"] == 1
        assert len(response_data["items"]) == 1
        item = response_data["items"][0]
        assert item["id"] == str(test_dataset_id)
        assert item["name"] == test_dataset_data["name"]
        assert item["description"] == test_dataset_data["description"]
        assert item["status"] == DatasetStatus.PROCESSING
        assert item["storage_path"] == test_dataset_data["storage_path"]
        assert item["metadata"] == test_dataset_data["metadata"]
        mock_list.assert_called_once_with(status=None, offset=0, limit=10)
        mock_count.assert_called_once_with(status=None)


@pytest.mark.asyncio
async def test_update_dataset(app: FastAPI):
    """测试更新数据集"""
    update_data = {
        "name": "更新的数据集名称",
        "description": "更新的数据集描述",
        "status": DatasetStatus.READY,
        "metadata": {
            "format": "csv",
            "size": 2048,
            "rows": 2000,
            "columns": ["id", "name", "value", "label"],
        },
    }

    with patch("app.algo.views.dataset.update_dataset", new_callable=AsyncMock) as mock_update:
        mock_dataset = Dataset(
            id=test_dataset_id,
            name=update_data["name"],
            description=update_data["description"],
            status=update_data["status"],
            storage_path=test_dataset_data["storage_path"],
            metadata=update_data["metadata"],
        )
        mock_update.return_value = mock_dataset

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.patch(f"/datasets/{test_dataset_id}", json=update_data)

        assert response.status_code == 200
        response_data = response.json()
        assert response_data["id"] == str(test_dataset_id)
        assert response_data["name"] == update_data["name"]
        assert response_data["description"] == update_data["description"]
        assert response_data["status"] == update_data["status"]
        assert response_data["metadata"] == update_data["metadata"]

        # 使用 model_dump 来获取实际传递的参数
        actual_call_args = mock_update.call_args[1]
        expected_args = {
            "dataset_id": test_dataset_id,
            "name": update_data["name"],
            "description": update_data["description"],
            "status": update_data["status"],
            "metadata": update_data["metadata"],
        }
        assert actual_call_args == expected_args


@pytest.mark.asyncio
async def test_delete_dataset(app: FastAPI):
    """测试删除数据集"""
    with patch("app.algo.views.dataset.delete_dataset", new_callable=AsyncMock) as mock_delete:
        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.delete(f"/datasets/{test_dataset_id}")

        assert response.status_code == 204
        mock_delete.assert_called_once_with(test_dataset_id)


@pytest.mark.asyncio
async def test_delete_dataset_not_found(app: FastAPI):
    """测试删除数据集 - 数据集不存在"""
    with patch("app.algo.views.dataset.delete_dataset", new_callable=AsyncMock) as mock_delete:
        mock_delete.side_effect = ResourceNotFoundError(f"数据集 {test_dataset_id} 不存在")

        async with AsyncClient(
            app=app, base_url="http://test", transport=httpx.ASGITransport(app=app)
        ) as client:
            response = await client.delete(f"/datasets/{test_dataset_id}")

        assert response.status_code == 404
        assert response.json()["detail"] == f"数据集 {test_dataset_id} 不存在"
