from typing import AsyncGenerator
from uuid import UUID

import pytest
from fastapi import FastAPI
from httpx import AsyncClient
from tortoise import Tortoise

from algo_flow.app.algo.views.compute import router as compute_router
from algo_flow.app.system.views.auth import get_current_active_user
from algo_flow.cores.constant.algo import ComputeNodeStatus, ComputeNodeType

# 测试数据
test_compute_data = {
    "name": "测试计算节点",
    "description": "这是一个测试计算节点",
    "type": ComputeNodeType.GPU.value,
    "ip_address": "192.168.1.100",
    "port": 8000,
    "resources": {
        "gpu": {
            "count": 4,
            "memory": 16384,
            "model": "NVIDIA A100",
        },
        "cpu": {
            "cores": 32,
            "memory": 131072,
        },
        "disk": {
            "total": 1024000,
            "available": 512000,
        },
    },
}


# 模拟认证用户
async def mock_get_current_active_user():
    """模拟认证用户"""
    return {
        "id": UUID("00000000-0000-0000-0000-000000000000"),
        "username": "test",
        "email": "test@example.com",
        "is_active": True,
        "scopes": [
            "algo:compute:create",
            "algo:compute:read",
            "algo:compute:update",
            "algo:compute:delete",
        ],
    }


@pytest.fixture
def app() -> FastAPI:
    """创建测试应用"""
    app = FastAPI()
    app.include_router(compute_router, prefix="/compute")
    app.dependency_overrides[get_current_active_user] = mock_get_current_active_user
    return app


@pytest.fixture(autouse=True)
async def setup_database() -> AsyncGenerator:
    """设置测试数据库"""
    await Tortoise.init(
        db_url="sqlite://:memory:",
        modules={"models": ["app.algo.models"]},
    )
    await Tortoise.generate_schemas()

    yield

    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_create_compute_node(app: FastAPI):
    """测试创建计算节点接口"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.post("/compute", json=test_compute_data)
        assert response.status_code == 201
        data = response.json()
        assert data["name"] == test_compute_data["name"]
        assert data["description"] == test_compute_data["description"]
        assert data["type"] == test_compute_data["type"]
        assert data["ip_address"] == test_compute_data["ip_address"]
        assert data["port"] == test_compute_data["port"]
        assert data["resources"] == test_compute_data["resources"]
        assert data["status"] == ComputeNodeStatus.OFFLINE.value


@pytest.mark.asyncio
async def test_create_compute_node_with_duplicate_name(app: FastAPI):
    """测试创建同名计算节点接口"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个计算节点
        await client.post("/compute", json=test_compute_data)

        # 尝试创建同名计算节点
        response = await client.post("/compute", json=test_compute_data)
        assert response.status_code == 409


@pytest.mark.asyncio
async def test_create_compute_node_with_duplicate_ip_port(app: FastAPI):
    """测试创建具有相同IP和端口的计算节点接口"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个计算节点
        await client.post("/compute", json=test_compute_data)

        # 尝试创建具有相同IP和端口的计算节点
        response = await client.post(
            "/compute",
            json={
                **test_compute_data,
                "name": "另一个计算节点",
            },
        )
        assert response.status_code == 409


@pytest.mark.asyncio
async def test_get_compute_node(app: FastAPI):
    """测试获取计算节点接口"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个计算节点
        create_response = await client.post("/compute", json=test_compute_data)
        node_id = create_response.json()["id"]

        # 获取计算节点
        response = await client.get(f"/compute/{node_id}")
        assert response.status_code == 200
        data = response.json()
        assert data["id"] == node_id
        assert data["name"] == test_compute_data["name"]
        assert data["type"] == test_compute_data["type"]
        assert data["ip_address"] == test_compute_data["ip_address"]
        assert data["port"] == test_compute_data["port"]


@pytest.mark.asyncio
async def test_get_nonexistent_compute_node(app: FastAPI):
    """测试获取不存在的计算节点接口"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.get(f"/compute/{UUID('00000000-0000-0000-0000-000000000000')}")
        assert response.status_code == 404


@pytest.mark.asyncio
async def test_list_compute_nodes(app: FastAPI):
    """测试获取计算节点列表接口"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 创建多个计算节点
        await client.post("/compute", json=test_compute_data)
        await client.post(
            "/compute",
            json={
                **test_compute_data,
                "name": "测试计算节点2",
                "ip_address": "192.168.1.101",
            },
        )

        # 测试无过滤条件
        response = await client.get("/compute")
        assert response.status_code == 200
        data = response.json()
        assert len(data["items"]) == 2
        assert data["total"] == 2

        # 测试按类型过滤
        response = await client.get("/compute", params={"type": ComputeNodeType.GPU.value})
        assert response.status_code == 200
        data = response.json()
        assert len(data["items"]) == 2
        assert data["total"] == 2

        # 测试按状态过滤
        response = await client.get("/compute", params={"status": ComputeNodeStatus.OFFLINE.value})
        assert response.status_code == 200
        data = response.json()
        assert len(data["items"]) == 2
        assert data["total"] == 2

        # 测试分页
        response = await client.get("/compute", params={"limit": 1})
        assert response.status_code == 200
        data = response.json()
        assert len(data["items"]) == 1
        assert data["total"] == 2


@pytest.mark.asyncio
async def test_update_compute_node(app: FastAPI):
    """测试更新计算节点接口"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个计算节点
        create_response = await client.post("/compute", json=test_compute_data)
        node_id = create_response.json()["id"]

        # 更新计算节点
        updated_data = {
            "name": "更新后的计算节点",
            "description": "这是更新后的描述",
            "status": ComputeNodeStatus.ONLINE.value,
            "resources": {
                "gpu": {
                    "count": 8,
                    "memory": 32768,
                    "model": "NVIDIA A100",
                },
                "cpu": {
                    "cores": 64,
                    "memory": 262144,
                },
                "disk": {
                    "total": 2048000,
                    "available": 1024000,
                },
            },
        }
        response = await client.patch(f"/compute/{node_id}", json=updated_data)
        assert response.status_code == 200
        data = response.json()
        assert data["id"] == node_id
        assert data["name"] == updated_data["name"]
        assert data["description"] == updated_data["description"]
        assert data["status"] == updated_data["status"]
        assert data["resources"] == updated_data["resources"]


@pytest.mark.asyncio
async def test_update_nonexistent_compute_node(app: FastAPI):
    """测试更新不存在的计算节点接口"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.patch(
            f"/compute/{UUID('00000000-0000-0000-0000-000000000000')}",
            json={"name": "新名称"},
        )
        assert response.status_code == 404


@pytest.mark.asyncio
async def test_update_compute_node_with_duplicate_name(app: FastAPI):
    """测试更新计算节点时使用重复的名称"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 创建两个计算节点
        await client.post("/compute", json=test_compute_data)
        node2_response = await client.post(
            "/compute",
            json={
                **test_compute_data,
                "name": "测试计算节点2",
                "ip_address": "192.168.1.101",
            },
        )
        node2_id = node2_response.json()["id"]

        # 尝试将 node2 的名称更新为 node1 的名称
        response = await client.patch(
            f"/compute/{node2_id}",
            json={"name": test_compute_data["name"]},
        )
        assert response.status_code == 409


@pytest.mark.asyncio
async def test_update_compute_node_with_duplicate_ip_port(app: FastAPI):
    """测试更新计算节点时使用重复的IP和端口"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 创建两个计算节点
        await client.post("/compute", json=test_compute_data)
        node2_response = await client.post(
            "/compute",
            json={
                **test_compute_data,
                "name": "测试计算节点2",
                "ip_address": "192.168.1.101",
            },
        )
        node2_id = node2_response.json()["id"]

        # 尝试将 node2 的IP和端口更新为 node1 的IP和端口
        response = await client.patch(
            f"/compute/{node2_id}",
            json={
                "ip_address": test_compute_data["ip_address"],
                "port": test_compute_data["port"],
            },
        )
        assert response.status_code == 409


@pytest.mark.asyncio
async def test_delete_compute_node(app: FastAPI):
    """测试删除计算节点接口"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个计算节点
        create_response = await client.post("/compute", json=test_compute_data)
        node_id = create_response.json()["id"]

        # 删除计算节点
        response = await client.delete(f"/compute/{node_id}")
        assert response.status_code == 204

        # 验证计算节点已被删除
        response = await client.get(f"/compute/{node_id}")
        assert response.status_code == 404


@pytest.mark.asyncio
async def test_delete_nonexistent_compute_node(app: FastAPI):
    """测试删除不存在的计算节点接口"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        response = await client.delete(f"/compute/{UUID('00000000-0000-0000-0000-000000000000')}")
        assert response.status_code == 404


@pytest.mark.asyncio
async def test_update_compute_node_resources(app: FastAPI):
    """测试更新计算节点资源信息"""
    async with AsyncClient(app=app, base_url="http://test") as client:
        # 先创建一个计算节点
        create_response = await client.post("/compute", json=test_compute_data)
        node_id = create_response.json()["id"]

        # 更新计算节点资源信息
        updated_resources = {
            "gpu": {
                "count": 8,
                "memory": 32768,
                "model": "NVIDIA A100",
            },
            "cpu": {
                "cores": 64,
                "memory": 262144,
            },
            "disk": {
                "total": 2048000,
                "available": 1024000,
            },
        }

        response = await client.patch(f"/compute/{node_id}", json={"resources": updated_resources})

        assert response.status_code == 200
        data = response.json()
        assert data["resources"] == updated_resources
