import uuid
from typing import AsyncGenerator

import pytest
from tortoise import Tortoise

from algo_flow.app.algo.crud.project import (
    count_projects,
    create_project,
    delete_project,
    get_project,
    list_projects,
    update_project,
)
from algo_flow.cores.constant.algo import ProjectStatus
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

# 测试数据
test_project_data = {
    "name": "测试项目",
    "description": "这是一个测试项目",
    "status": ProjectStatus.ACTIVE,
}


@pytest.fixture(autouse=True)
async def setup_database() -> AsyncGenerator:
    """设置测试数据库"""
    await Tortoise.init(
        db_url="sqlite://:memory:",
        modules={"models": ["app.algo.models"]},
    )
    await Tortoise.generate_schemas()
    yield
    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_create_project():
    """测试创建项目"""
    # 创建项目
    project = await create_project(**test_project_data)

    # 验证项目信息
    assert project.name == test_project_data["name"]
    assert project.description == test_project_data["description"]
    assert project.status == test_project_data["status"]


@pytest.mark.asyncio
async def test_create_project_with_duplicate_name():
    """测试创建同名项目"""
    # 先创建一个项目
    await create_project(**test_project_data)

    # 尝试创建同名项目
    with pytest.raises(ResourceConflictError):
        await create_project(**test_project_data)


@pytest.mark.asyncio
async def test_get_project():
    """测试获取项目"""
    # 先创建一个项目
    created_project = await create_project(**test_project_data)

    # 获取项目
    project = await get_project(created_project.id)

    # 验证项目信息
    assert project.id == created_project.id
    assert project.name == test_project_data["name"]
    assert project.description == test_project_data["description"]


@pytest.mark.asyncio
async def test_get_nonexistent_project():
    """测试获取不存在的项目"""
    with pytest.raises(ResourceNotFoundError):
        await get_project(uuid.uuid4())


@pytest.mark.asyncio
async def test_list_projects():
    """测试获取项目列表"""
    # 创建多个项目
    project1 = await create_project(**test_project_data)
    await create_project(
        **{
            **test_project_data,
            "name": "测试项目2",
            "status": ProjectStatus.ARCHIVED,
        }
    )

    # 测试无过滤条件
    projects = await list_projects()
    assert len(projects) == 2

    # 测试按状态过滤
    projects = await list_projects(status=ProjectStatus.ACTIVE)
    assert len(projects) == 1
    assert projects[0].id == project1.id

    # 测试分页
    projects = await list_projects(limit=1)
    assert len(projects) == 1


@pytest.mark.asyncio
async def test_update_project():
    """测试更新项目"""
    # 先创建一个项目
    project = await create_project(**test_project_data)

    # 更新项目
    updated_data = {
        "name": "更新后的项目",
        "description": "这是更新后的描述",
        "status": ProjectStatus.ARCHIVED,
    }
    updated_project = await update_project(project.id, **updated_data)

    # 验证更新后的信息
    assert updated_project.id == project.id
    assert updated_project.name == updated_data["name"]
    assert updated_project.description == updated_data["description"]
    assert updated_project.status == updated_data["status"]


@pytest.mark.asyncio
async def test_update_nonexistent_project():
    """测试更新不存在的项目"""
    with pytest.raises(ResourceNotFoundError):
        await update_project(
            uuid.uuid4(),
            name="新名称",
        )


@pytest.mark.asyncio
async def test_update_project_with_duplicate_name():
    """测试更新项目时使用重复的名称"""
    # 创建两个项目
    project1 = await create_project(**test_project_data)
    project2 = await create_project(
        **{
            **test_project_data,
            "name": "测试项目2",
        }
    )

    # 尝试将 project2 的名称更新为 project1 的名称
    with pytest.raises(ResourceConflictError):
        await update_project(
            project2.id,
            name=project1.name,
        )


@pytest.mark.asyncio
async def test_delete_project():
    """测试删除项目"""
    # 先创建一个项目
    project = await create_project(**test_project_data)

    # 删除项目
    await delete_project(project.id)

    # 验证项目已被删除
    with pytest.raises(ResourceNotFoundError):
        await get_project(project.id)


@pytest.mark.asyncio
async def test_delete_nonexistent_project():
    """测试删除不存在的项目"""
    with pytest.raises(ResourceNotFoundError):
        await delete_project(uuid.uuid4())


@pytest.mark.asyncio
async def test_count_projects():
    """测试获取项目总数"""
    # 创建多个项目
    await create_project(**test_project_data)
    await create_project(
        **{
            **test_project_data,
            "name": "测试项目2",
            "status": ProjectStatus.ARCHIVED,
        }
    )

    # 测试无过滤条件
    count = await count_projects()
    assert count == 2

    # 测试按状态过滤
    count = await count_projects(status=ProjectStatus.ACTIVE)
    assert count == 1
