import uuid
from typing import AsyncGenerator

import pytest
from tortoise import Tortoise

from algo_flow.app.algo.crud.dataset import (
    count_datasets,
    create_dataset,
    delete_dataset,
    get_dataset,
    list_datasets,
    update_dataset,
)
from algo_flow.cores.constant.algo import DatasetStatus
from algo_flow.cores.exceptions import ResourceConflictError, ResourceNotFoundError

# 测试数据
test_dataset_data = {
    "name": "测试数据集",
    "description": "这是一个测试数据集",
    "status": DatasetStatus.READY,
    "storage_path": "/path/to/dataset",
    "metadata": {
        "format": "COCO",
        "size": 1000,
        "features": {
            "image_size": [224, 224],
            "channels": 3,
            "classes": ["cat", "dog"],
        },
        "statistics": {
            "class_distribution": {
                "cat": 500,
                "dog": 500,
            },
        },
    },
}


@pytest.fixture(autouse=True)
async def setup_database() -> AsyncGenerator:
    """设置测试数据库"""
    await Tortoise.init(
        db_url="sqlite://:memory:",
        modules={"models": ["app.algo.models"]},
    )
    await Tortoise.generate_schemas()

    yield

    await Tortoise.close_connections()


@pytest.mark.asyncio
async def test_create_dataset():
    """测试创建数据集"""
    # 创建数据集
    dataset = await create_dataset(**test_dataset_data)

    # 验证数据集信息
    assert dataset.name == test_dataset_data["name"]
    assert dataset.description == test_dataset_data["description"]
    assert dataset.status == test_dataset_data["status"]
    assert dataset.storage_path == test_dataset_data["storage_path"]
    assert dataset.metadata == test_dataset_data["metadata"]


@pytest.mark.asyncio
async def test_create_dataset_with_duplicate_name():
    """测试创建同名数据集"""
    # 先创建一个数据集
    await create_dataset(**test_dataset_data)

    # 尝试创建同名数据集
    with pytest.raises(ResourceConflictError):
        await create_dataset(**test_dataset_data)


@pytest.mark.asyncio
async def test_get_dataset():
    """测试获取数据集"""
    # 先创建一个数据集
    created_dataset = await create_dataset(**test_dataset_data)

    # 获取数据集
    dataset = await get_dataset(created_dataset.id)

    # 验证数据集信息
    assert dataset.id == created_dataset.id
    assert dataset.name == test_dataset_data["name"]


@pytest.mark.asyncio
async def test_get_nonexistent_dataset():
    """测试获取不存在的数据集"""
    with pytest.raises(ResourceNotFoundError):
        await get_dataset(uuid.uuid4())


@pytest.mark.asyncio
async def test_list_datasets():
    """测试获取数据集列表"""
    # 创建多个数据集
    dataset1 = await create_dataset(**test_dataset_data)
    await create_dataset(
        **{
            **test_dataset_data,
            "name": "测试数据集2",
            "status": DatasetStatus.PROCESSING,
        },
    )

    # 测试无过滤条件
    datasets = await list_datasets()
    assert len(datasets) == 2

    # 测试按状态过滤
    datasets = await list_datasets(status=DatasetStatus.READY)
    assert len(datasets) == 1
    assert datasets[0].id == dataset1.id

    # 测试分页
    datasets = await list_datasets(limit=1)
    assert len(datasets) == 1


@pytest.mark.asyncio
async def test_update_dataset():
    """测试更新数据集"""
    # 先创建一个数据集
    dataset = await create_dataset(**test_dataset_data)

    # 更新数据集
    updated_data = {
        "name": "更新后的数据集",
        "description": "这是更新后的描述",
        "status": DatasetStatus.PROCESSING,
        "metadata": {
            "format": "COCO",
            "size": 1000,
            "features": {
                "image_size": [448, 448],
                "channels": 3,
                "classes": ["cat", "dog", "bird"],
            },
            "statistics": {
                "class_distribution": {
                    "cat": 400,
                    "dog": 400,
                    "bird": 200,
                },
            },
        },
    }
    updated_dataset = await update_dataset(dataset.id, **updated_data)

    # 验证更新后的信息
    assert updated_dataset.id == dataset.id
    assert updated_dataset.name == updated_data["name"]
    assert updated_dataset.description == updated_data["description"]
    assert updated_dataset.status == updated_data["status"]
    assert updated_dataset.metadata == updated_data["metadata"]


@pytest.mark.asyncio
async def test_update_nonexistent_dataset():
    """测试更新不存在的数据集"""
    with pytest.raises(ResourceNotFoundError):
        await update_dataset(
            uuid.uuid4(),
            name="新名称",
        )


@pytest.mark.asyncio
async def test_update_dataset_with_duplicate_name():
    """测试更新数据集时使用重复的名称"""
    # 创建两个数据集
    dataset1 = await create_dataset(**test_dataset_data)
    dataset2 = await create_dataset(
        **{
            **test_dataset_data,
            "name": "测试数据集2",
        },
    )

    # 尝试将 dataset2 的名称更新为 dataset1 的名称
    with pytest.raises(ResourceConflictError):
        await update_dataset(
            dataset2.id,
            name=dataset1.name,
        )


@pytest.mark.asyncio
async def test_delete_dataset():
    """测试删除数据集"""
    # 先创建一个数据集
    dataset = await create_dataset(**test_dataset_data)

    # 删除数据集
    await delete_dataset(dataset.id)

    # 验证数据集已被删除
    with pytest.raises(ResourceNotFoundError):
        await get_dataset(dataset.id)


@pytest.mark.asyncio
async def test_delete_nonexistent_dataset():
    """测试删除不存在的数据集"""
    with pytest.raises(ResourceNotFoundError):
        await delete_dataset(uuid.uuid4())


@pytest.mark.asyncio
async def test_count_datasets():
    """测试获取数据集总数"""
    # 创建多个数据集
    await create_dataset(**test_dataset_data)
    await create_dataset(
        **{
            **test_dataset_data,
            "name": "测试数据集2",
            "status": DatasetStatus.PROCESSING,
        },
    )

    # 测试无过滤条件
    count = await count_datasets()
    assert count == 2

    # 测试按状态过滤
    count = await count_datasets(status=DatasetStatus.READY)
    assert count == 1


@pytest.mark.asyncio
async def test_update_dataset_metadata():
    """测试更新数据集元数据"""
    # 先创建一个数据集
    dataset = await create_dataset(**test_dataset_data)

    # 更新元数据
    new_metadata = {
        "file_count": 2000,
        "total_size": 2048000,
        "file_types": ["jpg", "png"],
        "labels": ["cat", "dog", "bird"],
        "updated_at": "2024-03-20",
    }

    updated_dataset = await update_dataset(dataset_id=dataset.id, metadata=new_metadata)

    assert updated_dataset.metadata == new_metadata
    assert updated_dataset.name == dataset.name  # 其他字段保持不变
    assert updated_dataset.storage_path == dataset.storage_path  # 其他字段保持不变
