from typing import List, Optional

from algo_flow.cores.async_http import async_http_request
from algo_flow.cores.config import settings

oauth_api = "https://github.com/login/oauth/access_token"
emails_api = "https://api.github.com/user/emails"

from pydantic import BaseModel, Field


class OAuth2GithubRequestForm(BaseModel):
    """GitHub OAuth2 请求表单"""

    client_id: str = Field(..., description="客户端ID")
    client_secret: str = Field(..., description="客户端密钥")
    code: str = Field(..., description="授权码")
    scope: Optional[str] = Field(None, description="请求的权限范围")
    state: Optional[str] = Field(None, description="状态码")


class OAuth2GithubResponse(BaseModel):
    """GitHub OAuth2 响应"""

    access_token: str = Field(..., description="访问令牌")
    token_type: str = Field(..., description="令牌类型")
    scope: str = Field(..., description="权限范围")


class GithubToken(BaseModel):
    """GitHub Token响应模型"""

    access_token: str = Field(..., description="访问令牌")
    token_type: str = Field(..., description="令牌类型")
    scope: str = Field(..., description="权限范围")


class GithubUser(BaseModel):
    """GitHub 用户信息"""

    login: str = Field(..., description="用户名")
    id: int = Field(..., description="用户ID")
    node_id: str = Field(..., description="节点ID")
    avatar_url: str = Field(..., description="头像URL")
    url: str = Field(..., description="API URL")
    html_url: str = Field(..., description="主页URL")
    name: Optional[str] = Field(None, description="姓名")
    company: Optional[str] = Field(None, description="公司")
    blog: Optional[str] = Field(None, description="博客")
    location: Optional[str] = Field(None, description="位置")
    email: Optional[str] = Field(None, description="邮箱")
    bio: Optional[str] = Field(None, description="简介")


class GithubEmail(BaseModel):
    """GitHub 邮箱信息"""

    email: str = Field(..., description="邮箱地址")
    primary: bool = Field(..., description="是否主邮箱")
    verified: bool = Field(..., description="是否已验证")
    visibility: Optional[str] = Field(None, description="可见性")


class GithubEmailList(BaseModel):
    """GitHub 邮箱列表"""

    emails: List[GithubEmail] = Field(..., description="邮箱列表")


async def get_access_token(code: str) -> GithubToken:
    response = await async_http_request(
        url=oauth_api,
        headers={"Accept": "application/json"},
        json={
            "client_id": settings.github.client,
            "client_secret": settings.github.secret,
            "code": code,
        },
    )
    json_data = response.json()
    return GithubToken(**json_data)


async def get_emails_by_access_token(token: GithubToken) -> List[GithubEmail]:
    response = await async_http_request(
        url=emails_api,
        headers={
            "Accept": "application/json",
            "Authorization": f"{token.token_type} {token.access_token}",
        },
    )
    json_data = response.json()
    return [GithubEmail(**email_data) for email_data in json_data]


async def get_primary_email_by_access_token(token: GithubToken) -> Optional[GithubEmail]:
    emails = await get_emails_by_access_token(token)
    return next((email for email in emails if email.primary), None)
